/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentStats;
import com.sleepycat.je.RunRecoveryException;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.latch.Latch;
import com.sleepycat.je.log.DbChecksumException;
import com.sleepycat.je.log.FSyncManager;
import com.sleepycat.je.log.FileHandle;
import com.sleepycat.je.log.FileHeader;
import com.sleepycat.je.log.JEFileFilter;
import com.sleepycat.je.log.LogBuffer;
import com.sleepycat.je.log.LogException;
import com.sleepycat.je.log.LogFileNotFoundException;
import com.sleepycat.je.log.LogManager;
import com.sleepycat.je.log.entry.LogEntry;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.HexFormatter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Random;
import java.util.Set;

public class FileManager {
    private static final String DEBUG_NAME = FileManager.class.getName();
    private static long writeCount = 0L;
    private static long stopOnWriteCount = Long.MAX_VALUE;
    public static final String JE_SUFFIX = ".jdb";
    public static final String CIF_SUFFIX = ".cif";
    public static final String DEL_SUFFIX = ".del";
    public static final String BAD_SUFFIX = ".bad";
    public static final String LOCK_SUFFIX = ".lck";
    static final String[] DEL_SUFFIXES = new String[]{".del"};
    static final String[] JE_SUFFIXES = new String[]{".jdb"};
    private static final String[] JE_AND_DEL_SUFFIXES = new String[]{".jdb", ".del"};
    private EnvironmentImpl envImpl;
    private long maxFileSize;
    private File dbEnvHome;
    private boolean includeDeletedFiles = false;
    private FileCache fileCache;
    private Latch fileCacheLatch;
    private RandomAccessFile lockFile;
    private FileChannel channel;
    private FileLock envLock;
    private FileLock exclLock;
    private boolean readOnly;
    private long currentFileNum;
    private long nextAvailableLsn;
    private long lastUsedLsn;
    private long prevOffset;
    private boolean forceNewFile;
    private long savedNextAvailableLsn;
    private long savedLastUsedLsn;
    private long savedPrevOffset;
    private LogEndFileDescriptor endOfLog;
    private FSyncManager syncManager;
    private Map perFileLastUsedLsn;
    private static final boolean IO_EXCEPTION_TESTING = false;
    private static final int IO_EXCEPTION_MAX = 100;
    private int ioExceptionCounter = 0;
    private boolean ioExceptionThrown = false;
    private Random ioExceptionRandom = null;

    public FileManager(EnvironmentImpl envImpl, File dbEnvHome, boolean readOnly) throws DatabaseException {
        this.envImpl = envImpl;
        this.dbEnvHome = dbEnvHome;
        this.readOnly = readOnly;
        DbConfigManager configManager = envImpl.getConfigManager();
        this.maxFileSize = configManager.getLong(EnvironmentParams.LOG_FILE_MAX);
        this.lockEnvironment(readOnly, false);
        this.fileCache = new FileCache(configManager);
        this.fileCacheLatch = new Latch(DEBUG_NAME + "_fileCache", envImpl);
        if (!dbEnvHome.exists()) {
            throw new LogException("Environment home " + dbEnvHome + " doesn't exist");
        }
        this.currentFileNum = 0L;
        this.nextAvailableLsn = DbLsn.makeLsn(this.currentFileNum, FileManager.firstLogEntryOffset());
        this.lastUsedLsn = -1L;
        this.perFileLastUsedLsn = new HashMap();
        this.prevOffset = 0L;
        this.endOfLog = new LogEndFileDescriptor();
        this.saveLastPosition();
        String stopOnWriteProp = System.getProperty("je.debug.stopOnWrite");
        if (stopOnWriteProp != null) {
            stopOnWriteCount = Long.parseLong(stopOnWriteProp);
        }
        this.syncManager = new FSyncManager(envImpl);
    }

    public void setLastPosition(long nextAvailableLsn, long lastUsedLsn, long prevOffset) {
        this.lastUsedLsn = lastUsedLsn;
        this.perFileLastUsedLsn.put(new Long(DbLsn.getFileNumber(lastUsedLsn)), new Long(lastUsedLsn));
        this.nextAvailableLsn = nextAvailableLsn;
        this.currentFileNum = DbLsn.getFileNumber(this.nextAvailableLsn);
        this.prevOffset = prevOffset;
        this.saveLastPosition();
    }

    void saveLastPosition() {
        this.savedNextAvailableLsn = this.nextAvailableLsn;
        this.savedLastUsedLsn = this.lastUsedLsn;
        this.savedPrevOffset = this.prevOffset;
    }

    void restoreLastPosition() {
        this.nextAvailableLsn = this.savedNextAvailableLsn;
        this.lastUsedLsn = this.savedLastUsedLsn;
        this.prevOffset = this.savedPrevOffset;
    }

    public Long getFirstFileNum() {
        return this.getFileNum(true);
    }

    public boolean getReadOnly() {
        return this.readOnly;
    }

    public Long getLastFileNum() {
        return this.getFileNum(false);
    }

    public void setIncludeDeletedFiles(boolean includeDeletedFiles) {
        this.includeDeletedFiles = includeDeletedFiles;
    }

    public Long[] getAllFileNumbers() {
        String[] names = this.listFiles(JE_SUFFIXES);
        Long[] nums = new Long[names.length];
        for (int i = 0; i < nums.length; ++i) {
            nums[i] = this.getNumFromName(names[i]);
        }
        return nums;
    }

    public Long getFollowingFileNum(long currentFileNum, boolean forward) {
        Object[] names = this.listFiles(JE_SUFFIXES);
        String searchName = this.getFileName(currentFileNum, JE_SUFFIX);
        int foundIdx = Arrays.binarySearch(names, searchName);
        boolean foundTarget = false;
        if (foundIdx >= 0) {
            foundIdx = forward ? ++foundIdx : --foundIdx;
        } else {
            foundIdx = Math.abs(foundIdx + 1);
            if (!forward) {
                --foundIdx;
            }
        }
        if (forward && foundIdx < names.length) {
            foundTarget = true;
        } else if (!forward && foundIdx > -1) {
            foundTarget = true;
        }
        if (foundTarget) {
            return this.getNumFromName((String)names[foundIdx]);
        }
        return null;
    }

    public boolean filesExist() {
        String[] names = this.listFiles(JE_SUFFIXES);
        return names.length != 0;
    }

    private Long getFileNum(boolean first) {
        String[] names = this.listFiles(JE_SUFFIXES);
        if (names.length == 0) {
            return null;
        }
        int index = 0;
        if (!first) {
            index = names.length - 1;
        }
        return this.getNumFromName(names[index]);
    }

    private Long getNumFromName(String fileName) {
        String fileNumber = fileName.substring(0, fileName.indexOf("."));
        return new Long(Long.parseLong(fileNumber, 16));
    }

    String[] listFiles(String[] suffixes) {
        Object[] fileNames = this.dbEnvHome.list(new JEFileFilter(suffixes));
        Arrays.sort(fileNames);
        return fileNames;
    }

    public static String[] listFiles(File envDirFile, String[] suffixes) {
        Object[] fileNames = envDirFile.list(new JEFileFilter(suffixes));
        Arrays.sort(fileNames);
        return fileNames;
    }

    String[] getFullFileNames(long fileNum) {
        if (this.includeDeletedFiles) {
            int nSuffixes = JE_AND_DEL_SUFFIXES.length;
            String[] ret = new String[nSuffixes];
            for (int i = 0; i < nSuffixes; ++i) {
                ret[i] = this.getFullFileName(this.getFileName(fileNum, JE_AND_DEL_SUFFIXES[i]));
            }
            return ret;
        }
        return new String[]{this.getFullFileName(this.getFileName(fileNum, JE_SUFFIX))};
    }

    public String getFullFileName(long fileNum, String suffix) {
        return this.getFullFileName(this.getFileName(fileNum, suffix));
    }

    private String getFullFileName(String fileName) {
        return this.dbEnvHome + File.separator + fileName;
    }

    private String getFileName(long fileNum, String suffix) {
        return HexFormatter.formatLong(fileNum).substring(10) + suffix;
    }

    public void renameFile(long fileNum, String newSuffix) throws DatabaseException, IOException {
        int repeatNum = 0;
        boolean renamed = false;
        while (!renamed) {
            String newName;
            File targetFile;
            String generation = "";
            if (repeatNum > 0) {
                generation = "." + repeatNum;
            }
            if ((targetFile = new File(newName = this.getFullFileName(this.getFileName(fileNum, newSuffix) + generation))).exists()) {
                ++repeatNum;
                continue;
            }
            String oldFileName = this.getFullFileNames(fileNum)[0];
            this.clearFileCache(fileNum);
            File oldFile = new File(oldFileName);
            if (oldFile.renameTo(targetFile)) {
                renamed = true;
                continue;
            }
            throw new LogException("Couldn't rename " + oldFileName + " to " + newName);
        }
    }

    public void deleteFile(long fileNum) throws DatabaseException, IOException {
        String fileName = this.getFullFileNames(fileNum)[0];
        this.clearFileCache(fileNum);
        File file = new File(fileName);
        boolean done = file.delete();
        if (!done) {
            throw new LogException("Couldn't delete " + file);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    FileHandle getFileHandle(long fileNum) throws LogException, DatabaseException {
        Long fileId = new Long(fileNum);
        FileHandle fileHandle = null;
        while (true) {
            if ((fileHandle = this.fileCache.get(fileId)) == null) {
                this.fileCacheLatch.acquire();
                try {
                    fileHandle = this.fileCache.get(fileId);
                    if (fileHandle == null) {
                        fileHandle = this.makeFileHandle(fileNum, FileMode.READ_MODE);
                        this.fileCache.add(fileId, fileHandle);
                    }
                }
                finally {
                    this.fileCacheLatch.release();
                }
            }
            fileHandle.latch();
            if (fileHandle.getFile() != null) break;
            fileHandle.release();
        }
        return fileHandle;
    }

    private FileHandle makeFileHandle(long fileNum, FileMode mode) throws DatabaseException {
        String[] fileNames = this.getFullFileNames(fileNum);
        RandomAccessFile newFile = null;
        String fileName = null;
        try {
            FileNotFoundException FNFE = null;
            for (int i = 0; i < fileNames.length; ++i) {
                fileName = fileNames[i];
                try {
                    newFile = new RandomAccessFile(fileName, mode.getModeValue());
                    break;
                }
                catch (FileNotFoundException e) {
                    if (FNFE != null) continue;
                    FNFE = e;
                    continue;
                }
            }
            if (newFile == null) {
                throw FNFE;
            }
            boolean oldHeaderVersion = false;
            if (newFile.length() == 0L && mode == FileMode.READWRITE_MODE) {
                long lastLsn = DbLsn.longToLsn((Long)this.perFileLastUsedLsn.remove(new Long(fileNum - 1L)));
                long headerPrevOffset = 0L;
                if (lastLsn != -1L) {
                    headerPrevOffset = DbLsn.getFileOffset(lastLsn);
                }
                FileHeader fileHeader = new FileHeader(fileNum, headerPrevOffset);
                this.writeFileHeader(newFile, fileName, fileHeader);
            } else {
                oldHeaderVersion = this.readAndValidateFileHeader(newFile, fileName, fileNum);
            }
            return new FileHandle(newFile, fileName, this.envImpl, oldHeaderVersion);
        }
        catch (FileNotFoundException e) {
            throw new LogFileNotFoundException("Couldn't open file " + fileName + ": " + e.getMessage());
        }
        catch (DbChecksumException e) {
            this.closeFileInErrorCase(newFile);
            throw new DbChecksumException(this.envImpl, "Couldn't open file " + fileName, e);
        }
        catch (Throwable t) {
            this.closeFileInErrorCase(newFile);
            throw new DatabaseException("Couldn't open file " + fileName + ": " + t, t);
        }
    }

    private void closeFileInErrorCase(RandomAccessFile file) {
        try {
            if (file != null) {
                file.close();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private boolean readAndValidateFileHeader(RandomAccessFile file, String fileName, long fileNum) throws DatabaseException, IOException {
        LogManager logManager = this.envImpl.getLogManager();
        LogEntry headerEntry = logManager.getLogEntry(DbLsn.makeLsn(fileNum, 0L), file);
        FileHeader header = (FileHeader)headerEntry.getMainItem();
        return header.validate(fileName, fileNum);
    }

    private void writeFileHeader(RandomAccessFile file, String fileName, FileHeader header) throws DatabaseException, IOException {
        int bytesWritten;
        this.envImpl.checkIfInvalid();
        if (this.envImpl.mayNotWrite()) {
            return;
        }
        int headerSize = header.getLogSize();
        int entrySize = headerSize + 14;
        ByteBuffer headerBuf = this.envImpl.getLogManager().putIntoBuffer(header, headerSize, 0L, false, entrySize);
        if (++writeCount >= stopOnWriteCount) {
            Runtime.getRuntime().halt(255);
        }
        try {
            bytesWritten = file.getChannel().write(headerBuf);
        }
        catch (ClosedChannelException e) {
            throw new RunRecoveryException(this.envImpl, "Channel closed, may be due to thread interrupt", e);
        }
        catch (IOException e) {
            throw new RunRecoveryException(this.envImpl, "IOException caught: " + e);
        }
        if (bytesWritten != entrySize) {
            throw new LogException("File " + fileName + " was created with an incomplete header." + " Only " + bytesWritten + " bytes were written.");
        }
    }

    long getFileHeaderPrevOffset(long fileNum) throws IOException, DatabaseException {
        LogEntry headerEntry = this.envImpl.getLogManager().getLogEntry(DbLsn.makeLsn(fileNum, 0L));
        FileHeader header = (FileHeader)headerEntry.getMainItem();
        return header.getLastEntryInPrevFileOffset();
    }

    long getPrevEntryOffset() {
        return this.prevOffset;
    }

    boolean bumpLsn(long size) {
        boolean flippedFiles = false;
        if (this.forceNewFile || DbLsn.getFileOffset(this.nextAvailableLsn) + size > this.maxFileSize) {
            this.forceNewFile = false;
            ++this.currentFileNum;
            if (this.lastUsedLsn != -1L) {
                this.perFileLastUsedLsn.put(new Long(DbLsn.getFileNumber(this.lastUsedLsn)), new Long(this.lastUsedLsn));
            }
            this.prevOffset = 0L;
            this.lastUsedLsn = DbLsn.makeLsn(this.currentFileNum, FileManager.firstLogEntryOffset());
            flippedFiles = true;
        } else {
            this.prevOffset = this.lastUsedLsn == -1L ? 0L : DbLsn.getFileOffset(this.lastUsedLsn);
            this.lastUsedLsn = this.nextAvailableLsn;
        }
        this.nextAvailableLsn = DbLsn.makeLsn(DbLsn.getFileNumber(this.lastUsedLsn), DbLsn.getFileOffset(this.lastUsedLsn) + size);
        return flippedFiles;
    }

    void writeLogBuffer(LogBuffer fullBuffer) throws IOException, DatabaseException {
        this.envImpl.checkIfInvalid();
        if (this.envImpl.mayNotWrite()) {
            return;
        }
        long firstLsn = fullBuffer.getFirstLsn();
        if (firstLsn != -1L) {
            FileChannel channel = this.endOfLog.getWriteChannel(DbLsn.getFileNumber(firstLsn));
            ByteBuffer data = fullBuffer.getDataBuffer();
            if (++writeCount >= stopOnWriteCount) {
                Runtime.getRuntime().halt(255);
            }
            try {
                channel.write(data, DbLsn.getFileOffset(firstLsn));
            }
            catch (ClosedChannelException e) {
                throw new RunRecoveryException(this.envImpl, "Channel closed, may be due to thread interrupt", e);
            }
            catch (IOException IOE) {
                throw new RunRecoveryException(this.envImpl, "IOException during write: " + IOE);
            }
            this.saveLastPosition();
            if (EnvironmentImpl.getForcedYield()) {
                Thread.yield();
            }
        }
    }

    void syncLogEnd() throws DatabaseException {
        try {
            this.endOfLog.force();
        }
        catch (IOException e) {
            throw new DatabaseException(e);
        }
    }

    void syncLogEndAndFinishFile() throws DatabaseException, IOException {
        this.syncLogEnd();
        this.endOfLog.close();
    }

    void groupSync() throws DatabaseException {
        this.syncManager.fsync();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() throws IOException, DatabaseException {
        this.fileCacheLatch.acquire();
        try {
            this.fileCache.clear();
        }
        finally {
            this.fileCacheLatch.release();
        }
        this.endOfLog.close();
    }

    public void close() throws IOException, DatabaseException {
        if (this.envLock != null) {
            this.envLock.release();
        }
        if (this.exclLock != null) {
            this.exclLock.release();
        }
        if (this.channel != null) {
            this.channel.close();
        }
        if (this.lockFile != null) {
            this.lockFile.close();
        }
    }

    public boolean lockEnvironment(boolean readOnly, boolean exclusive) throws DatabaseException {
        try {
            if (this.checkEnvHomePermissions(readOnly)) {
                return true;
            }
            if (this.lockFile == null) {
                this.lockFile = new RandomAccessFile(new File(this.dbEnvHome, "je.lck"), "rw");
            }
            this.channel = this.lockFile.getChannel();
            boolean throwIt = false;
            try {
                if (exclusive) {
                    this.exclLock = this.channel.tryLock(1L, 2L, false);
                    return this.exclLock != null;
                }
                this.envLock = readOnly ? this.channel.tryLock(1L, 2L, true) : this.channel.tryLock(0L, 1L, false);
                if (this.envLock == null) {
                    throwIt = true;
                }
            }
            catch (OverlappingFileLockException e) {
                throwIt = true;
            }
            if (throwIt) {
                throw new LogException("A je..lckfile exists in " + this.dbEnvHome + " The environment can not be locked for " + (readOnly ? "shared" : "single writer") + " access.");
            }
        }
        catch (IOException IOE) {
            throw new LogException(IOE.toString());
        }
        return true;
    }

    public void releaseExclusiveLock() throws DatabaseException {
        try {
            if (this.exclLock != null) {
                this.exclLock.release();
            }
        }
        catch (IOException IOE) {
            throw new DatabaseException(IOE);
        }
    }

    private boolean checkEnvHomePermissions(boolean readOnly) throws DatabaseException {
        boolean envDirIsReadOnly;
        boolean bl = envDirIsReadOnly = !this.dbEnvHome.canWrite();
        if (envDirIsReadOnly && !readOnly) {
            throw new DatabaseException("The Environment directory is not writable, but the Environment was opened for read-write access.");
        }
        return envDirIsReadOnly;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void truncateLog(long fileNum, long offset) throws IOException, DatabaseException {
        FileHandle handle = this.makeFileHandle(fileNum, FileMode.READWRITE_MODE);
        RandomAccessFile file = handle.getFile();
        try {
            file.getChannel().truncate(offset);
        }
        finally {
            file.close();
        }
        if (handle.isOldHeaderVersion()) {
            this.forceNewFile = true;
        }
    }

    public static int firstLogEntryOffset() {
        return FileHeader.entrySize() + 14;
    }

    public long getNextLsn() {
        return this.nextAvailableLsn;
    }

    public long getLastUsedLsn() {
        return this.lastUsedLsn;
    }

    public long getNFSyncs() {
        return this.syncManager.getNFSyncs();
    }

    public long getNFSyncRequests() {
        return this.syncManager.getNFSyncRequests();
    }

    public long getNFSyncTimeouts() {
        return this.syncManager.getNTimeouts();
    }

    void loadStats(StatsConfig config, EnvironmentStats stats) throws DatabaseException {
        this.syncManager.loadStats(config, stats);
    }

    Set getCacheKeys() {
        return this.fileCache.getCacheKeys();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void clearFileCache(long fileNum) throws IOException, DatabaseException {
        this.fileCacheLatch.acquire();
        try {
            this.fileCache.remove(fileNum);
        }
        finally {
            this.fileCacheLatch.release();
        }
    }

    private void generateIOException() throws IOException {
        if (this.ioExceptionThrown) {
            try {
                throw new Exception("Write after RunRecoveryException");
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        ++this.ioExceptionCounter;
        if (this.ioExceptionCounter >= 100) {
            this.ioExceptionCounter = 0;
        }
        if (this.ioExceptionRandom == null) {
            this.ioExceptionRandom = new Random();
        }
        if (this.ioExceptionCounter == this.ioExceptionRandom.nextInt(100)) {
            this.ioExceptionThrown = true;
            throw new IOException("Randomly generated for testing");
        }
    }

    class LogEndFileDescriptor {
        private RandomAccessFile endOfLogRWFile = null;
        private RandomAccessFile endOfLogSyncFile = null;

        LogEndFileDescriptor() {
        }

        FileChannel getWriteChannel(long fileNumber) throws DatabaseException {
            if (this.endOfLogRWFile == null) {
                this.endOfLogRWFile = FileManager.this.makeFileHandle(fileNumber, FileMode.READWRITE_MODE).getFile();
                this.endOfLogSyncFile = FileManager.this.makeFileHandle(fileNumber, FileMode.READWRITE_MODE).getFile();
            }
            return this.endOfLogRWFile.getChannel();
        }

        void force() throws DatabaseException, IOException {
            RandomAccessFile file = this.endOfLogSyncFile;
            if (file != null) {
                FileChannel channel = file.getChannel();
                try {
                    channel.force(false);
                }
                catch (ClosedChannelException e) {
                    throw new RunRecoveryException(FileManager.this.envImpl, "Channel closed, may be due to thread interrupt", e);
                }
                if (EnvironmentImpl.getForcedYield()) {
                    Thread.yield();
                }
            }
        }

        void close() throws IOException {
            RandomAccessFile file;
            IOException firstException = null;
            if (this.endOfLogRWFile != null) {
                file = this.endOfLogRWFile;
                this.endOfLogRWFile = null;
                try {
                    file.close();
                }
                catch (IOException e) {
                    firstException = e;
                }
            }
            if (this.endOfLogSyncFile != null) {
                file = this.endOfLogSyncFile;
                this.endOfLogSyncFile = null;
                file.close();
            }
            if (firstException != null) {
                throw firstException;
            }
        }
    }

    private static class FileCache {
        private Map fileMap = new Hashtable();
        private LinkedList fileList = new LinkedList();
        private int fileCacheSize;

        FileCache(DbConfigManager configManager) throws DatabaseException {
            this.fileCacheSize = configManager.getInt(EnvironmentParams.LOG_FILE_CACHE_SIZE);
        }

        private FileHandle get(Long fileId) {
            return (FileHandle)this.fileMap.get(fileId);
        }

        private void add(Long fileId, FileHandle fileHandle) throws DatabaseException {
            if (this.fileList.size() >= this.fileCacheSize) {
                Iterator iter = this.fileList.iterator();
                while (iter.hasNext()) {
                    Long evictId = (Long)iter.next();
                    FileHandle evictTarget = (FileHandle)this.fileMap.get(evictId);
                    if (!evictTarget.latchNoWait()) continue;
                    try {
                        this.fileMap.remove(evictId);
                        iter.remove();
                        evictTarget.close();
                        break;
                    }
                    catch (IOException e) {
                        throw new DatabaseException(e);
                    }
                    finally {
                        evictTarget.release();
                    }
                }
            }
            this.fileList.add(fileId);
            this.fileMap.put(fileId, fileHandle);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void remove(long fileNum) throws IOException, DatabaseException {
            Iterator iter = this.fileList.iterator();
            while (iter.hasNext()) {
                Long evictId = (Long)iter.next();
                if (evictId != fileNum) continue;
                FileHandle evictTarget = (FileHandle)this.fileMap.get(evictId);
                try {
                    evictTarget.latch();
                    this.fileMap.remove(evictId);
                    iter.remove();
                    evictTarget.close();
                }
                finally {
                    evictTarget.release();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void clear() throws IOException, DatabaseException {
            Iterator iter = this.fileMap.values().iterator();
            while (iter.hasNext()) {
                FileHandle fileHandle = (FileHandle)iter.next();
                try {
                    fileHandle.latch();
                    fileHandle.close();
                    iter.remove();
                }
                finally {
                    fileHandle.release();
                }
            }
            this.fileMap.clear();
            this.fileList.clear();
        }

        private Set getCacheKeys() {
            return this.fileMap.keySet();
        }
    }

    public static class FileMode {
        public static final FileMode READ_MODE = new FileMode("r");
        public static final FileMode READWRITE_MODE = new FileMode("rw");
        private String fileModeValue;

        private FileMode(String fileModeValue) {
            this.fileModeValue = fileModeValue;
        }

        public String getModeValue() {
            return this.fileModeValue;
        }
    }
}

