/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2013 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.log;

import java.io.IOException;
import java.util.Collection;
import java.util.Map;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.cleaner.DbFileSummary;
import com.sleepycat.je.cleaner.LocalUtilizationTracker;
import com.sleepycat.je.cleaner.TrackedFileSummary;
import com.sleepycat.je.cleaner.UtilizationTracker;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentImpl;

/**
 * The SyncedLogManager uses the synchronized keyword to implement protected
 * regions.
 */
public class SyncedLogManager extends LogManager {

    /**
     * There is a single log manager per database environment.
     */
    public SyncedLogManager(EnvironmentImpl envImpl,
                            boolean readOnly)
        throws DatabaseException {

        super(envImpl, readOnly);
    }

    @Override
    LogWriteInfo serialLog(LogItem item,
                           LogContext context,
                           boolean forceNewLogFile,
                           boolean flushRequired,
                           UtilizationTracker tracker) throws IOException {
        synchronized (logWriteLatch) {
            return serialLogInternal(item,
                                     context,
                                     forceNewLogFile,
                                     flushRequired,
                                     tracker);
        }
    }

    @Override
    void updateObsolete(LogContext context, UtilizationTracker tracker) {
        synchronized (logWriteLatch) {
            updateObsoleteInternal(context, tracker);
        }
    }

    @Override
    protected void flushInternal(boolean flushRequired)
        throws DatabaseException {

        /*
         * If we cannot bump the current buffer because there are no
         * free buffers, the only recourse is to write all buffers
         * under the LWL.
         */
        synchronized (logWriteLatch) {
            if (!logBufferPool.bumpCurrent(0)) {
                logBufferPool.bumpAndWriteSynced(0, flushRequired);
                return;
            }
        }

        /*
         * We bumped the current buffer but did not write any buffers above.
         * Write the dirty buffers now.  Hopefully this is the common case.
         */
        logBufferPool.writeDirty(flushRequired);
    }

    /**
     * @see LogManager#getUnflushableTrackedSummary
     */
    @Override
    public TrackedFileSummary getUnflushableTrackedSummary(long file) {
        synchronized (logWriteLatch) {
            return getUnflushableTrackedSummaryInternal(file);
        }
    }

    /**
     * @see LogManager#removeTrackedFile
     */
    @Override
    public void removeTrackedFile(TrackedFileSummary tfs) {
        synchronized (logWriteLatch) {
            removeTrackedFileInternal(tfs);
        }
    }

    /**
     * @see LogManager#countObsoleteNode
     */
    @Override
    public void countObsoleteNode(long lsn,
                                  LogEntryType type,
                                  int size,
                                  DatabaseImpl nodeDb,
                                  boolean countExact) {
        synchronized (logWriteLatch) {
            countObsoleteNodeInternal(lsn, type, size, nodeDb, countExact);
        }
    }

    /**
     * @see LogManager#countObsoleteNodeDupsAllowed
     */
    @Override
    public void countObsoleteNodeDupsAllowed(long lsn,
                                             LogEntryType type,
                                             int size,
                                             DatabaseImpl nodeDb) {
        synchronized (logWriteLatch) {
            countObsoleteNodeDupsAllowedInternal(lsn, type, size, nodeDb);
        }
    }

    /**
     * @see LogManager#transferToUtilizationTracker
     */
    @Override
    public void transferToUtilizationTracker(LocalUtilizationTracker
                                             localTracker)
        throws DatabaseException {

        synchronized (logWriteLatch) {
            transferToUtilizationTrackerInternal(localTracker);
        }
    }

    /**
     * @see LogManager#countObsoleteDb
     */
    @Override
    public void countObsoleteDb(DatabaseImpl db) {
        synchronized (logWriteLatch) {
            countObsoleteDbInternal(db);
        }
    }

    /**
     * @see LogManager#removeDbFileSummaries
     */
    @Override
    public boolean removeDbFileSummaries(DatabaseImpl db,
                                         Collection<Long> fileNums) {
        synchronized (logWriteLatch) {
            return removeDbFileSummariesInternal(db, fileNums);
        }
    }

    /**
     * @see LogManager#cloneDbFileSummaries
     */
    @Override
    public Map<Long, DbFileSummary> cloneDbFileSummaries(DatabaseImpl db) {
        synchronized (logWriteLatch) {
            return db.cloneDbFileSummariesInternal();
        }
    }

    /**
     * @see LogManager#loadEndOfLogStat
     */
    @Override
    public void loadEndOfLogStat() {
        synchronized (logWriteLatch) {
            loadEndOfLogStatInternal();
        }
    }

    /**
     * @see LogManager#resetPool(DbConfigManager)
     */
    @Override
    public void resetPool(DbConfigManager configManager)
        throws DatabaseException {
        synchronized (logWriteLatch) {
           logBufferPool.reset(configManager);
        }
    }
}
