/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.evictor;

import static com.sleepycat.je.evictor.EvictorStatDefinition.AVG_BATCH_DESC;
import static com.sleepycat.je.evictor.EvictorStatDefinition.BIN_DELTA_FETCH_MISS;
import static com.sleepycat.je.evictor.EvictorStatDefinition.BIN_EVICTION_TYPE_DESC;
import static com.sleepycat.je.evictor.EvictorStatDefinition.BIN_FETCH;
import static com.sleepycat.je.evictor.EvictorStatDefinition.BIN_FETCH_MISS;
import static com.sleepycat.je.evictor.EvictorStatDefinition.CACHED_IN_COMPACT_KEY;
import static com.sleepycat.je.evictor.EvictorStatDefinition.CACHED_IN_NO_TARGET;
import static com.sleepycat.je.evictor.EvictorStatDefinition.CACHED_IN_SPARSE_TARGET;
import static com.sleepycat.je.evictor.EvictorStatDefinition.EVICTOR_BINS_MUTATED;
import static com.sleepycat.je.evictor.EvictorStatDefinition.EVICTOR_BINS_STRIPPED;
import static com.sleepycat.je.evictor.EvictorStatDefinition.EVICTOR_EVICT_PASSES;
import static com.sleepycat.je.evictor.EvictorStatDefinition.EVICTOR_NODES_EVICTED;
import static com.sleepycat.je.evictor.EvictorStatDefinition.EVICTOR_NODES_SCANNED;
import static com.sleepycat.je.evictor.EvictorStatDefinition.EVICTOR_ROOT_NODES_EVICTED;
import static com.sleepycat.je.evictor.EvictorStatDefinition.GROUP_DESC;
import static com.sleepycat.je.evictor.EvictorStatDefinition.GROUP_NAME;
import static com.sleepycat.je.evictor.EvictorStatDefinition.LN_FETCH;
import static com.sleepycat.je.evictor.EvictorStatDefinition.LN_FETCH_MISS;
import static com.sleepycat.je.evictor.EvictorStatDefinition.NUM_BATCHES_DESC;
import static com.sleepycat.je.evictor.EvictorStatDefinition.THREAD_UNAVAILABLE;
import static com.sleepycat.je.evictor.EvictorStatDefinition.UPPER_IN_EVICTION_TYPE_DESC;
import static com.sleepycat.je.evictor.EvictorStatDefinition.UPPER_IN_FETCH;
import static com.sleepycat.je.evictor.EvictorStatDefinition.UPPER_IN_FETCH_MISS;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import java.util.logging.Logger;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentMutableConfig;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.evictor.LRUEvictor.EvictionDebugStats;
import com.sleepycat.je.dbi.DatabaseId;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvConfigObserver;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.tree.IN;
import com.sleepycat.je.utilint.AtomicLongStat;
import com.sleepycat.je.utilint.IntegralLongAvgStat;
import com.sleepycat.je.utilint.LoggerUtils;
import com.sleepycat.je.utilint.LongStat;
import com.sleepycat.je.utilint.StatDefinition;
import com.sleepycat.je.utilint.StatGroup;
import com.sleepycat.je.utilint.StoppableThreadFactory;
import com.sleepycat.je.utilint.TestHook;

/**
 * The Evictor is responsible for maintaining the JE cache.
 *
 * There are currently two implementations of the abstract Evictor class:
 * The new LRUEvictor and the old INListEvictor. See those classes for further
 * details. The INListEvictor will be completely removed once the LRUEvictor
 * has been tested enough and its performance benefits are confirmed. 
 */
public abstract class Evictor implements EnvConfigObserver {

    /*
     * If new eviction source enums are added, a new stat is created, and
     * EnvironmentStats must be updated to add a getter method.
     *
     * CRITICAL eviction is called by operations executed app or daemon
     * threads which detect that the cache has reached its limits
     * CACHE_MODE eviction is called by operations that use a specific
     * Cursor.
     * EVICTORThread is the eviction pool
     * MANUAL is the call to Environment.evictMemory, called by recovery or
     *   application code.
     */
    public enum EvictionSource {
        /* Using ordinal for array values! */
        EVICTORTHREAD, MANUAL, CRITICAL, CACHEMODE, DAEMON;

        public StatDefinition getBINStatDef() {
            return new StatDefinition("nBINsEvicted" + toString(),
                                      BIN_EVICTION_TYPE_DESC);
        }

        public StatDefinition getUpperINStatDef() {
            return new StatDefinition("nUpperINsEvicted" + toString(),
                                      UPPER_IN_EVICTION_TYPE_DESC);
        }

        public StatDefinition getNumBatchesStatDef() {
            return new StatDefinition("nBatches" + toString(),
                                      NUM_BATCHES_DESC);
        }

        public StatDefinition getAvgBatchStatDef() {
            return new StatDefinition("avgBatch" + toString(),
                                      AVG_BATCH_DESC);
        }
    }

    /* Prevent endless eviction loops under extreme resource constraints. */
    static final int MAX_BATCHES_PER_RUN = 100;

    /*
     * Whether this evictor (and the memory cache) is shared by multiple
     * environments
     */
    final boolean isShared;

    /*
     * In case of multiple environments sharing a cache (and this Evictor),
     * firstEnvImpl references the 1st EnvironmentImpl to be created with
     * the shared cache.
     */
    final EnvironmentImpl firstEnvImpl;

    final Arbiter arbiter;

    final TargetSelector selector;

    /* The thread pool used to manage the background evictor threads. */
    private final ThreadPoolExecutor evictionPool;
    private int terminateMillis;

    /* Access count after which we clear the DatabaseImpl cache. */
    int dbCacheClearCount;

    /*
     * runEvictor is needed as a distinct flag, rather than setting maxThreads
     * to 0, because the ThreadPoolExecutor does not permit  maxThreads to be 0.
     */
    boolean runEvictor;

    /*
     * Whether to allow deltas when logging a dirty BIN that is being evicted.
     */
    final boolean allowBinDeltas;

    /*
     * Whether to mutate BINs to BIN deltas rather than evicting the full node.
     */
    final boolean mutateBins;

    /*
     * Stats
     */
    final StatGroup stats;

    /*
     * Number of doEvict() invocations per eviction source
     */
    final AtomicLongStat[] numBatches;

    /*
     * Number of evictBatch() invocations.
     */
    final LongStat nEvictPasses;

    /*
     * Number of probably-evictable nodes scanned in order to select
     * the eviction set out of which the "best" node is chosen as the
     * eviction target. 
     */
    final LongStat nNodesScanned;

    /*
     * Number of probably-evictable nodes chosen as eviction targets,
     * per eviction source.
     */
    final AtomicLong[] numBatchTargets;

    /*
     * Number of nodes evicted on this run. This could be understated, as a
     * whole subtree may have gone out with a single node.
     */
    final LongStat nNodesEvicted;

    /* Number of closed database root nodes evicted on this run. */
    final LongStat nRootNodesEvicted;

    /* Number of BINs stripped. */
    final LongStat nBINsStripped;

    /* Number of BINs mutated to deltas. */
    final LongStat nBINsMutated;

    /*
     * Array of stats is indexed into by the EvictionSource ordinal value.
     * A EnumMap could have been an alternative, but would be heavier weight.
     */
    final AtomicLongStat[] binEvictSources;
    final AtomicLongStat[] inEvictSources;

    /*
     * Tree related cache hit/miss stats. A subset of the cache misses recorded
     * by the log manager, in that these only record tree node hits and misses.
     * Recorded by IN.fetchTarget, but grouped with evictor stats. Use
     * AtomicLongStat for multithreading safety.
     */
    final AtomicLongStat nLNFetch;
    final AtomicLongStat nBINFetch;
    final AtomicLongStat nUpperINFetch;
    final AtomicLongStat nLNFetchMiss;
    final AtomicLongStat nBINFetchMiss;
    final AtomicLongStat nBINDeltaFetchMiss;
    final AtomicLongStat nUpperINFetchMiss;

    final AtomicLongStat nThreadUnavailable;

     /* Stats for IN compact array representations currently in cache. */
    final AtomicLong nINSparseTarget;
    final AtomicLong nINNoTarget;
    final AtomicLong nINCompactKey;

    /* Debugging and unit test support. */
    TestHook<Object> preEvictINHook;
    TestHook<IN> evictProfile;

    /* Eviction calls cannot be recursive. */
    final ReentrancyGuard reentrancyGuard;

    /* Flag to help shutdown launched eviction tasks. */
    final AtomicBoolean shutdownRequested;

    final Logger logger;

    Evictor(EnvironmentImpl envImpl)
        throws DatabaseException {

        isShared = envImpl.getSharedCache();

        firstEnvImpl = envImpl;

        /* Do the stats definitions. */
        stats = new StatGroup(GROUP_NAME, GROUP_DESC);
        nEvictPasses = new LongStat(stats, EVICTOR_EVICT_PASSES);
        nNodesScanned = new LongStat(stats, EVICTOR_NODES_SCANNED);
        nNodesEvicted = new LongStat(stats, EVICTOR_NODES_EVICTED);
        nRootNodesEvicted = new LongStat(stats, EVICTOR_ROOT_NODES_EVICTED);
        nBINsStripped = new LongStat(stats, EVICTOR_BINS_STRIPPED);
        nBINsMutated = new LongStat(stats, EVICTOR_BINS_MUTATED);

        nLNFetch = new AtomicLongStat(stats, LN_FETCH);
        nBINFetch = new AtomicLongStat(stats, BIN_FETCH);
        nUpperINFetch = new AtomicLongStat(stats, UPPER_IN_FETCH);
        nLNFetchMiss = new AtomicLongStat(stats, LN_FETCH_MISS);
        nBINFetchMiss = new AtomicLongStat(stats, BIN_FETCH_MISS);
        nBINDeltaFetchMiss = new AtomicLongStat(stats, BIN_DELTA_FETCH_MISS);
        nUpperINFetchMiss = new AtomicLongStat(stats, UPPER_IN_FETCH_MISS);
        nThreadUnavailable = new AtomicLongStat(stats, THREAD_UNAVAILABLE);

        nINSparseTarget = new AtomicLong(0);
        nINNoTarget = new AtomicLong(0);
        nINCompactKey = new AtomicLong(0);

        EnumSet<EvictionSource> allSources =
            EnumSet.allOf(EvictionSource.class);
        int numSources = allSources.size();

        binEvictSources = new AtomicLongStat[numSources];
        inEvictSources = new AtomicLongStat[numSources];
        numBatches = new AtomicLongStat[numSources];
        numBatchTargets = new AtomicLong[numSources];

        for (EvictionSource source : allSources) {
            int index = source.ordinal();
            binEvictSources[index] =
                new AtomicLongStat(stats, source.getBINStatDef());
            inEvictSources[index] =
                new AtomicLongStat(stats, source.getUpperINStatDef());
            numBatchTargets[index] = new AtomicLong();
            numBatches[index] =
                new AtomicLongStat(stats, source.getNumBatchesStatDef());
        }

        selector = makeSelector();
        arbiter = new Arbiter(firstEnvImpl);

        logger = LoggerUtils.getLogger(getClass());
        reentrancyGuard = new ReentrancyGuard(firstEnvImpl, logger);
        shutdownRequested = new AtomicBoolean(false);

        DbConfigManager configManager = firstEnvImpl.getConfigManager();

        int corePoolSize =
            configManager.getInt(EnvironmentParams.EVICTOR_CORE_THREADS);
        int maxPoolSize =
            configManager.getInt(EnvironmentParams.EVICTOR_MAX_THREADS);
        long keepAliveTime =
            configManager.getDuration(EnvironmentParams.EVICTOR_KEEP_ALIVE);
        terminateMillis = configManager.getDuration
            (EnvironmentParams.EVICTOR_TERMINATE_TIMEOUT);
        dbCacheClearCount =
            configManager.getInt(EnvironmentParams.ENV_DB_CACHE_CLEAR_COUNT);

        RejectedExecutionHandler rejectHandler =
            new RejectEvictHandler(nThreadUnavailable);

        evictionPool =
            new ThreadPoolExecutor(corePoolSize,
                                   maxPoolSize,
                                   keepAliveTime,
                                   TimeUnit.MILLISECONDS,
                                   new ArrayBlockingQueue<Runnable>(1),
                                   new StoppableThreadFactory(firstEnvImpl,
                                                              "JEEvictor",
                                                              logger),
                                   rejectHandler);

        runEvictor =
            configManager.getBoolean(EnvironmentParams.ENV_RUN_EVICTOR);

        allowBinDeltas = configManager.getBoolean
            (EnvironmentParams.EVICTOR_ALLOW_BIN_DELTAS);

        mutateBins = configManager.getBoolean
            (EnvironmentParams.EVICTOR_MUTATE_BINS);

        /*
         * Request notification of mutable property changes. Do this after all
         * fields in the evictor have been initialized, in case this is called
         * quite soon.
         */
        firstEnvImpl.addConfigObserver(this);
    }

    /**
     * Respond to config updates.
     */
    public void envConfigUpdate(DbConfigManager configManager,
                                EnvironmentMutableConfig ignore)
        throws DatabaseException {

        int corePoolSize =
            configManager.getInt(EnvironmentParams.EVICTOR_CORE_THREADS);
        int maxPoolSize =
            configManager.getInt(EnvironmentParams.EVICTOR_MAX_THREADS);
        long keepAliveTime =
            configManager.getDuration(EnvironmentParams.EVICTOR_KEEP_ALIVE);
        terminateMillis = configManager.getDuration
            (EnvironmentParams.EVICTOR_TERMINATE_TIMEOUT);
        dbCacheClearCount =
            configManager.getInt(EnvironmentParams.ENV_DB_CACHE_CLEAR_COUNT);

        evictionPool.setCorePoolSize(corePoolSize);
        evictionPool.setMaximumPoolSize(maxPoolSize);
        evictionPool.setKeepAliveTime(keepAliveTime, TimeUnit.MILLISECONDS);
        runEvictor =
            configManager.getBoolean(EnvironmentParams.ENV_RUN_EVICTOR);
    }

    /**
     * Whether this is the new LRU evictor or not.
     */
    public boolean isNewEvictor() {
        return false;
    }

    public boolean getMutateToBINDeltas() {
        return mutateBins;
    }

    /**
     * @hidden
     * Return the ThreadPool, used by unit testing only.
     */
    public ThreadPoolExecutor getThreadPool() {
        return evictionPool;
    }

    /* For unit test only */
    TargetSelector getSelector() {
       return selector;
    }

    /*
     * Node selection varies based on whether this is a private or shared
     * cache.
     */
    abstract TargetSelector makeSelector();

    public abstract void addEnvironment(EnvironmentImpl envImpl);

    public abstract void removeEnvironment(EnvironmentImpl envImpl);

    public abstract void setEnabled(boolean v);

    public abstract boolean isEnabled();

    public abstract boolean useDirtyLRUSet();

    /**
     * Only supported by SharedEvictor.
     */
    public abstract boolean checkEnv(EnvironmentImpl env);

    /**
     * Called whenever INs are added to, or removed from, the INList.
     */
    public abstract void noteINListChange(int nINs);

    public abstract void addBack(IN node);

    public abstract void addFront(IN node);

    public abstract void moveBack(IN node);

    public abstract void moveFront(IN node);

    public abstract void remove(IN node);

    public abstract void moveToMixedLRU(IN node);

    public abstract boolean contains(IN node);

    /**
     * This is where the real work is done.
     * Can execute concurrently, called by app threads or by background evictor
     */
    abstract void doEvict(EvictionSource source, boolean backgroundIO)
        throws DatabaseException;

    /**
     * Must appear in base Evictor class because it is used in unit test.
     */
    abstract long evictBatch(Evictor.EvictionSource source,
                             boolean backgroundIO,
                             long maxEvictBytes,
                             EvictionDebugStats evictionStats);

    /**
     * Do some eviction before proceeding on with another operation.
     *
     * Note that this method is intentionally not synchronized in order to
     * minimize overhead when checking for critical eviction.  This method is
     * called from application threads for every cursor operation, and by many
     * daemon threads.
     */
    public void doCriticalEviction(boolean backgroundIO) {

        if (arbiter.isOverBudget()) {

            /*
             * Any time there's excessive cache usage, let the thread pool know
             * there's work to do.
             */
            alert();

            /*
             * If this is an application thread, only do eviction if the
             * memory budget overage fulfills the critical eviction
             * requirements. We want to avoid having application thread do
             * eviction.
             */
            if (arbiter.needCriticalEviction()) {
                doEvict(EvictionSource.CRITICAL, backgroundIO);
            }
        }
    }

    /**
     * Do a check on whether synchronous eviction is needed.
     *
     * Note that this method is intentionally not synchronized in order to
     * minimize overhead when checking for critical eviction.  This method is
     * called from daemon threads for every operation.
     */
    public void doDaemonEviction(boolean backgroundIO) {

        if (arbiter.isOverBudget()) {

            /*
             * Any time there's excessive cache usage, let the thread pool know
             * there's work to do.
             */
            alert();

            /*
             * Only do eviction if the memory budget overage fulfills the
             * critical eviction requirements. This allows evictor threads to
             * take the burden of eviction whenever possible, rather than
             * slowing other threads and risking a growing cleaner or
             * compressor backlog.
             */
            if (arbiter.needCriticalEviction()) {
                doEvict(EvictionSource.DAEMON, backgroundIO);
            }
        }
    }

    /*
     * Eviction invoked by the API
     */
    public void doManualEvict()
        throws DatabaseException {

        doEvict(EvictionSource.MANUAL, true); // backgroundIO
    }

    /**
     * Evict a specific IN, used by cache modes.
     */
    public abstract void doEvictOneIN(IN target, EvictionSource source);

    /**
     * Let the eviction pool know there's work to do.
     */
    public void alert() {
        if (!runEvictor) {
            return;
        }

        evictionPool.execute
            (new BackgroundEvictTask(this, true /* backgroundIO */));
    }

    /**
     * Request and wait for a shutdown of all running eviction tasks.
     */
    public void shutdown() {

        /*
         * Set the shutdown flag so that outstanding eviction tasks end
         * early. The call to evictionPool.shutdown is a ThreadPoolExecutor
         * call, and is an orderly shutdown that waits for and in flight tasks
         * to end.
         */
        shutdownRequested.set(true);
        evictionPool.shutdown();

        /*
         * AwaitTermination will wait for the timeout period, or will be
         * interrupted, but we don't really care which it is. The evictor
         * shouldn't be interrupted, but if it is, something urgent is
         * happening.
         */
        boolean shutdownFinished = false;
        try {
            shutdownFinished =
                evictionPool.awaitTermination(terminateMillis,
                                              TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            /* We've been interrupted, just give up and end. */
        } finally {
            if (!shutdownFinished) {
                evictionPool.shutdownNow();
            }
        }
    }

    public void requestShutdownPool() {
        shutdownRequested.set(true);
        evictionPool.shutdown();
    }


    public boolean isCacheFull() {
        return arbiter.isCacheFull();
    }

    public boolean wasCacheEverFull() {
        return arbiter.wasCacheEverFull();
    }

   /* For unit testing only. */
    public void setRunnableHook(TestHook<Boolean> hook) {
        arbiter.setRunnableHook(hook);
    }

    /* For unit testing only. */
    public void setPreEvictINHook(TestHook<Object> hook) {
        preEvictINHook = hook;
    }

    /* For unit testing only. */
    public void setEvictProfileHook(TestHook<IN> hook) {
        evictProfile = hook;
    }

    public StatGroup getStatsGroup() {
        return stats;
    }

    /**
     * Load stats.
     */
    public StatGroup loadStats(StatsConfig config) {

        StatGroup copy = stats.cloneGroup(config.getClear());

        /*
         * These stats are not cleared. They represent the current state of
         * the cache.
         */
        new LongStat(copy, CACHED_IN_SPARSE_TARGET, nINSparseTarget.get());
        new LongStat(copy, CACHED_IN_NO_TARGET, nINNoTarget.get());
        new LongStat(copy, CACHED_IN_COMPACT_KEY, nINCompactKey.get());
        if (selector != null) {
            copy.addAll(selector.loadStats(config));
        }
        copy.addAll(arbiter.loadStats(config));

        /*
         * The number and average size of batches, by type of caller, is
         * calculated each time we collect stats.
         */
        EnumSet<EvictionSource> allSources =
            EnumSet.allOf(EvictionSource.class);

        for (EvictionSource source : allSources) {
            int index = source.ordinal();

            new IntegralLongAvgStat(copy,
                                    source.getAvgBatchStatDef(),
                                    numBatchTargets[index].get(),
                                    copy.getAtomicLong
                                        (source.getNumBatchesStatDef()));
            if (config.getClear()) {
                numBatchTargets[index].set(0);
            }

        }

        return copy;
    }

    public void incEvictStats(EvictionSource source, IN target) {
        if (target.isBIN()) {
            binEvictSources[source.ordinal()].increment();
        } else {
            inEvictSources[source.ordinal()].increment();
        }
    }

    /**
     * Update the appropriate fetch stat, based on node type.
     */
    public void incLNFetchStats(boolean isMiss) {
        nLNFetch.increment();
        if (isMiss) {
            nLNFetchMiss.increment();
        }
    }

    public void incBINFetchStats(boolean isMiss, boolean isDelta) {
        nBINFetch.increment();
        if (isMiss) {
            nBINFetchMiss.increment();
            if (isDelta) {
                nBINDeltaFetchMiss.increment();
            }
        }
    }

    public void incFullBINFetchMissStats() {
        nBINFetchMiss.increment();
    }

    public void incINFetchStats(boolean isMiss) {
        nUpperINFetch.increment();
        if (isMiss) {
            nUpperINFetchMiss.increment();
        }
    }

    public AtomicLong getNINSparseTarget() {
        return nINSparseTarget;
    }

    public AtomicLong getNINNoTarget() {
        return nINNoTarget;
    }

    public AtomicLong getNINCompactKey() {
        return nINCompactKey;
    }


    static class ReentrancyGuard {
        private final ConcurrentHashMap<Thread, Thread> activeThreads;
        private final EnvironmentImpl envImpl;
        private final Logger logger;

        ReentrancyGuard(EnvironmentImpl envImpl, Logger logger) {
            this.envImpl = envImpl;
            this.logger = logger;
            activeThreads = new ConcurrentHashMap<Thread, Thread>();
        }

        boolean enter() {
            Thread thisThread = Thread.currentThread();
            if (activeThreads.containsKey(thisThread)) {
                /* We don't really expect a reentrant call. */
                LoggerUtils.severe(logger, envImpl,
                                   "reentrant call to eviction from " +
                                   LoggerUtils.getStackTrace());

                /* If running w/assertions, in testing mode, assert here. */
                assert false: "reentrant call to eviction from " +
                    LoggerUtils.getStackTrace();
                return false;
            }

            activeThreads.put(thisThread, thisThread);
            return true;
        }

        void leave() {
            assert activeThreads.contains(Thread.currentThread());
            activeThreads.remove(Thread.currentThread());
        }
    }

    static class BackgroundEvictTask implements Runnable {

        private final Evictor evictor;
        private final boolean backgroundIO;

        BackgroundEvictTask(Evictor evictor,
                            boolean backgroundIO) {
            this.evictor = evictor;
            this.backgroundIO = backgroundIO;
        }

        public void run() {
            evictor.doEvict(EvictionSource.EVICTORTHREAD, backgroundIO);
        }
    }

    static class RejectEvictHandler implements RejectedExecutionHandler {
        private final AtomicLongStat threadUnavailableStat;

        RejectEvictHandler(AtomicLongStat threadUnavailableStat) {
            this.threadUnavailableStat = threadUnavailableStat;
        }

        public void rejectedExecution(Runnable r,
                                      ThreadPoolExecutor executor) {
            threadUnavailableStat.increment();
        }
    }

    /**
     * Caches DatabaseImpls to reduce DbTree.getDb overhead.
     *
     * SharedEvictor, unlike PrivateEvictor, must maintain a cache map for each
     * EnvironmentImpl, since each cache map is logically associated with a
     * single DbTree instance.
     */
    static class DbCache {

        boolean shared = false;

        int nOperations = 0;

        int dbCacheClearCount = 0;

        final Map<EnvironmentImpl, Map<DatabaseId, DatabaseImpl>> envMap;

        final Map<DatabaseId, DatabaseImpl> dbMap;

        DbCache(boolean shared, int dbCacheClearCount) {

            this.shared = shared;
            this.dbCacheClearCount = dbCacheClearCount;

            if (shared) {
                envMap = 
                new HashMap<EnvironmentImpl, Map<DatabaseId, DatabaseImpl>>();

                dbMap = null;
            } else {
                dbMap = new HashMap<DatabaseId, DatabaseImpl>();
                envMap = null;
            }
        }

        /**
         * Calls DbTree.getDb for the given environment and database ID, and
         * caches the result to optimize multiple calls for the same DB.
         *
         * @param envImpl identifies which environment the dbId parameter
         * belongs to.  For PrivateEvictor, it is the same as the
         * Evictor.firstEnvImpl field.
         *
         * @param dbId is the DB to get.
         */
        DatabaseImpl getDb(EnvironmentImpl env, DatabaseId dbId) {

            Map<DatabaseId, DatabaseImpl> map;

            if (shared) {
                map = envMap.get(env);
                if (map == null) {
                    map = new HashMap<DatabaseId, DatabaseImpl>();
                    envMap.put(env, map);
                }
            } else {
                map = dbMap;
            }

            /*
             * Clear DB cache after dbCacheClearCount operations, to
             * prevent starving other threads that need exclusive access to
             * the MapLN (for example, DbTree.deleteMapLN).  [#21015]
             *
             * Note that we clear the caches for all environments after
             * dbCacheClearCount total operations, rather than after
             * dbCacheClearCount operations for a single environment,
             * because the total is a more accurate representation of
             * elapsed time, during which other threads may be waiting for
             * exclusive access to the MapLN.
             */
            nOperations += 1;
            if ((nOperations % dbCacheClearCount) == 0) {
                releaseDbs(env);
            }

            return env.getDbTree().getDb(dbId, -1, map);
        }

        /**
         * Calls DbTree.releaseDb for cached DBs, and clears the cache.
         */
        void releaseDbs(EnvironmentImpl env) {
            if (shared) {
                for (Map.Entry<EnvironmentImpl, Map<DatabaseId, DatabaseImpl>>
                     entry : envMap.entrySet()) {

                    final EnvironmentImpl sharingEnv = entry.getKey();
                    final Map<DatabaseId, DatabaseImpl> map = entry.getValue();

                    sharingEnv.getDbTree().releaseDbs(map);
                    map.clear();
                }
            } else {
                env.getDbTree().releaseDbs(dbMap);
                dbMap.clear();
            }
        }
    }
}
