/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.evictor;

import java.util.logging.Level;

import com.sleepycat.je.CacheMode;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.dbi.INList;
import com.sleepycat.je.evictor.TargetSelector.ScanInfo;
import com.sleepycat.je.evictor.TargetSelector.SetupInfo;
import com.sleepycat.je.evictor.LRUEvictor.EvictionDebugStats;
import com.sleepycat.je.recovery.Checkpointer;
import com.sleepycat.je.tree.BIN;
import com.sleepycat.je.tree.ChildReference;
import com.sleepycat.je.tree.IN;
import com.sleepycat.je.tree.Node;
import com.sleepycat.je.tree.SearchResult;
import com.sleepycat.je.tree.Tree;
import com.sleepycat.je.tree.WithRootLatched;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.LoggerUtils;
import com.sleepycat.je.utilint.TestHookExecute;

/**
 * The Evictor is responsible for maintaining the JE cache. Since object sizes
 * are not directly manipulated in a Java application, the cache is actually a
 * collection of in-memory btree nodes, implemented by
 * com.sleepycat.je.dbi.INList. Nodes are selected from the INList for removal,
 * which is done by detaching them from the in-memory tree, and by removing
 * them from the INList. Once all references to them are removed, they can be
 * GC'd by the JVM.
 *
 * There are three main components.
 *
 * Arbiter: queries the memory budget to decide whether eviction is needed
 * TargetSelector : chooses a target node
 * Evictor: does the work of detaching the node.
 *
 * The TargetSelector and Evictor classes are subclassed to provide
 * private/shared cache implementations. A shared cache is used by multiple
 * environments within a single JVM, and is seen logically as a single INList
 * collection, although it is implemented by an umbrella over multiple INLists.
 *
 * The Evictor owns a thread pool which is available to handle eviction tasks.
 * Eviction is carried out by three types of threads:
 * 1. The application thread, in the course of doing critical eviction
 * 2. Daemon threads, such as the cleaner or INCompressor, in the course of
 *    doing their respective duties
 * 3. Eviction pool threads
 *
 * We prefer that the eviction pool threads do as much of the eviction as
 * possible, and that the application threads do as little, because eviction
 * adds latency to the perceived application response time. To date, it has
 * been impossible to completely remove eviction responsiblities from the
 * application threads, because the process doesn't have sufficient feedback,
 * and can incur an OutOfMemoryException.
 *
 * The eviction pool is a standard java.util.concurrent thread pool, and can
 * be mutably configured in terms of core threads, max threads, and keepalive
 * times.
 *
 * Since three types of threads can concurrently do eviction, it's important
 * that eviction is both thread safe and as parallel as possible.  Memory
 * thresholds are generally accounted for in an unsynchronized fashion, and are
 * seen as advisory. The only point of true synchronization is around the
 * selection of a node for eviction. The act of eviction itself can be done
 * concurrently.
 *
 * The eviction method is not reentrant, and a simple concurrent hash map
 * of threads is used to prevent recursive calls.
 */
public abstract class INListEvictor extends Evictor {

    INListEvictor(EnvironmentImpl envImpl)
        throws DatabaseException {

        super(envImpl);
    }

    /**
     * Only supported by SharedEvictor.
     * Noop for PrivateSelector
     */
    @Override
    public void addEnvironment(EnvironmentImpl envImpl) {
        selector.addEnvironment(envImpl);
    }

    /**
     * Only supported by SharedEvictor.
     * Noop for PrivateSelector
     */
    @Override
    public void removeEnvironment(EnvironmentImpl envImpl) {
        selector.removeEnvironment(envImpl);
    }

    /**
     * Only supported by SharedEvictor.
     * Noop for PrivateSelector
     */
    @Override
    public boolean checkEnv(EnvironmentImpl env) {
        return selector.checkEnv(env);
    }

    @Override
    public void setEnabled(boolean v) {
    }

    @Override
    public boolean isEnabled() {
        assert(false);
        return true;
    }

    @Override
    public boolean useDirtyLRUSet() {
        assert(false);
        return false;
    }

    /**
     * Called whenever INs are added to, or removed from, the INList.
     * Noop for PrivateSelector
     */
    @Override
    public void noteINListChange(int nINs) {
        selector.noteINListChange(nINs);
    }

    @Override
    public void addBack(IN node) {
    }
    
    @Override
    public void addFront(IN node) {
    }
    
    @Override
    public void moveBack(IN node) {
    }
    
    @Override
    public void moveFront(IN node) {
    }
    
    @Override
    public void remove(IN node) {
    }
    
    @Override
    public void moveToMixedLRU(IN node) {
    }

    @Override
    public boolean contains(IN node) {
        return false;
    }

    /**
     * Evict a specific IN, used by cache modes.
     */
    public void doEvictOneIN(IN target, EvictionSource source) {
        if (!reentrancyGuard.enter()) {
            return;
        }

        try {
            assert(target.isBIN());
            assert(target.isLatchExclusiveOwner());

            target.releaseLatch();

            evictIN(target, false /* backgroundIO */, source, null);
        } finally {
            reentrancyGuard.leave();
        }
    }

    /**
     * Can execute concurrently, called by app threads or by background evictor
     */
    @Override
    void doEvict(EvictionSource source, boolean backgroundIO)
        throws DatabaseException {

        if (!reentrancyGuard.enter()) {
            return;
        }

        try {

            /*
             * Repeat as necessary to keep up with allocations.  Stop if no
             * progress is made, to prevent an infinite loop.
             */
            boolean progress = true;
            int nBatches = 0;
            long bytesEvicted = 0;
            numBatches[source.ordinal()].increment();

            while (progress &&
                   (nBatches < MAX_BATCHES_PER_RUN) &&
                   !shutdownRequested.get()) {

                /* Get some work from the arbiter. */
                long maxEvictBytes = arbiter.getEvictionPledge();

                /* Nothing to do. */
                if (maxEvictBytes == 0) {
                    break;
                }

                bytesEvicted = evictBatch(source,
                                          backgroundIO,
                                          maxEvictBytes,
                                          null);
                if (bytesEvicted == 0) {
                    progress = false;
                }

                nBatches += 1;
            }

            /* Really for debugging. */
            if (source == EvictionSource.EVICTORTHREAD) {
                if (logger.isLoggable(Level.FINEST)) {
                    LoggerUtils.finest(logger, firstEnvImpl,
                                       "Thread evicted " + bytesEvicted +
                                       " bytes in " + nBatches + " batches");
                }
            }
        } finally {
            reentrancyGuard.leave();
        }
    }

    /**
     * Each iteration will attempt to evict maxEvictBytes, but will give up
     * after a complete pass over the INList, or if there is nothing more to
     * evict, due to actions by concurrently executing threads. This method is
     * thread safe and may be called concurrently.
     *
     * @return the number of bytes evicted, or zero if no progress was made.
     * Note that if the INList is completely empty, it's still possible to
     * return a non-zero number of bytes due to special eviction of items such
     * as utilization info, even though no IN eviction happened.
     */
    @Override
    long evictBatch(Evictor.EvictionSource source,
                    boolean backgroundIO,
                    long maxEvictBytes,
                    EvictionDebugStats evictionStats)
        throws DatabaseException {

        int numNodesScannedThisBatch = 0;
        nEvictPasses.increment();

        assert TestHookExecute.doHookSetupIfSet(evictProfile);

        /*
         * Perform class-specific per-batch processing, in advance of getting a
         * batch. This is done under the TargetSelector mutex.
         *
         * TODO: special eviction is done serially. We may want to absolve
         * application threads of that responsibility, to avoid blocking, and
         * only have evictor threads do special eviction.
         */
        final SetupInfo setupInfo =
            selector.startBatch(true /*doSpecialEviction*/);
        long evictBytes = setupInfo.specialEvictionBytes;
        final int maxINsPerBatch = setupInfo.maxINsPerBatch;
        if (maxINsPerBatch == 0) {
            return evictBytes; // The INList(s) are empty.
        }

        /* Use local caching to reduce DbTree.getDb overhead. [#21330] */
        final DbCache dbCache = new DbCache(firstEnvImpl.getSharedCache(),
                                            dbCacheClearCount);

        try {

            /*
             * Keep evicting until we've freed enough memory or we've visited
             * the maximum number of nodes allowed. Each iteration of the while
             * loop is called an eviction batch.
             *
             * In order to prevent endless evicting, limit this run to one pass
             * over the IN list(s).
             */
            while ((evictBytes < maxEvictBytes) &&
                   (numNodesScannedThisBatch <= maxINsPerBatch) &&
                   arbiter.stillNeedsEviction()) {

                final ScanInfo scanInfo = selector.selectIN(maxINsPerBatch);
                final IN target = scanInfo.target;
                numNodesScannedThisBatch += scanInfo.numNodesScanned;

                if (target == null) {
                    break;
                }

                numBatchTargets[source.ordinal()].incrementAndGet();

                assert TestHookExecute.doHookIfSet(evictProfile, target);

                /*
                 * Check to make sure the DB was not deleted after selecting
                 * it, and prevent the DB from being deleted while we're
                 * working with it.
                 *
                 * Also check that the refreshedDb is the same instance as the
                 * targetDb.  If not, then targetDb was recently evicted; it
                 * and its IN are orphaned and cannot be processed.  [#21686]
                 */
                final DatabaseImpl targetDb = target.getDatabase();

                final DatabaseImpl refreshedDb = 
                dbCache.getDb(targetDb.getDbEnvironment(), targetDb.getId());

                if (refreshedDb != null &&
                    refreshedDb == targetDb &&
                    !refreshedDb.isDeleted()) {
                    if (target.isDbRoot()) {
                        evictBytes += evictRoot(target, backgroundIO);
                    } else {
                        evictBytes +=
                            evictIN(target, backgroundIO, source, null);
                    }
                } else {

                    /*
                     * We don't expect to see an IN that is resident on the
                     * INList with a database that has finished delete
                     * processing, because it should have been removed from the
                     * INList during post-delete cleanup.  It may have been
                     * returned by the INList iterator after being removed from
                     * the INList (because we're using ConcurrentHashMap), but
                     * then IN.getInListResident should return false.
                     */
                    if (targetDb.isDeleteFinished() &&
                        target.getInListResident()) {
                        final String inInfo =
                            " IN type=" + target.getLogType() + " id=" +
                            target.getNodeId() + " not expected on INList";
                        final String errMsg = (refreshedDb == null) ?
                            inInfo :
                            ("Database " + refreshedDb.getDebugName() +
                             " id=" + refreshedDb.getId() + " rootLsn=" +
                             DbLsn.getNoFormatString
                             (refreshedDb.getTree().getRootLsn()) +
                              ' ' + inInfo);
                        throw EnvironmentFailureException.
                            unexpectedState(errMsg);
                    }
                }
            }
        } finally {
            nNodesScanned.add(numNodesScannedThisBatch);
            dbCache.releaseDbs(firstEnvImpl);
        }

        return evictBytes;
    }

    /**
     * Evict this DB root node.  [#13415] Must be thread safe, executes
     * concurrently.
     *
     * @return number of bytes evicted.
     */
    private long evictRoot(final IN target,
                           final boolean backgroundIO)
        throws DatabaseException {

        final DatabaseImpl db = target.getDatabase();
        /* SharedEvictor uses multiple envs, do not use superclass envImpl. */
        final EnvironmentImpl useEnvImpl = db.getDbEnvironment();
        final INList inList = useEnvImpl.getInMemoryINs();

        class RootEvictor implements WithRootLatched {

            boolean flushed = false;
            long evictBytes = 0;

            public IN doWork(ChildReference root)
                throws DatabaseException {

                /*
                 * Do not call fetchTarget since this root or DB should be
                 * resident already if it is to be the target of eviction. If
                 * it is not present, it has been evicted by another thread and
                 * should not be fetched for two reasons: 1) this would be
                 * counterproductive, 2) to guard against bringing in a root
                 * for an evicted DB.  The check for getInListResident below
                 * also guards against this later possibility.  [#21686]
                 */
                IN rootIN = (IN) root.getTarget();
                if (rootIN == null) {
                    return null;
                }
                rootIN.latch(CacheMode.UNCHANGED);
                try {
                    /* Re-check that all conditions still hold. */
                    boolean isDirty = rootIN.getDirty();
                    if (rootIN == target &&
                        rootIN.getInListResident() &&
                        rootIN.isDbRoot() &&
                        rootIN.isEvictable() &&
                        !(useEnvImpl.isReadOnly() && isDirty)) {
                        boolean logProvisional =
                            coordinateWithCheckpoint(rootIN, null /*parent*/);

                        /* Flush if dirty. */
                        if (isDirty) {
                            long newLsn = rootIN.log
                                (useEnvImpl.getLogManager(),
                                 false, // allowDeltas
                                 false, // allowCompress
                                 logProvisional,
                                 backgroundIO,
                                 null); // parent
                            root.setLsn(newLsn);
                            flushed = true;
                        }

                        /* Take off the INList and adjust memory budget. */
                        inList.remove(rootIN);
                        evictBytes = rootIN.getBudgetedMemorySize();

                        /* Evict IN. */
                        root.clearTarget();

                        /* Stats */
                        nRootNodesEvicted.increment();
                    }
                } finally {
                    rootIN.releaseLatch();
                }

                return null;
            }
        }

        /* Attempt to evict the DB root IN. */
        RootEvictor evictor = new RootEvictor();
        db.getTree().withRootLatchedExclusive(evictor);

        /* If the root IN was flushed, write the dirtied MapLN. */
        if (evictor.flushed) {
            useEnvImpl.getDbTree().modifyDbRoot(db);
        }

        return evictor.evictBytes;
    }

    /**
     * Strip or evict this node. Must be thread safe, executes concurrently.
     *
     * @param source is EvictSource.CRITICAL or EVICTORTHREAD when this
     * operation is invoked by the evictor (either critical eviction or the
     * evictor background thread), and is EvictSource.CACHEMODE if invoked by a
     * user operation using CacheMode.EVICT_BIN.  If CACHEMODE, we will perform
     * the eviction regardless of whether:
     *  1) we have to wait for a latch, or
     *  2) the IN generation changes, or
     *  3) we are able to strip LNs.
     *
     * If not CACHEMODE, any of the above conditions will prevent eviction.
     *
     * @return number of bytes evicted.
     */
    long evictIN(IN target,
                 boolean backgroundIO,
                 EvictionSource source,
                 EvictionDebugStats evictionStats)
        throws DatabaseException {

        DatabaseImpl db = target.getDatabase();
        /* SharedEvictor uses multiple envs, do not use superclass envImpl. */
        EnvironmentImpl useEnvImpl = db.getDbEnvironment();
        long evictedBytes = 0;

        /*
         * Non-BIN INs are evicted by detaching them from their parent.  For
         * BINS, the first step is to remove deleted entries by compressing
         * the BIN. The evictor indicates that we shouldn't fault in
         * non-resident children during compression. After compression,
         * LN logging and LN stripping may be performed.
         *
         * If LN stripping is used, first we strip the BIN by logging any dirty
         * LN children and detaching all its resident LN targets.  If we make
         * progress doing that, we stop and will not evict the BIN itself until
         * possibly later.  If it has no resident LNs then we evict the BIN
         * itself using the "regular" detach-from-parent routine.
         *
         * If the cleaner is doing clustering, we don't do BIN stripping if we
         * can write out the BIN.  Specifically LN stripping is not performed
         * if the BIN is dirty AND the BIN is evictable AND cleaner
         * clustering is enabled.  In this case the BIN is going to be written
         * out soon, and with clustering we want to be sure to write out the
         * LNs with the BIN; therefore we don't do stripping.
         */

        /*
         * Use latchNoWait because if it's latched we don't want the cleaner
         * to hold up eviction while it migrates an entire BIN.  Latched INs
         * have a high generation value, so not evicting makes sense.  Pass
         * false because we don't want to change the generation during the
         * eviction process.
         */
        boolean inline = (source == EvictionSource.CACHEMODE);
        if (inline) {
            target.latch(CacheMode.UNCHANGED);
        } else {
            if (!target.latchNoWait(CacheMode.UNCHANGED)) {
                return evictedBytes;
            }
        }
        boolean targetIsLatched = true;
        try {

            /*
             * After latching it, ensure that this node was not evicted by
             * another thread.  Do this now, before the Btree lookup, since we
             * should not compress or evict LNs for an orphaned IN. [#21686]
             */
            if (!target.getInListResident()) {
                return evictedBytes;
            }

            /*
             * Try to reclaim memory without evicting the entire node. Note
             * this may dirty the IN, for example, if dirty LNs are logged.
             */
            evictedBytes = target.partialEviction();
            evictedBytes = evictedBytes & ~IN.NON_EVICTABLE_IN;

            if (target.isBIN() && (evictedBytes > 0)) {
                nBINsStripped.increment();
            }

            /*
             * If we were able to free any memory by partial eviction above,
             * then we postpone eviction of the BIN until a later pass.
             */
            if (!inline && evictedBytes != 0) {
                return evictedBytes;
            }

            if (!target.isEvictable()) {
                return evictedBytes;
            }
            /* Regular eviction. */
            Tree tree = db.getTree();

            /*
             * Unit testing.  The target is latched and we are about to release
             * that latch and search for the parent.  Make sure that other
             * operations, such as dirtying an LN in the target BIN, can occur
             * safely in this window.  [#18227]
             */
            assert TestHookExecute.doHookIfSet(preEvictINHook);

            /* getParentINForChildIN unlatches target. */
            targetIsLatched = false;

            /*
             * Pass false for doFetch to avoid fetching a full BIN when a
             * delta is in cache.  This also avoids a fetch when the node
             * was evicted while unlatched, but that should be very rare.
             */
            SearchResult result = tree.getParentINForChildIN(
                target, true /*requireExactMatch*/, CacheMode.UNCHANGED,
                -1 /*level*/, null /*trackingList*/, false /*doFetch*/);

            if (result.exactParentFound) {
                evictedBytes = evictIN(target, result.parent,
                                       result.index, backgroundIO, source);
            } else {
                if (result.parent != null) {
                    result.parent.releaseLatch();
                }
            }
            return evictedBytes;
        } finally {
            if (targetIsLatched) {
                target.releaseLatch();
            }
        }
    }

    /**
     * Evict an IN. Dirty nodes are logged before they're evicted.
     */
    private long evictIN(IN child,
                         IN parent,
                         int index,
                         boolean backgroundIO,
                         EvictionSource source)
        throws DatabaseException {

        long evictBytes = 0;
        try {
            assert parent.isLatchExclusiveOwner();

            long oldGenerationCount = child.getGeneration();

            /*
             * Get a new reference to the child, in case the reference
             * saved in the selection list became out of date because of
             * changes to that parent.
             */
            IN renewedChild = (IN) parent.getTargetAllowBINDelta(index);

            if (renewedChild == null) {
                return evictBytes;
            }

            boolean inline = (source == EvictionSource.CACHEMODE);
            if (!inline && renewedChild.getGeneration() > oldGenerationCount) {
                return evictBytes;
            }

            /*
             * See the evictIN() method in this class for an explanation for
             * calling latchNoWait().
             */
            if (inline) {
                renewedChild.latch(CacheMode.UNCHANGED);
            } else {
                if (!renewedChild.latchNoWait(CacheMode.UNCHANGED)) {
                    return evictBytes;
                }
            }
            try {
                if (mutateBins && renewedChild.isBIN() && !inline) {
                    final BIN bin = (BIN) renewedChild;
                    if (bin.canMutateToBINDelta()) {
                        evictBytes = bin.mutateToBINDelta();
                        if (evictBytes > 0) {
                            nBINsMutated.increment();
                            bin.setGeneration(CacheMode.DEFAULT);
                            return evictBytes;
                        }
                    }
                }

                if (!renewedChild.isEvictable()) {
                    return evictBytes;
                }

                DatabaseImpl db = renewedChild.getDatabase();
                /* Do not use superclass envImpl. */
                EnvironmentImpl useEnvImpl = db.getDbEnvironment();

                /*
                 * Log the child if dirty and env is not r/o. Remove
                 * from IN list.
                 */
                long renewedChildLsn = DbLsn.NULL_LSN;
                boolean newChildLsn = false;
                if (renewedChild.getDirty()) {
                    if (!useEnvImpl.isReadOnly()) {
                        boolean logProvisional =
                            coordinateWithCheckpoint(renewedChild, parent);

                        /*
                         * Log a full version (no deltas) and with cleaner
                         * migration allowed.  Allow compression of deleted
                         * slots in full version BINs.
                         */
                        renewedChildLsn = renewedChild.log
                            (useEnvImpl.getLogManager(),
                             allowBinDeltas,
                             true /*allowCompress*/,
                             logProvisional,
                             backgroundIO,
                             parent);
                        newChildLsn = true;
                    }
                } else {
                    renewedChildLsn = parent.getLsn(index);
                }

                if (renewedChildLsn != DbLsn.NULL_LSN) {
                    /* Take this off the inlist. */
                    useEnvImpl.getInMemoryINs().remove(renewedChild);

                    evictBytes = renewedChild.getBudgetedMemorySize();
                    if (newChildLsn) {

                        /*
                         * Update the parent so its reference is
                         * null and it has the proper LSN.
                         */
                        parent.updateNode
                            (index, null /*node*/, renewedChildLsn,
                             0 /*lastLoggedSize*/, null /*lnSlotKey*/);
                    } else {

                        /*
                         * Null out the reference, but don't dirty
                         * the node since only the reference
                         * changed.
                         */
                        parent.updateNode
                            (index, (Node) null /*node*/,
                             null /*lnSlotKey*/);
                    }

                    /* Stats */
                    nNodesEvicted.increment();
                    incEvictStats(source, renewedChild);
                }
            } finally {
                renewedChild.releaseLatch();
            }
        } finally {
            parent.releaseLatch();
        }

        return evictBytes;
    }

    /**
     * Coordinates an eviction with an in-progress checkpoint and returns
     * whether provisional logging is needed.
     *
     * @return true if the target must be logged provisionally.
     */
    private boolean coordinateWithCheckpoint(IN target, IN parent) {

        /* SharedEvictor uses multiple envs, do not use superclass envImpl. */
        EnvironmentImpl useEnvImpl = target.getDatabase().getDbEnvironment();

        /*
         * The checkpointer could be null if it was shutdown or never
         * started.
         */
        Checkpointer ckpter = useEnvImpl.getCheckpointer();
        if (ckpter == null) {
            return false;
        }
        return ckpter.coordinateEvictionWithCheckpoint(target, parent);
    }
}
