/*
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.log.entry;

import java.nio.ByteBuffer;

import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.tree.BIN;
import com.sleepycat.je.tree.IN;
import com.sleepycat.je.utilint.DbLsn;

/**
 * Holds a partial BIN that serves as a live BIN delta.
 *
 * A live delta (unlike a the obsolete OldBINDelta, which is contained in an
 * OldBINDeltaLogEntry) may appear in the Btree to serve as an incomplete BIN.
 */
public class BINDeltaLogEntry extends INLogEntry<BIN> {

    public BINDeltaLogEntry(Class<BIN> logClass) {
        super(logClass);
    }

    /**
     * When constructing an entry for writing to the log, use LOG_BIN_DELTA.
     */
    public BINDeltaLogEntry(BIN bin) {
        super(bin, true /*isBINDelta*/);
    }

    @Override
    void readMainItem(BIN bin, ByteBuffer entryBuffer, int logVersion) {
        bin.readDeltasFromLog(entryBuffer, logVersion);
        bin.setLastFullLsn(LogUtils.readPackedLong(entryBuffer));
    }

    @Override
    int getMainItemSize(BIN bin) {
        return bin.getDeltasLogSize() +
            LogUtils.getPackedLongLogSize(bin.getLastFullVersion());

    }

    @Override
    void writeMainItem(BIN bin, ByteBuffer destBuffer) {
        bin.writeDeltasToLog(destBuffer);
        LogUtils.writePackedLong(destBuffer, getLastFullLsn());
    }

    /**
     * Currently used by recovery to fetch full BIN.  Override can be removed
     * when recovery is optimized for deltas.
     */
    @Override
    public IN getIN(DatabaseImpl dbImpl) {
        return getMainItem().reconstituteBIN(dbImpl);
    }

    public long getLastFullLsn() {
        return getMainItem().getLastFullVersion();
    }

    @Override
    public long getPrevFullLsn() {
        return DbLsn.NULL_LSN;
    }

    @Override
    public StringBuilder dumpEntry(StringBuilder sb, boolean verbose) {
        super.dumpEntry(sb, verbose);
        if (getLastFullLsn() != DbLsn.NULL_LSN) {
            sb.append("<lastFullLsn>");
            sb.append(DbLsn.getNoFormatString(getLastFullLsn()));
            sb.append("</lastFullLsn>");
        }
        return sb;
    }
}
