/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.log.entry;

import java.nio.ByteBuffer;

import com.sleepycat.je.dbi.DatabaseId;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.LogEntryHeader;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.tree.IN;
import com.sleepycat.je.utilint.DbLsn;

/**
 * INLogEntry embodies all IN log entries.
 * On disk, an IN log entry contains (pre version 6)
 * <pre>
 *        IN
 *        database id
 *        obsolete LSN  -- in version 2
 *
 * (version 6)
 *        database id
 *        obsolete LSN
 *        IN
 * </pre>
 */
public class INLogEntry<T extends IN> extends BaseEntry<T>
    implements LogEntry, INContainingEntry {

    /*
     * Persistent fields in an IN entry.
     */
    private T in;
    private DatabaseId dbId;

    /*
     * obsoleteFile was added in version 1, and changed to obsoleteLsn in
     * version 2.  If the offset is zero in the LSN, we read a version 1 entry
     * since only the file number was stored.  In version 8, we renamed
     * obsoleteLsn to prevFullLsn and added prevDeltaLsn.
     */
    private long prevFullLsn;
    private long prevDeltaLsn;

    /**
     * Construct a log entry for reading.
     */
    public static <T extends IN> INLogEntry<T> create(final Class<T> INClass) {
        return new INLogEntry<T>(INClass);
    }

    INLogEntry(Class<T> INClass) {
        super(INClass);
    }

    /**
     * Construct a log entry for writing to the log.
     */
    public INLogEntry(T in) {
        this(in, false /*isBINDelta*/);
    }

    INLogEntry(T in, boolean isBINDelta) {
        setLogType(isBINDelta ? LogEntryType.LOG_BIN_DELTA : in.getLogType());
        this.in = in;
        this.dbId = in.getDatabase().getId();
        this.prevFullLsn =
            isBINDelta ? DbLsn.NULL_LSN : in.getLastFullVersion();
        this.prevDeltaLsn = in.getLastDeltaVersion();
    }

    /*
     * Read support
     */

    @Override
    public void readEntry(EnvironmentImpl envImpl,
                          LogEntryHeader header,
                          ByteBuffer entryBuffer) {

        int logVersion = header.getVersion();
        boolean version6OrLater = (logVersion >= 6);
        if (version6OrLater) {
            dbId = new DatabaseId();
            dbId.readFromLog(entryBuffer, logVersion);
            prevFullLsn = LogUtils.readLong(entryBuffer, false/*unpacked*/);
            if (logVersion >= 8) {
                prevDeltaLsn = LogUtils.readPackedLong(entryBuffer);
            }
        }
        /* Read IN. */
        in = newInstanceOfType();
        readMainItem(in, entryBuffer, logVersion);
        if (!version6OrLater) {
            dbId = new DatabaseId();
            dbId.readFromLog(entryBuffer, logVersion);
        }
        if (logVersion < 1) {
            prevFullLsn = DbLsn.NULL_LSN;
        } else if (logVersion == 1) {
            long fileNum = LogUtils.readUnsignedInt(entryBuffer);
            if (fileNum == 0xffffffffL) {
                prevFullLsn = DbLsn.NULL_LSN;
            } else {
                prevFullLsn = DbLsn.makeLsn(fileNum, 0);
            }
        } else if (!version6OrLater) {
            prevFullLsn = LogUtils.readLong(entryBuffer, true/*unpacked*/);
        }
    }

    @Override
    public long getPrevFullLsn() {
        return prevFullLsn;
    }

    @Override
    public long getPrevDeltaLsn() {
        return prevDeltaLsn;
    }

    @Override
    public StringBuilder dumpEntry(StringBuilder sb, boolean verbose) {
        in.dumpLog(sb, verbose);
        dbId.dumpLog(sb, verbose);
        if (prevFullLsn != DbLsn.NULL_LSN) {
            sb.append("<prevFullLsn>");
            sb.append(DbLsn.getNoFormatString(prevFullLsn));
            sb.append("</prevFullLsn>");
        }
        if (prevDeltaLsn != DbLsn.NULL_LSN) {
            sb.append("<prevDeltaLsn>");
            sb.append(DbLsn.getNoFormatString(prevDeltaLsn));
            sb.append("</prevDeltaLsn>");
        }
        return sb;
    }

    /** Never replicated. */
    public void dumpRep(@SuppressWarnings("unused") StringBuilder sb) {
    }

    @Override
    public T getMainItem() {
        return in;
    }

    void readMainItem(T in, ByteBuffer entryBuffer, int logVersion) {
        in.readFromLog(entryBuffer, logVersion);
    }

    int getMainItemSize(T in) {
        return in.getLogSize();
    }

    void writeMainItem(T in, ByteBuffer destBuffer) {
        in.writeToLog(destBuffer);
    }

    @Override
    public long getTransactionId() {
        return 0;
    }

    /*
     * Writing support
     */

    @Override
    public int getSize() {
        return (getMainItemSize(in) +
                dbId.getLogSize() +
                LogUtils.getPackedLongLogSize(prevFullLsn) +
                LogUtils.getPackedLongLogSize(prevDeltaLsn));
    }

    @Override
    public void writeEntry(ByteBuffer destBuffer) {
        dbId.writeToLog(destBuffer);
        LogUtils.writePackedLong(destBuffer, prevFullLsn);
        LogUtils.writePackedLong(destBuffer, prevDeltaLsn);
        writeMainItem(in, destBuffer);
    }

    @Override
    public IN getIN(@SuppressWarnings("unused") DatabaseImpl dbImpl) {
        return in;
    }

    public long getNodeId() {
        return in.getNodeId();
    }

    @Override
    public DatabaseId getDbId() {
        return dbId;
    }

    /**
     * INs from two different environments are never considered equal,
     * because they have lsns that are environment-specific.
     */
    @Override
    public boolean logicalEquals(@SuppressWarnings("unused") LogEntry other) {
        return false;
    }
}
