/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.tree;

import java.util.Collections;
import java.util.Iterator;
import java.util.Set;

import com.sleepycat.je.CacheMode;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.cleaner.LocalUtilizationTracker;
import com.sleepycat.je.dbi.CursorImpl;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.DbTree;
import com.sleepycat.je.dbi.EnvironmentFailureReason;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.dbi.MemoryBudget;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogManager;
import com.sleepycat.je.log.ReplicationContext;
import com.sleepycat.je.log.entry.BINDeltaLogEntry;
import com.sleepycat.je.log.entry.INLogEntry;
import com.sleepycat.je.txn.BasicLocker;
import com.sleepycat.je.txn.LockGrantType;
import com.sleepycat.je.txn.LockResult;
import com.sleepycat.je.txn.LockType;
import com.sleepycat.je.utilint.DatabaseUtil;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.RelatchRequiredException;
import com.sleepycat.je.utilint.SizeofMarker;
import com.sleepycat.je.utilint.TinyHashSet;
import com.sleepycat.je.utilint.VLSN;

/**
 * A BIN represents a Bottom Internal Node in the JE tree.
 *
 * BIN-deltas
 * ==========
 * A BIN-delta is a BIN with the non-dirty slots omitted. A "full BIN", OTOH
 * contains all slots.  On disk and in memory, the format of a BIN-delta is the
 * same as that of a BIN.  In memory, a BIN object is actually a BIN-delta when
 * the BIN-delta flag is set (IN.isBINDelta).  On disk, the NewBINDelta log
 * entry type (class BINDeltaLogEntry) is the only thing that distinguishes it
 * from a full BIN, which has the BIN log entry type.
 *
 * BIN-deltas provides two benefits: Reduced writing and reduced memory usage.
 *
 * Reduced Writing
 * ---------------
 * Logging a BIN-delta rather a full BIN reduces writing significantly.  The
 * cost, however, is that two reads are necessary to reconstruct a full BIN
 * from scratch.  The reduced writing is worth this cost, particularly because
 * less writing means less log cleaning.
 *
 * A BIN-delta is logged when 25% or less (configured with EnvironmentConfig
 * TREE_BIN_DELTA) of the slots in a BIN are dirty. When a BIN-delta is logged,
 * the dirty flag is cleared on the the BIN in cache.  If more slots are
 * dirtied and another BIN-delta is logged, it will contain all entries dirtied
 * since the last full BIN was logged.  In other words, BIN-deltas are
 * cumulative and not chained, to avoid reading many (more than two) log
 * entries to reconstruct a full BIN.  The dirty flag on each slot is cleared
 * only when a full BIN is logged.
 *
 * In addition to the cost of fetching two entries on a BIN cache miss, another
 * drawback of the current approach is that dirtiness propagates upward in the
 * Btree due to BIN-delta logging, causing repeated logging of upper INs.  The
 * slot of the parent IN contains the LSN of the most recent BIN-delta or full
 * BIN that was logged.  A BINDeltaLogEntry in turn contains the LSN of the
 * last full BIN logged.
 *
 *   Historical note:  The pre-JE 5 implementation of OldBINDeltas worked
 *   differently and had a different cost/benefit trade-off.  When an
 *   OldBINDelta was logged, its dirty flag was not cleared, causing it to be
 *   logged repeatedly at every checkpoint.  A full BIN was logged after 10
 *   deltas, to prevent endless logging of the same BIN.  One benefit of this
 *   approach is that the BIN's parent IN was not dirtied when logging the
 *   OldBINDelta, preventing dirtiness from propagating upward.  Another
 *   benefit is that the OldBINDelta was only processed by recovery, and did
 *   not have to be fetched to reconstruct a full BIN from scratch on a cache
 *   miss.  But the cost (the logging of an OldBINDelta every checkpoint, even
 *   when it hadn't changed since the last time logged) outweighed the
 *   benefits.  When the current approach was implemented in JE 5, performance
 *   improved due to less logging.
 *
 *   In JE 6, deltas were also maintained in the Btree cache.  This was done to
 *   provide the reduced memory benefits described in the next section.  The
 *   log format for a delta was also changed.  The OldBINDelta log format is
 *   different (not the same as the BIN format) and is supported for backward
 *   compatibility as the OldBINDeltaLogEntry.  Its log entry type name is
 *   still BINDelta, which is why the new type is named NewBINDelta (for
 *   backward compatibility, log entry type names cannot be changed.)  This is
 *   also why the spelling "BIN-delta" is used to refer to deltas in the new
 *   approach.  The old BINDelta class was renamed to OldBINDelta and there is
 *   no longer a class named BINDelta.
 *
 * Reduced Memory Usage
 * --------------------
 * In the Btree cache, a BIN may be represented as a full BIN or a BIN-delta.
 * Eviction will mutate a full BIN to a BIN-delta in preference to discarding
 * the entire BIN. A BIN-delta in cache occupies less memory than a full BIN,
 * and can be exploited as follows:
 *
 *  - When a full BIN is needed, it can be constructed with only one fetch
 *    rather than two, reducing IO overall.  IN.fetchTarget implements this
 *    optimization.
 *
 *  - Certain operations can sometimes be performed using the BIN-delta alone,
 *    allowing such operations on a given data set to take place using less
 *    less IO (for a given cache size).
 *
 * The latter benefit is not yet implemented.   No user CRUD operations are
 * currently implemented using BIN-deltas. In the future we plan to implement
 * the following operations using the BIN-delta alone.
 *
 *  - Consider recording deletions in a BIN-delta.  Currently, slot deletion
 *    prohibits a BIN-delta from being logged.  To record deletion in
 *    BIN-deltas, slot deletion will have to be deferred until a full BIN is
 *    logged.
 *
 *  - User reads by key, updates and deletions can be implemented if the key
 *    happens to appear in the BIN-delta.
 *
 *  - The Cleaner can migrate an LN if its key happens to appear in the
 *    BIN-delta.  This is similar to a user update operation, but in a
 *    different code path.
 *
 *  - Insertions, deletions and updates can always be performed in a BIN-delta
 *    during replica replay, since the Master operation has already determined
 *    whether the key exists.
 *
 *  - Recovery LN redo could also apply insertions, updates and inserts in the
 *    manner described.
 *
 *  - Add idempotent put/delete operations, which can always be applied in a
 *    BIN-delta.
 *
 *  - Store a hash of the keys in the full BIN in the BIN-delta and use it to
 *    perform the following in the delta:
 *    - putIfAbsent (true insertion)
 *    - get/delete/putIfPresent operations that return NOTFOUND
 *    - to avoid accumulating unnecessary deletions
 *
 * However, some internal operations do currently exploit BIN-deltas to avoid
 * unnecessary IO.  The following are currently implemented.
 *
 *  - The Evictor and Checkpointer log a BIN-delta that is present in the
 *    cache, without having to fetch the full BIN.
 *
 *  - The Cleaner can use the BIN-delta to avoid fetching when processing a BIN
 *    log entry (delta or full) and the BIN is not present in cache,
 *
 * To support the above internal operations, the IN.getTargetAllowBINDelta and
 * fetchTargetAllowBINDelta methods are used.  These may return the BIN as a
 * BIN-delta and the caller must therefore be "BIN-delta aware".
 *
 * Currently getTarget and fetchTarget never return a delta.  getTarget returns
 * null when a BIN-delta is present in cache, and fetchTarget will fetch the
 * full BIN if not already cached.  This means that callers of these methods
 * (e.g., CRUD operations) need not be BIN-delta aware.  This has a negative
 * side effect: The BIN's parent (rather than only the BIN) must be latched
 * when mutating a cached BIN from/to its BIN-delta representation.  The
 * Evictor therefore must currently search for the parent IN before mutating a
 * full BIN to a BIN-delta.
 *
 * In the future when CRUD operations are changed to exploit deltas, the
 * situation will be reversed: getTarget and fetchTarget will be changed to
 * return a BIN-delta, all callers will be made BIN-delta aware, the Evictor
 * will mutate to a BIN-delta while holding only the BIN latch, and we can
 * remove the getTargetAllowBINDelta and fetchTargetAllowBINDelta methods.  For
 * example, in the future a BIN-delta aware caller will do something like this:
 *  - Call getTarget or fetchTarget.
 *  - If the returned BIN is a delta, determine whether the operation can be
 *    performed with the delta alone; if so, perform it.
 *  - If the returned BIN is a delta and the operation requires a full BIN,
 *    mutateToFullBIN will be called before proceeding with the operation.
 */
public class BIN extends IN {

    private static final String BEGIN_TAG = "<bin>";
    private static final String END_TAG = "</bin>";

    /*
     * The set of cursors that are currently referring to this BIN.
     * This field is set to null when there are no cursors on this BIN.
     */
    private TinyHashSet<CursorImpl> cursorSet;

    /*
     * Support for logging BIN deltas. (Partial BIN logging)
     */

    /* Location of last delta, for cleaning. */
    private long lastDeltaVersion = DbLsn.NULL_LSN;
    private boolean prohibitNextDelta;  // disallow delta on next log

    /*
     * Caches the VLSN sequence for the LN entries in a BIN, when VLSN
     * preservation and caching are configured.
     *
     * A VLSN is added to the cache when an LN is evicted from a BIN.  When the
     * LN is resident, there is no need for caching because the LN contains the
     * VLSN. See BIN.setTarget.  This strategy works because an LN is always
     * cached during a read or write operation, and only evicted after that,
     * based on eviction policies.
     *
     * An EMPTY_REP is used initially until the need arises to add a non-zero
     * value.  The cache will remain empty if LNs are never evicted or version
     * caching is not configured, which is always the case for standalone JE.
     */
    private INLongRep vlsnCache = INLongRep.EMPTY_REP;

    /*
     * Stores the last logged size of each LN, or zero if the size is unknown.
     * We use INLongRep in spite of the fact that sizes are int not long;
     * INLongRep will store the minimum number of bytes.
     *
     * An EMPTY_REP is used initially until the need arises to add a non-zero
     * value.
     */
    private INLongRep lastLoggedSizes = INLongRep.EMPTY_REP;

    /**
     * Can be set to true by tests to prevent last logged sizes from being
     * stored.
     */
    public static boolean TEST_NO_LAST_LOGGED_SIZES = false;

    public BIN() {
    }

    public BIN(DatabaseImpl db,
               byte[] identifierKey,
               int maxEntriesPerNode,
               int level) {
        super(db, identifierKey, maxEntriesPerNode, level);
    }

    /**
     * For Sizeof.
     */
    public BIN(@SuppressWarnings("unused") SizeofMarker marker) {
        super(marker);
    }

    private void setCachedVLSN(int idx, long vlsn) {

        /*
         * We do not cache the VLSN for dup DBs, because dup DBs are typically
         * used only for indexes, and the overhead of VLSN maintenance would be
         * wasted.  Plus, although technically VLSN preservation might apply to
         * dup DBs, the VLSNs are not reliably available since the LNs are
         * immediately obsolete.
         */
        if (databaseImpl.getSortedDuplicates() || !getEnv().getCacheVLSN()) {
            return;
        }
        setCachedVLSNUnconditional(idx, vlsn);
    }

    private void setCachedVLSNUnconditional(int idx, long vlsn) {
        vlsnCache = vlsnCache.set(
            idx,
            (vlsn == VLSN.NULL_VLSN.getSequence()) ? 0 : vlsn,
            this, getEnv().getCachedVLSNMinLength());
    }

    private long getCachedVLSN(int idx) {
        final long vlsn = vlsnCache.get(idx);
        return (vlsn == 0) ? VLSN.NULL_VLSN.getSequence() : vlsn;
    }

    /**
     * Returns the VLSN.  VLSN.NULL_VLSN.getSequence() (-1) is returned in two
     * cases:
     * 1) This is a standalone environment.
     * 2) The VLSN is not cached (perhaps VLSN caching is not configured), and
     *    the allowFetch param is false.
     *
     * WARNING: Because the vlsnCache is only updated when an LN is evicted, it
     * is critical that getVLSN returns the VLSN for a resident LN before
     * getting the VLSN from the cache.
     */
    public long getVLSN(int idx, boolean allowFetch, CacheMode cacheMode) {
            
        /* Must return the VLSN from the LN, if it is resident. */
        LN ln = (LN) getTarget(idx);
        if (ln != null) {
            return ln.getVLSNSequence();
        }

        /* Next try the vlsnCache. */
        final long vlsn = getCachedVLSN(idx);
        if (!VLSN.isNull(vlsn)) {
            return vlsn;
        }

        /* As the last resort, fetch the LN if fetching is allowed. */
        if (!allowFetch) {
            return vlsn;
        }
        ln = (LN) fetchTarget(idx, cacheMode);
        return ln.getVLSNSequence();
    }

    /** For unit testing. */
    public INLongRep getVLSNCache() {
        return vlsnCache;
    }

    /**
     * The last logged size is never needed when the LN is counted obsolete
     * immediately, since it is only needed for counting an LN obsolete
     * during an update or deletion.
     *
     * This method may not be called until after the database is initialized,
     * i,e., it may not be called during readFromLog.
     */
    @Override
    boolean isLastLoggedSizeStored() {

        /* Check final static first so all test code is optimized away. */
        if (DatabaseUtil.TEST) {
            /* Don't skew test measurements with internal DBs. */
            if (TEST_NO_LAST_LOGGED_SIZES &&
                !databaseImpl.getDbType().isInternal()) {
                return false;
            }
        }

        return !databaseImpl.isLNImmediatelyObsolete();
    }

    /**
     * Sets last logged size if necessary.
     *
     * This method does not dirty the IN because the caller methods dirty it,
     * for example, when setting the LSN, key, or node.
     *
     * This method is sometimes called to add the logged size for a pre log
     * version 9 BIN, for example, during fetchTarget and preload.  This makes
     * the logged size available for obsolete counting but does not dirty the
     * IN, since that could cause an unexpected write of the IN being read.
     *
     * @param lastLoggedSize is positive if the size is known, zero if the size
     * is unknown, or -1 if the size should not be changed because logging of
     * the LN was deferred.
     */
    @Override
    public void setLastLoggedSize(int idx, int lastLoggedSize) {
        if ((lastLoggedSize < 0) || !isLastLoggedSizeStored()) {
            return;
        }
        setLastLoggedSizeUnconditional(idx, lastLoggedSize);
    }

    /**
     * Sets the size without checking whether it is necessary.
     *
     * This method is used when reading from the log because the databaseImpl
     * is not yet initialized and isLastLoggedSizeStored cannot be called.
     * It is also called for efficiency reasons when it is known that storing
     * the logged size is necessary, for example, when copying values between
     * slots.
     */
    void setLastLoggedSizeUnconditional(int idx, int lastLoggedSize) {
        /* minLength (last param) is 1 since log sizes are unpredictable. */
        lastLoggedSizes = lastLoggedSizes.set(idx, lastLoggedSize, this, 1);
    }

    /**
     * @return a positive value if the size is known, or zero if unknown.
     */
    @Override
    public int getLastLoggedSize(int idx) {
        return (int) lastLoggedSizes.get(idx);
    }

    /**
     * Updates the vlsnCache when an LN target is evicted.  See vlsnCache.
     */
    @Override
    void setTarget(int idx, Node target) {
        if (target == null) {
            final Node oldTarget = getTarget(idx);
            if (oldTarget instanceof LN) {
                setCachedVLSN(idx, ((LN) oldTarget).getVLSNSequence());
            }
        }
        super.setTarget(idx, target);
    }

    /**
     * Overridden to account for vlsnCache and lastLoggedSizes.
     */
    @Override
    void copyEntry(int idx, IN from, int fromIdx) {
        super.copyEntry(idx, from, fromIdx);
        setCachedVLSNUnconditional(idx, ((BIN) from).getCachedVLSN(fromIdx));
        setLastLoggedSizeUnconditional(idx, from.getLastLoggedSize(fromIdx));
    }

    /**
     * Overridden to account for vlsnCache and lastLoggedSizes.
     */
    @Override
    void copyEntries(int from, int to, int n) {
        super.copyEntries(from, to, n);
        vlsnCache = vlsnCache.copy(from, to, n);
        lastLoggedSizes = lastLoggedSizes.copy(from, to, n);
    }

    /**
     * Overridden to account for vlsnCache and lastLoggedSizes.
     */
    @Override
    void clearEntry(int idx) {
        super.clearEntry(idx);
        setCachedVLSNUnconditional(idx, VLSN.NULL_VLSN.getSequence());
        setLastLoggedSizeUnconditional(idx, 0);
    }

    /**
     * Overridden to account for vlsnCache and lastLoggedSizes.
     */
    @Override
    void resetContent(final IN other) {
        final BIN otherBin = (BIN) other;
        vlsnCache = otherBin.vlsnCache;
        lastLoggedSizes = otherBin.lastLoggedSizes;
        /* Call super method last because it recalculates the memory size. */
        super.resetContent(other);
    }

    /**
     * Calls super method and accounts for vlsnCache and lastLoggedSizes.
     */
    void resetContent(final int capacity,
                      final int newNEntries,
                      final long[] lsns,
                      final byte[] states,
                      final byte[][] keys,
                      final Node[] targets,
                      final long[] vlsns,
                      final int[] loggedSizes) {
        for (int i = 0; i < newNEntries; i += 1) {
            setCachedVLSNUnconditional(i, vlsns[i]);
            setLastLoggedSizeUnconditional(i, loggedSizes[i]);
        }
        /* Call super method last because it recalculates the memory size. */
        resetContent(capacity, newNEntries, lsns, states, keys, targets);
    }

    /**
     * Adds vlsnCache size to computed memory size.
     */
    @Override
    public long computeMemorySize() {

        /* 
         * These fields are null only when this method is called by the
         * superclass constructor, i.e., before this class constructor has
         * run.  Luckily the initial representations have a memory size of
         * zero, so we can ignore them in this case.
         */
        long size = super.computeMemorySize();
        if (vlsnCache != null) {
            size += vlsnCache.getMemorySize();
        }
        if (lastLoggedSizes != null) {
            size += lastLoggedSizes.getMemorySize();
        }
        return size;
    }

    /* Utility method used during unit testing. */
    @Override
    protected long printMemorySize() {
        final long inTotal = super.printMemorySize();
        final long vlsnCacheOverhead = vlsnCache.getMemorySize();
        final long logSizesOverhead = lastLoggedSizes.getMemorySize();
        final long binTotal = inTotal + vlsnCacheOverhead + logSizesOverhead;
        System.out.format(
            "BIN: %d vlsns: %d logSizes: %d %n",
            binTotal, vlsnCacheOverhead, logSizesOverhead);
        return binTotal;
    }

    /**
     * Create a holder object that encapsulates information about this BIN for
     * the INCompressor.
     */
    public BINReference createReference() {
      return new BINReference(getNodeId(), getDatabase().getId(),
                              getIdentifierKey());
    }

    /**
     * Create a new BIN.  Need this because we can't call newInstance()
     * without getting a 0 for nodeId.
     */
    @Override
    protected IN createNewInstance(byte[] identifierKey,
                                   int maxEntries,
                                   int level) {
        return new BIN(getDatabase(), identifierKey, maxEntries, level);
    }

    /*
     * Return whether the shared latch for this kind of node should be of the
     * "always exclusive" variety.  Presently, only IN's are actually latched
     * shared.  BINs are latched exclusive only.
     */
    @Override
    boolean isAlwaysLatchedExclusively() {
        return true;
    }

    @Override
    public boolean isBIN() {
        return true;
    }

    /**
     * Overrides the IN method to account for deltas.
     *
     * This method relies on setLastFullVersion being called first when a delta
     * is fetched, which is done by reconstituteBIN.
     */
    @Override
    public void setLastLoggedLsn(long lsn) {
        if (getLastFullVersion() == DbLsn.NULL_LSN) {
            setLastFullLsn(lsn);
        } else {
            lastDeltaVersion = lsn;
        }
    }

    /**
     * Overrides the IN method to account for deltas.
     */
    @Override
    public long getLastLoggedVersion() {
        return (lastDeltaVersion != DbLsn.NULL_LSN) ?
               lastDeltaVersion :
               getLastFullVersion();
    }

    /**
     * Overrides the IN method to account for deltas.
     * Public for unit testing.
     */
    @Override
    public long getLastDeltaVersion() {
        return lastDeltaVersion;
    }

    /**
     * If cleaned or compressed, must log full version.
     */
    @Override
    public void setProhibitNextDelta() {
        prohibitNextDelta = true;
    }

    /**
     * Note that the IN may or may not be latched when this method is called.
     * Returning the wrong answer is OK in that case (it will be called again
     * later when latched), but an exception should not occur.
     */
    @Override
    protected boolean isEvictionProhibited() {
        return (nCursors() > 0);
    }

    /**
     * Note that the IN may or may not be latched when this method is called.
     * Returning the wrong answer is OK in that case (it will be called again
     * later when latched), but an exception should not occur.
     */
    @Override
    boolean hasPinnedChildren() {

        DatabaseImpl db = getDatabase();

        /*
         * For the mapping DB, if any MapLN is resident we cannot evict this
         * BIN.  If a MapLN was not previously stripped, then the DB may be
         * open.  [#13415]
         */
        if (db.getId().equals(DbTree.ID_DB_ID)) {
            return hasResidentChildren();
        }

        /*
         * We can always evict this BIN because its children are limited to
         * LNs.  When logging the BIN, any dirty LNs will be logged and
         * non-dirty LNs can be discarded.
         */
        return false;
    }

    /**
     * Note that the IN may or may not be latched when this method is called.
     * Returning the wrong answer is OK in that case (it will be called again
     * later when latched), but an exception should not occur.
     */
    @Override
    int getChildEvictionType() {

        boolean nodeIsEvictable = true;

        for (int i = 0; i < getNEntries(); i++) {
            Node node = getTarget(i);
            if (node == null) {
                continue;
            }

            if (!(node instanceof LN)) {
                /*
                 * This is only possible during conversion of a pre-JE 5.0
                 * environment with duplicate DBs.  See DupConvert.
                 */
                nodeIsEvictable = false;
                continue;
            }

            LN ln = (LN) node;

            /*
             * isEvictableInexact is used here as a fast check, to avoid the
             * overhead of acquiring a handle lock while selecting an IN for
             * eviction.   See evictInternal which will call LN.isEvictable to
             * acquire an handle lock and guarantee that another thread cannot
             * open the MapLN.  [#13415]
             */
            if (!ln.isEvictableInexact()) {
                nodeIsEvictable = false;
                continue;
            }

            /* At least one LN is evictable. */
            return MAY_EVICT_LNS;
        }

        /* No LNs are evictable. */
        return nodeIsEvictable ? MAY_EVICT_NODE : MAY_NOT_EVICT;
    }

    /**
     * Indicates whether entry 0's key is "special" in that it always compares
     * less than any other key.  BIN's don't have the special key, but IN's do.
     */
    @Override
    boolean entryZeroKeyComparesLow() {
        return false;
    }

    /**
     * Mark this entry as deleted, using the delete flag. Only BINS may do
     * this.
     *
     * @param index indicates target entry
     */
    @Override
    public void setKnownDeleted(int index) {

        /*
         * The target is cleared to save memory, since a known deleted entry
         * will never be fetched.
         */
        super.setKnownDeleted(index);

        /*
         * We know it's an LN because we never call setKnownDeleted for
         * an IN.
         */
        LN oldLN = (LN) getTarget(index);
        updateMemorySize(oldLN, null /* newNode */);
        if (oldLN != null) {
            oldLN.releaseMemoryBudget();
        }
        setTarget(index, null);
        setDirty(true);
    }

    public void setKnownDeletedClearAll(int index) {
        setKnownDeleted(index);
        setLsnElement(index, DbLsn.NULL_LSN);
    }

    /**
     * Clear the known deleted flag. Only BINS may do this.
     * @param index indicates target entry
     */
    @Override
    public void clearKnownDeleted(int index) {
        super.clearKnownDeleted(index);
        setDirty(true);
    }

    @Override
    protected long getFixedMemoryOverhead() {
        return MemoryBudget.BIN_FIXED_OVERHEAD;
    }

    /**
     * Returns the treeAdmin memory in objects referenced by this BIN.
     * Specifically, this refers to the DbFileSummaryMap held by
     * MapLNs
     */
    @Override
    public long getTreeAdminMemorySize() {

        if (getDatabase().getId().equals(DbTree.ID_DB_ID)) {
            long treeAdminMem = 0;
            for (int i = 0; i < getMaxEntries(); i++) {
                Node n = getTarget(i);
                if (n != null) {
                    MapLN mapLN = (MapLN) n;
                    treeAdminMem += mapLN.getDatabase().getTreeAdminMemory();
                }
            }
            return treeAdminMem;
        } else {
            return 0;
        }
    }

    /*
     * Cursors
     */

    /* public for the test suite. */
    public Set<CursorImpl> getCursorSet() {
       if (cursorSet == null) {
           return Collections.emptySet();
       }
       return cursorSet.copy();
    }

    /**
     * Register a cursor with this BIN.  Caller has this BIN already latched.
     * @param cursor Cursor to register.
     */
    public void addCursor(CursorImpl cursor) {
        assert isLatchExclusiveOwner();
        if (cursorSet == null) {
            cursorSet = new TinyHashSet<CursorImpl>();
        }
        cursorSet.add(cursor);
    }

    /**
     * Unregister a cursor with this bin.  Caller has this BIN already
     * latched.
     *
     * @param cursor Cursor to unregister.
     */
    public void removeCursor(CursorImpl cursor) {
        assert isLatchExclusiveOwner();
        if (cursorSet == null) {
            return;
        }
        cursorSet.remove(cursor);
        if (cursorSet.size() == 0) {
            cursorSet = null;
        }
    }

    /**
     * @return the number of cursors currently referring to this BIN.
     */
    public int nCursors() {

        /*
         * Use a local var to concurrent assignment to the cursorSet field by
         * another thread. This method is called via eviction without latching.
         * LRU-TODO: with the new evictor this method is called with the node
         * EX-latched. So, cleanup after the old evictor is scrapped. 
         */
        final TinyHashSet<CursorImpl> cursors = cursorSet;
        if (cursors == null) {
            return 0;
        }
        return cursors.size();
    }

    /**
     * Called when we know we are about to split on behalf of a key that is the
     * minimum (leftSide) or maximum (!leftSide) of this node.  This is
     * achieved by just forcing the split to occur either one element in from
     * the left or the right (i.e. splitIndex is 1 or nEntries - 1).
     */
    @Override
    void splitSpecial(IN parent,
                      int parentIndex,
                      int maxEntriesPerNode,
                      byte[] key,
                      boolean leftSide,
                      CacheMode cacheMode)
        throws DatabaseException {

        int index = findEntry(key, true, false);
        int nEntries = getNEntries();
        boolean exact = (index & IN.EXACT_MATCH) != 0;
        index &= ~IN.EXACT_MATCH;
        if (leftSide &&
            index < 0) {
            splitInternal(parent, parentIndex, maxEntriesPerNode,
                          1, cacheMode);
        } else if (!leftSide &&
                   !exact &&
                   index == (nEntries - 1)) {
            splitInternal(parent, parentIndex, maxEntriesPerNode,
                          nEntries - 1, cacheMode);
        } else {
            split(parent, parentIndex, maxEntriesPerNode, cacheMode);
        }
    }

    /**
     * Adjust any cursors that are referring to this BIN.  This method is
     * called during a split operation.  "this" is the BIN being split.
     * newSibling is the new BIN into which the entries from "this" between
     * newSiblingLow and newSiblingHigh have been copied.
     *
     * @param newSibling - the newSibling into which "this" has been split.
     * @param newSiblingLow
     * @param newSiblingHigh - the low and high entry of
     * "this" that were moved into newSibling.
     */
    @Override
    void adjustCursors(IN newSibling,
                       int newSiblingLow,
                       int newSiblingHigh) {
        assert newSibling.isLatchExclusiveOwner();
        assert this.isLatchExclusiveOwner();
        if (cursorSet == null) {
            return;
        }
        int adjustmentDelta = (newSiblingHigh - newSiblingLow);
        Iterator<CursorImpl> iter = cursorSet.iterator();
        while (iter.hasNext()) {
            CursorImpl cursor = iter.next();
            if (cursor.getBINToBeRemoved() == this) {

                /*
                 * This BIN will be removed from the cursor by CursorImpl
                 * following advance to next BIN; ignore it.
                 */
                continue;
            }
            int cIdx = cursor.getIndex();
            BIN cBin = cursor.getBIN();
            assert cBin == this :
                "nodeId=" + getNodeId() +
                " cursor=" + cursor.dumpToString(true);
            assert newSibling instanceof BIN;

            /*
             * There are four cases to consider for cursor adjustments,
             * depending on (1) how the existing node gets split, and (2) where
             * the cursor points to currently.  In cases 1 and 2, the id key of
             * the node being split is to the right of the splitindex so the
             * new sibling gets the node entries to the left of that index.
             * This is indicated by "new sibling" to the left of the vertical
             * split line below.  The right side of the node contains entries
             * that will remain in the existing node (although they've been
             * shifted to the left).  The vertical bar (^) indicates where the
             * cursor currently points.
             *
             * case 1:
             *
             *   We need to set the cursor's "bin" reference to point at the
             *   new sibling, but we don't need to adjust its index since that
             *   continues to be correct post-split.
             *
             *   +=======================================+
             *   |  new sibling        |  existing node  |
             *   +=======================================+
             *         cursor ^
             *
             * case 2:
             *
             *   We only need to adjust the cursor's index since it continues
             *   to point to the current BIN post-split.
             *
             *   +=======================================+
             *   |  new sibling        |  existing node  |
             *   +=======================================+
             *                              cursor ^
             *
             * case 3:
             *
             *   Do nothing.  The cursor continues to point at the correct BIN
             *   and index.
             *
             *   +=======================================+
             *   |  existing Node        |  new sibling  |
             *   +=======================================+
             *         cursor ^
             *
             * case 4:
             *
             *   Adjust the "bin" pointer to point at the new sibling BIN and
             *   also adjust the index.
             *
             *   +=======================================+
             *   |  existing Node        |  new sibling  |
             *   +=======================================+
             *                                 cursor ^
             */
            BIN ns = (BIN) newSibling;
            if (newSiblingLow == 0) {
                if (cIdx < newSiblingHigh) {
                    /* case 1 */
                    iter.remove();
                    cursor.setBIN(ns);
                    ns.addCursor(cursor);
                } else {
                    /* case 2 */
                    cursor.setIndex(cIdx - adjustmentDelta);
                }
            } else {
                if (cIdx >= newSiblingLow) {
                    /* case 4 */
                    cursor.setIndex(cIdx - newSiblingLow);
                    iter.remove();
                    cursor.setBIN(ns);
                    ns.addCursor(cursor);
                }
            }
        }
    }

    /**
     * For each cursor in this BIN's cursor set, ensure that the cursor is
     * actually referring to this BIN.
     */
    public void verifyCursors() {
        if (cursorSet == null) {
            return;
        }
        for (CursorImpl cursor : cursorSet) {
            if (cursor.getBINToBeRemoved() != this) {
                BIN cBin = cursor.getBIN();
                assert cBin == this;
            }
        }
    }

    /**
     * Adjust cursors referring to this BIN following an insert.
     *
     * @param insertIndex - The index of the new entry.
     */
    @Override
    void adjustCursorsForInsert(int insertIndex) {
        assert this.isLatchExclusiveOwner();
        if (cursorSet == null) {
            return;
        }
        for (CursorImpl cursor : cursorSet) {
            if (cursor.getBINToBeRemoved() != this) {
                int cIdx = cursor.getIndex();
                if (insertIndex <= cIdx) {
                    cursor.setIndex(cIdx + 1);
                }
            }
        }
    }

    /**
     * Compress this BIN by removing any entries that are deleted.  No cursors
     * may be present on the BIN.  Caller is responsible for latching and
     * unlatching this node.
     *
     * @param localTracker is used only for temporary DBs, and may be specified
     * to consolidate multiple tracking operations.  If null, the tracking is
     * performed immediately in this method.
     *
     * @return true if all deleted slots were compressed, or false if one or
     * more slots could not be compressed because we were unable to obtain a
     * lock.
     */
    public boolean compress(LocalUtilizationTracker localTracker)
        throws DatabaseException {

        /* 
         * If the environment is not yet recovered we can't rely on locks
         * being set up to safeguard active data and so we can't compress
         * safely.
         */
        if (!databaseImpl.getDbEnvironment().isValid()) {
            return false;
        }
        
        if (nCursors() > 0) {
            throw EnvironmentFailureException.unexpectedState();
        }

        boolean setNewIdKey = false;
        boolean anyLocksDenied = false;
        final DatabaseImpl db = getDatabase();
        final EnvironmentImpl envImpl = db.getDbEnvironment();

        for (int i = 0; i < getNEntries(); i++) {

            /* KD and PD determine deletedness. */
            if (!isEntryPendingDeleted(i) && !isEntryKnownDeleted(i)) {
                continue;
            }

            /*
             * We have to be able to lock the LN before we can compress the
             * entry. If we can't, then skip over it.
             *
             * We must lock the LN even if isKnownDeleted is true, because
             * locks protect the aborts. (Aborts may execute multiple
             * operations, where each operation latches and unlatches. It's the
             * LN lock that protects the integrity of the whole multi-step
             * process.)
             *
             * For example, during abort, there may be cases where we have
             * deleted and then added an LN during the same txn.  This means
             * that to undo/abort it, we first delete the LN (leaving
             * knownDeleted set), and then add it back into the tree.  We want
             * to make sure the entry is in the BIN when we do the insert back
             * in.
             */
            final BasicLocker lockingTxn =
                BasicLocker.createBasicLocker(envImpl);
            /* Don't allow this short-lived lock to be preempted/stolen. */
            lockingTxn.setPreemptable(false);
            try {
                /* Lock LSN. Can discard a NULL_LSN entry without locking. */
                final long lsn = getLsn(i);
                if (lsn != DbLsn.NULL_LSN) {
                    final LockResult lockRet = lockingTxn.nonBlockingLock
                        (lsn, LockType.READ, false /*jumpAheadOfWaiters*/, db);
                    if (lockRet.getLockGrant() == LockGrantType.DENIED) {
                        anyLocksDenied = true;
                        continue;
                    }
                }

                /* At this point, we know we can delete. */
                if (Key.compareKeys(getKey(i), getIdentifierKey(),
                                    getKeyComparator()) == 0) {

                    /*
                     * We're about to remove the entry with the idKey so the
                     * node will need a new idkey.
                     */
                    setNewIdKey = true;
                }

                if (db.isDeferredWriteMode()) {
                    final LN ln = (LN) getTarget(i);
                    if (ln != null &&
                        ln.isDirty() &&
                        !DbLsn.isTransient(lsn)) {
                        if (db.isTemporary()) {

                            /*
                             * When a previously logged LN in a temporary DB is
                             * dirty, we can count the LSN of the last logged
                             * LN as obsolete without logging.  There is no
                             * requirement for the dirty deleted LN to be
                             * durable past recovery.  There is no danger of
                             * the last logged LN being accessed again (after
                             * log cleaning, for example), since temporary DBs
                             * do not survive recovery.
                             */
                            if (localTracker != null) {
                                localTracker.countObsoleteNode
                                    (lsn, ln.getGenericLogType(),
                                     getLastLoggedSize(i), db);
                            } else {
                                envImpl.getLogManager().countObsoleteNode
                                    (lsn, ln.getGenericLogType(),
                                     getLastLoggedSize(i), db,
                                     true /*countExact*/);
                            }
                        } else {

                            /*
                             * When a previously logged deferred-write LN is
                             * dirty, we log the dirty deleted LN to make the
                             * deletion durable.  The act of logging will also
                             * count the last logged LSN as obsolete.
                             */
                            logDirtyLN(i, ln, false /*ensureDurableLsn*/,
                                       true /*allowEviction*/);
                        }
                    }
                }

                boolean deleteSuccess = deleteEntry(i, true);
                assert deleteSuccess;

                /*
                 * Since we're deleting the current entry, bump the current
                 * index back down one.
                 */
                i--;
            } finally {
                lockingTxn.operationEnd();
            }
        }

        if (getNEntries() != 0 && setNewIdKey) {
            setIdentifierKey(getKey(0));
        }

        /* This BIN is empty and expendable. */
        if (getNEntries() == 0) {
            setGeneration(CacheMode.MAKE_COLD);
        }

        return !anyLocksDenied;
    }

    /**
     * This method is called whenever a deleted slot is observed (when the
     * slot's PendingDeleted or KnownDeleted flag is set), to ensure that the
     * slot is compressed away.  This is an attempt to process slots that were
     * not compressed during the mainstream record deletion process because of
     * cursors on the BIN during compress, or a crash prior to compression.
     */
    public void queueSlotDeletion() {

        /*
         * If we will log a delta (which includes the case where no slots are
         * dirty), set the BIN dirty to ensure we compress it later in the
         * beforeLog method.
         */
        if (shouldLogDelta()) {
            setDirty(true);
            return;
        }

        /* If we will next log a full version, add to the queue. */
        final EnvironmentImpl envImpl = getDatabase().getDbEnvironment();
        envImpl.addToCompressorQueue(this, false);
    }

    @Override
    public boolean isCompressible() {
        return true;
    }

    /**
     * Reduce memory consumption by evicting all LN targets. Note that this may
     * cause LNs to be logged, which will mark this BIN dirty.
     *
     * The BIN should be latched by the caller.
     *
     * @return a long number encoding (a) the number of evicted bytes, and
     * (b) whether this BIN  is evictable. (b) will be false if the BIN has
     * any cursors on it, or has any non-evictable children.
     */
    @Override
    public long partialEviction() {
        return evictLNs();
        //TODO: call discard vlsnCache as well
    }

    /**
     * Reduce memory consumption by evicting all LN targets. Note that this may
     * cause LNs to be logged, which will mark this BIN dirty.
     *
     * The BIN should be latched by the caller.
     *
     * @return a long number encoding (a) the number of evicted bytes, and
     * (b) whether this BIN  is evictable. (b) will be false if the BIN has
     * any cursors on it, or has any non-evictable children.
     */
    public long evictLNs()
        throws DatabaseException {

        assert isLatchExclusiveOwner() :
            "BIN must be latched before evicting LNs";

        /*
         * We can't evict an LN which is pointed to by a cursor, in case that
         * cursor has a reference to the LN object. We'll take the cheap choice
         * and avoid evicting any LNs if there are cursors on this BIN. We
         * could do a more expensive, precise check to see entries have which
         * cursors. This is something we might move to later.
         */
        if (nCursors() > 0) {
            return IN.NON_EVICTABLE_IN;
        }

        /* Try to evict each child LN. */
        long totalRemoved = 0;
        boolean haveNonEvictableLN = false;

        for (int i = 0; i < getNEntries(); i++) {
            long lnRemoved = evictInternal(i);
            if (lnRemoved < 0) {
                haveNonEvictableLN = true;
            } else {
                totalRemoved += lnRemoved;
            }
        }

        /*
         * compactMemory() may decrease the memory footprint by mutating the
         * representations of the target and key sets.
         */
        if (totalRemoved > 0) {
            updateMemorySize(totalRemoved, 0);
            compactMemory();
        }

        if (haveNonEvictableLN) {
            return (totalRemoved | IN.NON_EVICTABLE_IN);
        } else {
            return totalRemoved;
        }
    }

    /**
     * Evict a single LN if allowed and adjust the memory budget.
     */
    public void evictLN(int index)
        throws DatabaseException {

        final long removed = evictInternal(index);

        /* May decrease the memory footprint by changing the INTargetRep. */
        if (removed > 0) {
            updateMemorySize(removed, 0);
            compactMemory();
        }
    }

    /**
     * Evict a single LN if allowed. The amount of memory freed is returned
     * and must be subtracted from the memory budget by the caller.
     *
     * @return number of evicted bytes or -1 if the LN is not evictable.
     */
    private long evictInternal(int index)
        throws DatabaseException {

        final Node n = getTarget(index);

        assert(n == null || n instanceof LN);

        if (n == null) {
            return 0;
        }

        final LN ln = (LN) n;
        final long lsn = getLsn(index);

        /*
         * Don't evict MapLNs for open databases (LN.isEvictable) [#13415].
         */
        if (ln.isEvictable(lsn)) {

            /*
             * Log target if necessary. Do not allow eviction since we evict
             * here and that would cause double-counting of the memory freed.
             */
            logDirtyLN(index, ln, true /*ensureDurableLsn*/,
                       false /*allowEviction*/);
            
            /* Clear target. */
            setTarget(index, null);
            ln.releaseMemoryBudget();

            return n.getMemorySizeIncludedByParent();
        }

        return -1;
    }

    /**
     * Logs the LN at the given index if it is dirty.
     */
    private void logDirtyLN(int index,
                            LN ln,
                            boolean ensureDurableLsn,
                            boolean allowEviction)
        throws DatabaseException {

        final long oldLsn = getLsn(index);
        final boolean force = ensureDurableLsn &&
                              getDatabase().isDeferredWriteMode() &&
                              DbLsn.isTransientOrNull(oldLsn);
        if (force || ln.isDirty()) {
            final DatabaseImpl dbImpl = getDatabase();
            final EnvironmentImpl envImpl = dbImpl.getDbEnvironment();

            /* Only deferred write databases should have dirty LNs. */
            assert dbImpl.isDeferredWriteMode();

            /*
             * Do not lock while logging.  Locking of new LSN is performed by
             * lockAfterLsnChange. This should never be part of the replication
             * stream, because this is a deferred-write DB.
             */
            final LN.LogResult logResult = ln.log(
                envImpl, dbImpl, getKey(index), oldLsn,
                getLastLoggedSize(index), null /*locker*/,
                null /*writeLockInfo*/, true /*backgroundIO*/,
                ReplicationContext.NO_REPLICATE);
            updateEntry(index, logResult.newLsn, logResult.newSize);
            /* Lock new LSN on behalf of existing lockers. */
            CursorImpl.lockAfterLsnChange(
                dbImpl, oldLsn, logResult.newLsn, null /*excludeLocker*/);

            /*
             * It is desirable to evict a non-dirty LN in a duplicates DB
             * because it will never be fetched again.
             */
            if (allowEviction && databaseImpl.getSortedDuplicates()) {
                evictLN(index);
            }
        }
    }

    /* For debugging.  Overrides method in IN. */
    @Override
    boolean validateSubtreeBeforeDelete(int index) {
        return true;
    }

    /**
     * Check if this node fits the qualifications for being part of a deletable
     * subtree. It may not have any LN children.
     *
     * We assume that this is only called under an assert.
     */
    @Override
    boolean isValidForDelete()
        throws DatabaseException {

        int numValidEntries = 0;
        boolean needToLatch = !isLatchExclusiveOwner();
        try {
            if (needToLatch) {
                latch();
            }
            for (int i = 0; i < getNEntries(); i++) {
                if (!isEntryKnownDeleted(i)) {
                    numValidEntries++;
                }
            }

            if (numValidEntries > 0) { // any valid entries, not eligible
                return false;
            }
            if (nCursors() > 0) {      // cursors on BIN, not eligible
                return false;
            }
            return true;               // 0 entries, no cursors
        } finally {
            if (needToLatch &&
                isLatchExclusiveOwner()) {
                releaseLatch();
            }
        }
    }

    /*
     * DbStat support.
     */
    @Override
    void accumulateStats(TreeWalkerStatsAccumulator acc) {
        acc.processBIN(this, Long.valueOf(getNodeId()), getLevel());
    }

    @Override
    public String beginTag() {
        return BEGIN_TAG;
    }

    @Override
    public String endTag() {
        return END_TAG;
    }

    /*
     * Logging support
     */

    /**
     * @see IN#logDirtyChildren
     */
    @Override
    public void logDirtyChildren()
        throws DatabaseException {

        /* Look for targets that are dirty. */
        EnvironmentImpl envImpl = getDatabase().getDbEnvironment();
        for (int i = 0; i < getNEntries(); i++) {
            Node node = getTarget(i);
            if (node != null) {
                logDirtyLN(i, (LN) node, true /*ensureDurableLsn*/,
                           true /*allowEviction*/);
            }
        }
    }

    /**
     * @see Node#incFetchStats
     */
    @Override
    public void incFetchStats(EnvironmentImpl envImpl, boolean isMiss) {
        envImpl.getEvictor().incBINFetchStats(isMiss, isBINDelta());
    }

    /**
     * @see IN#getLogType
     */
    @Override
    public LogEntryType getLogType() {
        return LogEntryType.LOG_BIN;
    }

    @Override
    public String shortClassName() {
        return "BIN";
    }

    /**
     * Overrides the IN method to account for deltas.
     */
    @Override
    public void beforeLog(LogManager logManager,
                          INLogItem item,
                          INLogContext context) {

        final DatabaseImpl dbImpl = getDatabase();
        final EnvironmentImpl envImpl = dbImpl.getDbEnvironment();

        /* Determine whether we log a delta rather than full version. */
        item.isDelta =
            isBINDelta() || (context.allowDeltas && shouldLogDelta());

        /* Be sure that we didn't illegally mutate to a delta. */
        assert (!(item.isDelta && isDeltaProhibited()));

        /* Perform lazy compression when logging a full BIN. */
        if (context.allowCompress && !item.isDelta) {
            envImpl.lazyCompress(this);
        }

        /*
         * Write dirty LNs in deferred-write databases.  This is done after
         * compression to reduce total logging, at least for temp DBs.
         */
        if (dbImpl.isDeferredWriteMode()) {
            logDirtyLNs();
        }

        /*
         * In the Btree, the parent IN slot contains the latest full version
         * LSN or, if a delta was last logged, the delta LSN.  Somewhat
         * redundantly, the transient IN.lastFullVersion and
         * BIN.lastDeltaVersion fields contain the last logged full version and
         * delta version LSNs.
         *
         * For delta logging:
         *  + Count lastDeltaVersion obsolete, if non-null.
         *  + Set lastDeltaVersion to newly logged LSN.
         *  + Leave lastFullVersion unchanged.
         *
         * For full version logging:
         *  + Count lastFullVersion and lastDeltaVersion obsolete, if non-null.
         *  + Set lastFullVersion to newly logged LSN.
         *  + Set lastDeltaVersion to null.
         */
        beforeLogCommon(
            item, context,
            item.isDelta ? DbLsn.NULL_LSN : getLastFullVersion(),
            lastDeltaVersion);
        item.entry = item.isDelta ?
            (new BINDeltaLogEntry(this)) :
            (new INLogEntry<BIN>(this));
    }

    /**
     * Overrides the IN method to account for deltas.  See beforeLog.
     */
    @Override
    public void afterLog(LogManager logManager,
                         INLogItem item,
                         INLogContext context) {

        afterLogCommon(logManager, item, context,
                       item.isDelta ? DbLsn.NULL_LSN : getLastFullVersion(),
                       lastDeltaVersion);

        if (item.isDelta) {
            lastDeltaVersion = item.newLsn;
        } else {
            setLastFullLsn(item.newLsn);
            lastDeltaVersion = DbLsn.NULL_LSN;

            /*
             * Before logging a full version BIN we attempted to compress it.
             * If we could not compress a slot because of the presence of
             * cursors, we must re-queue (or at least re-dirty) the BIN so
             * that we will compress it later.  The BIN is set non-dirty by
             * afterLogCommon above.
             */
            for (int i = 0; i < getNEntries(); i += 1) {
                if (isEntryKnownDeleted(i) || isEntryPendingDeleted(i)) {
                    queueSlotDeletion();
                    break;
                }
            }
        }

        prohibitNextDelta = false;
    }

    private void logDirtyLNs()
        throws DatabaseException {

        for (int i = 0; i < getNEntries(); i++) {
            final Node node = getTarget(i);
            if ((node != null) && (node instanceof LN)) {
                logDirtyLN(i, (LN) node, true /*ensureDurableLsn*/,
                           true /*allowEviction*/);
            }
        }
    }

    private boolean isDeltaProhibited() {
        final DatabaseImpl dbImpl = getDatabase();
        return prohibitNextDelta ||
            dbImpl.isDeferredWriteMode() ||
            (getLastFullVersion() == DbLsn.NULL_LSN);
    }

    /**
     * Decide whether to log a full or partial BIN, depending on the ratio of
     * the delta size to full BIN size, and the number of deltas that have been
     * logged since the last full.
     *
     * Other factors are taken into account:
     * + a delta cannot be logged if the BIN has never been logged before
     * + deltas are not currently supported for DeferredWrite databases
     * + this particular delta may have been prohibited because the cleaner is
     *   migrating the BIN or a slot has been deleted
     * + if there are no dirty slots, we might as well log a full BIN
     *
     * @return true if we should log the deltas of this BIN
     */
    public boolean shouldLogDelta() {

        /* Cheapest checks first. */
        if (isDeltaProhibited()) {
            return false;
        }

        /* Must count deltas to check further. */
        final int numDeltas = getNDeltas();
        
        /* A delta with zero items is not valid. */
        if (numDeltas <= 0) {
            return false;
        }

        /* Check the configured BinDeltaPercent. */
        final int deltaLimit =
            (getNEntries() * databaseImpl.getBinDeltaPercent()) / 100;
        if (numDeltas > deltaLimit) {
            return false;
        }

        return true;
    }

    /**
     * We require exclusive latches on a BIN, so RelatchRequiredException is
     * never thrown.
     */
    @Override
    public Node fetchTarget(int idx, CacheMode cacheMode) {
        return fetchTargetWithExLatch(idx, cacheMode);
    }

    /**
     * Must be called only when the LSN being fetched is known to be active
     * (non-obsolete) i.e., will always be present in the log even when the LN
     * is "immediately obsolete".  For example, this is true when the LN is
     * part of an active txn during partial rollback, and when DupConvert reads
     * a log written prior to the "immediately obsolete" implementation.
     *
     * We require exclusive latches on a BIN, so RelatchRequiredException is
     * never thrown.
     */
    public Node fetchTargetKnownActive(int idx, CacheMode cacheMode) {
        try {
            return fetchTargetInternal(
                idx, cacheMode, false /*allowDelta*/, true /*knownActive*/);
        } catch (RelatchRequiredException e) {
            throw EnvironmentFailureException.unexpectedException(e);
        }
    }

    /**
     * Release the latch on this node if it is owned exclusively.
     *
     * This method is only available on BINs, not INs, because it is not
     * possible to detect whether a shared latch is held (BIN latches are
     * always exclusive).
     */
    public void releaseLatchIfOwner() {
        latch.releaseIfOwner();
    }

    /**
     * Returns whether mutateToBINDelta can be called.
     *
     * This method must be called with both this node and its parent node
     * latched exclusively.
     */
    public boolean canMutateToBINDelta() {
        return !isBINDelta() &&
            shouldLogDelta() &&
            (nCursors() == 0);
    }

    /**
     * Mutate to a delta (discard non-dirty entries and resize arrays).
     *
     * This method must be called with both this node and its parent node
     * latched exclusively, and canMutateToBINDelta must return true.
     *
     * @return the number of bytes freed.
     */
    public long mutateToBINDelta() {
        assert isLatchExclusiveOwner();
        assert canMutateToBINDelta();
        if (getInListResident()) {
            databaseImpl.getEnvironmentImpl().
                         getInMemoryINs().
                         updateBINDeltaStat(1);
        }
        final long oldSize = getInMemorySize();
        final int nDeltas = getNDeltas();

        /*
         * For now the capacity is the same as the number of deltas, to use
         * the minimum amount of memory.  In the future when deltas are used to
         * service operations, the capacity should be:
         *   (getMaxEntries() * databaseImpl.getBinDeltaPercent()) / 100;
         */
        initBINDelta(this, nDeltas, nDeltas);
        return oldSize - getInMemorySize();
    }

    /**
     * Replaces the contents of destBIN with the deltas in this BIN.
     */
    private void initBINDelta(final BIN destBIN,
                              final int nDeltas,
                              final int capacity) {
        final long[] lsns = new long[nDeltas];
        final long[] vlsns = new long[nDeltas];
        final int[] sizes= new int[nDeltas];
        final byte[][] keys = new byte[nDeltas][];
        final byte[] states = new byte[nDeltas];
        final Node[] targets = new Node[nDeltas];
        int j = 0;
        for (int i = 0; i < getNEntries(); i += 1) {
            if (!isDirty(i)) {
                continue;
            }
            lsns[j] = getLsn(i);
            vlsns[j] = getCachedVLSN(i);
            sizes[j] = getLastLoggedSize(i);
            keys[j] = getKey(i);
            states[j] = getState(i);
            targets[j] = getTarget(i);
            j += 1;
        }
        destBIN.resetContent(
            capacity, nDeltas, lsns, states, keys, targets, vlsns, sizes);
        destBIN.setBINDelta(true);
        destBIN.compactMemory();
    }

    /**
     * Creates a BIN delta using the deltas in this BIN.  The new BIN is
     * created as if this BIN were cloned and non-dirty slots were removed.
     * It is not added to the INList.
     */
    public BIN createBINDelta() {
        final BIN bin =
            new BIN(databaseImpl, getIdentifierKey(), 0, getLevel());
        final int nDeltas = getNDeltas();
        initBINDelta(bin, nDeltas, nDeltas);
        return bin;
    }

    /**
     * Fetch the full BIN and apply the deltas in this BIN to it, then use the
     * merged result to replace the contents of this BIN.
     *
     * This method must be called with both this node and its parent node
     * latched exclusively.  'this' must be a delta.
     */
    void mutateToFullBIN() {
        final BIN fullBIN = fetchFullBIN(databaseImpl);
        databaseImpl.getEnvironmentImpl().getEvictor().
            incFullBINFetchMissStats();
        mutateToFullBIN(fullBIN);
    }

    /**
     * Mutates this delta to a full BIN by applying this delta to the fullBIN
     * param and then replacing this BIN's contents with it.
     *
     * This method must be called with both this node and its parent node
     * latched exclusively.  'this' must be a delta.
     */
    public void mutateToFullBIN(BIN fullBIN) {
        assert isLatchExclusiveOwner();
        assert isBINDelta() : this;
        reconstituteBIN(databaseImpl, fullBIN);
        resetContent(fullBIN);
        setBINDelta(false);
        compactMemory();
        if (getInListResident()) {
            databaseImpl.getEnvironmentImpl().
                getInMemoryINs().
                updateBINDeltaStat(-1);
        }
    }

    /**
     * Create a BIN by fetching the full version and applying the deltas in
     * this BIN to it.  The BIN is not added to the INList.
     *
     * @return the full BIN with deltas applied.
     */
    public BIN reconstituteBIN(DatabaseImpl dbImpl) {
        final BIN fullBIN = fetchFullBIN(dbImpl);
        reconstituteBIN(dbImpl, fullBIN);
        return fullBIN;
    }

    private BIN fetchFullBIN(DatabaseImpl dbImpl) {
        final EnvironmentImpl envImpl = dbImpl.getDbEnvironment();
        final long lsn = getLastFullVersion();
        try {
            return (BIN) envImpl.getLogManager().
                getEntryHandleFileNotFound(lsn);
        } catch (EnvironmentFailureException e) {
            e.addErrorMessage(makeFetchErrorMsg(null, this, lsn, (byte) 0));
            throw e;
        } catch (RuntimeException e) {
            throw new EnvironmentFailureException(
                envImpl, EnvironmentFailureReason.LOG_INTEGRITY,
                makeFetchErrorMsg(e.toString(), this, lsn, (byte) 0), e);
        }
    }

    /**
     * Given a full version BIN, apply the deltas in this BIN. The fullBIN will
     * then be complete, but its memory will not be compacted.
     */
    public void reconstituteBIN(DatabaseImpl dbImpl, BIN fullBIN) {
        fullBIN.setDatabase(dbImpl);
        fullBIN.latch(CacheMode.UNCHANGED);
        try {

            /*
             * The BIN's lastFullLsn is set here, while its lastLoggedLsn is
             * set by postFetchInit or postRecoveryInit.
             */
            fullBIN.setLastFullLsn(getLastFullVersion());

            /* Process each delta. */
            for (int i = 0; i < getNEntries(); i++) {
                assert isDirty(i) : this;
                fullBIN.applyDelta(
                    getKey(i), getLsn(i), getState(i), getLastLoggedSize(i),
                    getCachedVLSN(i));
            }

            /*
             * The applied deltas will leave some slots dirty, which is
             * necessary as a record of changes that will be included in the
             * next delta.  However, the BIN itself should not be dirty,
             * because this delta is a persistent record of those changes.
             */
            fullBIN.setDirty(false);
        } finally {
            fullBIN.releaseLatch();
        }
    }

    /**
     * Apply (insert, update) a given delta slot in this full BIN.
     */
    void applyDelta(final byte[] key,
                    final long lsn,
                    final byte state,
                    final int lastLoggedSize,
                    final long vlsn) {

        /*
         * The delta is the authoritative version of the entry. In all cases,
         * it should supersede the entry in the full BIN.  This is true even if
         * the BIN Delta's entry is knownDeleted or if the full BIN's version
         * is knownDeleted. Therefore we use the flavor of findEntry that will
         * return a knownDeleted entry if the entry key matches (i.e. true,
         * false) but still indicates exact matches with the return index.
         * findEntry only returns deleted entries if third arg is false, but we
         * still need to know if it's an exact match or not so indicateExact is
         * true.
         */
        int foundIndex = findEntry(key, true, false);
        if ((foundIndex >= 0) &&
            ((foundIndex & IN.EXACT_MATCH) != 0)) {

            foundIndex &= ~IN.EXACT_MATCH;

            /*
             * The entry exists in the full version, update it with the delta
             * info.  Note that all state flags should be restored [#22848].
             */
            updateEntry(foundIndex, lsn, lastLoggedSize, state);
        } else {

            /*
             * The entry doesn't exist, insert the delta entry.  We insert the
             * entry even when it is known or pending deleted, since the
             * deleted (and dirty) entry will be needed to log the next delta.
             * [#20737]
             */
            final ChildReference entry =
                new ChildReference(null, key, lsn, state);
            final int result = insertEntry1(entry);
            assert (result & INSERT_SUCCESS) != 0;
            foundIndex = result & ~(IN.INSERT_SUCCESS | IN.EXACT_MATCH);
            setLastLoggedSizeUnconditional(foundIndex, lastLoggedSize);
        }
        setCachedVLSNUnconditional(foundIndex, vlsn);
    }
}
