/*-
 *
 *  This file is part of Oracle Berkeley DB Java Edition
 *  Copyright (C) 2002, 2016 Oracle and/or its affiliates.  All rights reserved.
 *
 *  Oracle Berkeley DB Java Edition is free software: you can redistribute it
 *  and/or modify it under the terms of the GNU Affero General Public License
 *  as published by the Free Software Foundation, version 3.
 *
 *  Oracle Berkeley DB Java Edition is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License in
 *  the LICENSE file along with Oracle Berkeley DB Java Edition.  If not, see
 *  <http://www.gnu.org/licenses/>.
 *
 *  An active Oracle commercial licensing agreement for this product
 *  supercedes this license.
 *
 *  For more information please contact:
 *
 *  Vice President Legal, Development
 *  Oracle America, Inc.
 *  5OP-10
 *  500 Oracle Parkway
 *  Redwood Shores, CA 94065
 *
 *  or
 *
 *  berkeleydb-info_us@oracle.com
 *
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  EOF
 *
 */

package com.sleepycat.collections;

import java.util.Map;

/**
 * A simple <code>Map.Entry</code> implementation that can be used as in
 * input parameter.  Since a <code>MapEntryParameter</code> is not obtained
 * from a map, it is not attached to any map in particular.  To emphasize that
 * changing this object does not change the map, the {@link #setValue} method
 * always throws <code>UnsupportedOperationException</code>.
 *
 * <p><b>Warning:</b> Use of this interface violates the Java Collections
 * interface contract since these state that <code>Map.Entry</code> objects
 * should only be obtained from <code>Map.entrySet()</code> sets, while this
 * class allows constructing them directly.  However, it is useful for
 * performing operations on an entry set such as add(), contains(), etc.  For
 * restrictions see {@link #getValue} and {@link #setValue}.</p>
 *
 * @author Mark Hayes
 */
public class MapEntryParameter<K, V> implements Map.Entry<K, V> {

    private K key;
    private V value;

    /**
     * Creates a map entry with a given key and value.
     *
     * @param key is the key to use.
     *
     * @param value is the value to use.
     */
    public MapEntryParameter(K key, V value) {

        this.key = key;
        this.value = value;
    }

    /**
     * Computes a hash code as specified by {@link
     * java.util.Map.Entry#hashCode}.
     *
     * @return the computed hash code.
     */
    public int hashCode() {

        return ((key == null)    ? 0 : key.hashCode()) ^
               ((value == null)  ? 0 : value.hashCode());
    }

    /**
     * Compares this entry to a given entry as specified by {@link
     * java.util.Map.Entry#equals}.
     *
     * @return the computed hash code.
     */
    public boolean equals(Object other) {

        if (!(other instanceof Map.Entry)) {
            return false;
        }

        Map.Entry e = (Map.Entry) other;

        return ((key == null) ? (e.getKey() == null)
                              : key.equals(e.getKey())) &&
               ((value == null) ? (e.getValue() == null)
                                : value.equals(e.getValue()));
    }

    /**
     * Returns the key of this entry.
     *
     * @return the key of this entry.
     */
    public final K getKey() {

        return key;
    }

    /**
     * Returns the value of this entry.  Note that this will be the value
     * passed to the constructor or the last value passed to {@link #setValue}.
     * It will not reflect changes made to a Map.
     *
     * @return the value of this entry.
     */
    public final V getValue() {

        return value;
    }

    /**
     * Always throws <code>UnsupportedOperationException</code> since this
     * object is not attached to a map.
     */
    public V setValue(V newValue) {

        throw new UnsupportedOperationException();
    }

    final void setValueInternal(V newValue) {

        this.value = newValue;
    }

    /**
     * Converts the entry to a string representation for debugging.
     *
     * @return the string representation.
     */
    public String toString() {

        return "[key [" + key + "] value [" + value + ']';
    }
}
