/*-
 *
 *  This file is part of Oracle Berkeley DB Java Edition
 *  Copyright (C) 2002, 2016 Oracle and/or its affiliates.  All rights reserved.
 *
 *  Oracle Berkeley DB Java Edition is free software: you can redistribute it
 *  and/or modify it under the terms of the GNU Affero General Public License
 *  as published by the Free Software Foundation, version 3.
 *
 *  Oracle Berkeley DB Java Edition is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License in
 *  the LICENSE file along with Oracle Berkeley DB Java Edition.  If not, see
 *  <http://www.gnu.org/licenses/>.
 *
 *  An active Oracle commercial licensing agreement for this product
 *  supercedes this license.
 *
 *  For more information please contact:
 *
 *  Vice President Legal, Development
 *  Oracle America, Inc.
 *  5OP-10
 *  500 Oracle Parkway
 *  Redwood Shores, CA 94065
 *
 *  or
 *
 *  berkeleydb-info_us@oracle.com
 *
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  EOF
 *
 */

package com.sleepycat.je;

/**
 * Specifies the attributes of a sequence.
 */
public class SequenceConfig implements Cloneable {

    /**
     * Default configuration used if null is passed to methods that create a
     * cursor.
     */
    public static final SequenceConfig DEFAULT = new SequenceConfig();

    /* Parameters */
    private int cacheSize = 0;
    private long rangeMin = Long.MIN_VALUE;
    private long rangeMax = Long.MAX_VALUE;
    private long initialValue = 0L;

    /* Flags */
    private boolean allowCreate;
    private boolean decrement;
    private boolean exclusiveCreate;
    private boolean autoCommitNoSync;
    private boolean wrap;

    /**
     * An instance created using the default constructor is initialized with
     * the system's default settings.
     */
    public SequenceConfig() {
    }

    /**
     * Configures the {@link com.sleepycat.je.Database#openSequence
     * Database.openSequence} method to create the sequence if it does not
     * already exist.
     *
     * <p>The default value is false.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param allowCreate If true, configure the {@link
     * com.sleepycat.je.Database#openSequence Database.openSequence} method to
     * create the sequence if it does not already exist.
     *
     * @return this
     */
    public SequenceConfig setAllowCreate(boolean allowCreate) {
        setAllowCreateVoid(allowCreate);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setAllowCreateVoid(boolean allowCreate) {
        this.allowCreate = allowCreate;
    }

    /**
     * Returns true if the {@link com.sleepycat.je.Database#openSequence
     * Database.openSequence} method is configured to create the sequence if it
     * does not already exist.
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return true if the {@link com.sleepycat.je.Database#openSequence
     * Database.openSequence} method is configured to create the sequence if it
     * does not already exist.
     */
    public boolean getAllowCreate() {
        return allowCreate;
    }

    /**
     * Set the Configure the number of elements cached by a sequence handle.
     *
     * <p>The default value is zero.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param cacheSize The number of elements cached by a sequence handle.
     * May not be larger than the size of the range defined by {@link
     * #setRange}.
     *
     * @return this
     */
    public SequenceConfig setCacheSize(int cacheSize) {
        setCacheSizeVoid(cacheSize);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setCacheSizeVoid(int cacheSize) {
        this.cacheSize = cacheSize;
    }

    /**
     * Returns the number of elements cached by a sequence handle..
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return The number of elements cached by a sequence handle..
     */
    public int getCacheSize() {
        return cacheSize;
    }

    /**
     * Specifies that the sequence should be decremented.
     *
     * <p>The default value is false.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param decrement If true, specify that the sequence should be
     * decremented.
     *
     * @return this
     */
    public SequenceConfig setDecrement(boolean decrement) {
        setDecrementVoid(decrement);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setDecrementVoid(boolean decrement) {
        this.decrement = decrement;
    }

    /**
     * Returns true if the sequence is configured to decrement.
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return true if the sequence is configured to decrement.
     */
    public boolean getDecrement() {
         return decrement;
    }

    /**
     * Configures the {@link com.sleepycat.je.Database#openSequence
     * Database.openSequence} method to fail if the database already exists.
     *
     * <p>The default value is false.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param exclusiveCreate If true, configure the {@link
     * com.sleepycat.je.Database#openSequence Database.openSequence} method to
     * fail if the database already exists.
     *
     * @return this
     */
    public SequenceConfig setExclusiveCreate(boolean exclusiveCreate) {
        setExclusiveCreateVoid(exclusiveCreate);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setExclusiveCreateVoid(boolean exclusiveCreate) {
        this.exclusiveCreate = exclusiveCreate;
    }

    /**
     * Returns true if the {@link com.sleepycat.je.Database#openSequence
     * Database.openSequence} method is configured to fail if the database
     * already exists.
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return true if the {@link com.sleepycat.je.Database#openSequence
     * Database.openSequence} method is configured to fail if the database
     * already exists.
     */
    public boolean getExclusiveCreate() {
        return exclusiveCreate;
    }

    /**
     * Sets the initial value for a sequence.
     *
     * <p>The default initial value is zero.</p>
     *
     * <p>This call is only effective when the sequence is being created.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param initialValue The initial value for a sequence.  Must be within
     * the range minimum and maximum values, inclusive.
     *
     * @return this
     */
    public SequenceConfig setInitialValue(long initialValue) {
        setInitialValueVoid(initialValue);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setInitialValueVoid(long initialValue) {
        this.initialValue = initialValue;
    }

    /**
     * Returns the initial value for a sequence..
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return The initial value for a sequence..
     */
    public long getInitialValue() {
        return initialValue;
    }

    /**
     * Configures auto-commit operations on the sequence to not flush the
     * transaction log.
     *
     * <p>The default value is false.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param autoCommitNoSync If true, configure auto-commit operations on
     * the sequence to not flush the transaction log.
     *
     * @return this
     */
    public SequenceConfig setAutoCommitNoSync(boolean autoCommitNoSync) {
        setAutoCommitNoSyncVoid(autoCommitNoSync);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setAutoCommitNoSyncVoid(boolean autoCommitNoSync) {
        this.autoCommitNoSync = autoCommitNoSync;
    }

    /**
     * Returns true if the auto-commit operations on the sequence are configure
     * to not flush the transaction log..
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return true if the auto-commit operations on the sequence are configure
     * to not flush the transaction log..
     */
    public boolean getAutoCommitNoSync() {
        return autoCommitNoSync;
    }

    /**
     * Configures a sequence range.  This call is only effective when the
     * sequence is being created.
     *
     * <p>The default minimum is {@code Long.MIN_VALUE} and the default maximum
     * is {@code Long.MAX_VALUE}.</p>
     *
     * @param min The minimum value for the sequence.  Must be less than max.
     *
     * @param max The maximum value for the sequence.  Must be greater than
     * min.
     *
     * @return this
     */
    public SequenceConfig setRange(long min, long max) {
        setRangeMin(min);
        setRangeMax(max);
        return this;
    }
    
    /**
     * @hidden
     * Configures a sequence range minimum value.  This call is only effective 
     * when the sequence is being created.
     *
     * <p>The default minimum is {@code Long.MIN_VALUE}</p>
     *
     * @param min The minimum value for the sequence.  Must be less than max.
     *
     * @return this
     */
    public SequenceConfig setRangeMin(long min) {
        setRangeMinVoid(min);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setRangeMinVoid(long min) {
        this.rangeMin = min;
    }
    
    /**
     * @hidden
     * Configures a sequence range maximum value.  This call is only effective 
     * when the sequence is being created.
     *
     * <p>the default maximum is {@code Long.MAX_VALUE}.</p>
     *
     * @param max The maximum value for the sequence.  Must be greater than
     * min.
     *
     * @return this
     */
    public SequenceConfig setRangeMax(long max) {
        setRangeMaxVoid(max);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setRangeMaxVoid(long max) {
        this.rangeMax = max;
    }

    /**
     * Returns the minimum value for the sequence.
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return The minimum value for the sequence.
     */
    public long getRangeMin() {
        return rangeMin;
    }

    /**
     * Returns the maximum value for the sequence.
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return The maximum value for the sequence.
     */
    public long getRangeMax() {
        return rangeMax;
    }

    /**
     * Specifies that the sequence should wrap around when it is incremented
     * (decremented) past the specified maximum (minimum) value.
     *
     * <p>The default value is false.</p>
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @param wrap If true, specify that the sequence should wrap around when
     * it is incremented (decremented) past the specified maximum (minimum)
     * value.
     *
     * @return this
     */
    public SequenceConfig setWrap(boolean wrap) {
        setWrapVoid(wrap);
        return this;
    }
    
    /**
     * @hidden
     * The void return setter for use by Bean editors.
     */
    public void setWrapVoid(boolean wrap) {
        this.wrap = wrap;
    }

    /**
     * Returns true if the sequence will wrap around when it is incremented
     * (decremented) past the specified maximum (minimum) value.
     *
     * <p>This method may be called at any time during the life of the
     * application.</p>
     *
     * @return true if the sequence will wrap around when it is incremented
     * (decremented) past the specified maximum (minimum) value.
     */
    public boolean getWrap() {
        return wrap;
    }

    /**
     * Returns a copy of this configuration object.
     */
    @Override
    public SequenceConfig clone() {
        try {
            return (SequenceConfig) super.clone();
        } catch (CloneNotSupportedException willNeverOccur) {
            return null;
        }
    }

    /**
     * Returns the values for each configuration attribute.
     *
     * @return the values for each configuration attribute.
     */
    @Override
    public String toString() {
        return "allowCreate=" + allowCreate +
            "\ncacheSize=" + cacheSize +
            "\ndecrement=" + decrement +
            "\nexclusiveCreate=" + exclusiveCreate +
            "\ninitialValue=" + initialValue +
            "\nautoCommitNoSync=" + autoCommitNoSync +
            "\nrangeMin=" + rangeMin +
            "\nrangeMax=" + rangeMax +
            "\nwrap=" + wrap +
            "\n";
    }
}
