/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.rep.impl.networkRestore;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.log.LogManager;
import com.sleepycat.je.log.RestoreMarker;
import com.sleepycat.je.log.entry.RestoreRequired;
import com.sleepycat.je.rep.impl.RepImpl;
import com.sleepycat.je.rep.impl.networkRestore.LogFileFeeder;
import com.sleepycat.je.rep.impl.networkRestore.NetworkBackupStatDefinition;
import com.sleepycat.je.rep.impl.networkRestore.NetworkBackupStats;
import com.sleepycat.je.rep.impl.networkRestore.Protocol;
import com.sleepycat.je.rep.impl.node.NameIdPair;
import com.sleepycat.je.rep.net.DataChannel;
import com.sleepycat.je.rep.net.DataChannelFactory;
import com.sleepycat.je.rep.utilint.BinaryProtocol;
import com.sleepycat.je.rep.utilint.ServiceDispatcher;
import com.sleepycat.je.utilint.AtomicIntStat;
import com.sleepycat.je.utilint.AtomicLongStat;
import com.sleepycat.je.utilint.LoggerUtils;
import com.sleepycat.je.utilint.LongAvgRateStat;
import com.sleepycat.je.utilint.StatGroup;
import com.sleepycat.je.utilint.TestHook;
import com.sleepycat.je.utilint.TestHookExecute;
import com.sleepycat.je.utilint.VLSN;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class NetworkBackup {
    private final InetSocketAddress serverAddress;
    private final File envDir;
    private final NameIdPair clientNameId;
    private final boolean retainLogfiles;
    private final VLSN minVLSN;
    final int serverLoadThreshold;
    private final RepImpl repImpl;
    private final FileManager fileManager;
    private final DataChannelFactory channelFactory;
    private Protocol protocol;
    private DataChannel channel;
    final MessageDigest messageDigest;
    private final StatGroup statistics;
    private final AtomicIntStat backupFileCount;
    private final AtomicIntStat disposedCount;
    private final AtomicIntStat fetchCount;
    private final AtomicIntStat skipCount;
    private final AtomicLongStat expectedBytes;
    private final AtomicLongStat transferredBytes;
    private final LongAvgRateStat transferRate;
    private final Logger logger;
    private CyclicBarrier testBarrier = null;
    private final int receiveBufferSize;
    private static final int SOCKET_TIMEOUT_MS = 10000;
    private static final int DIGEST_RETRIES = 5;
    private final Properties exceptionProperties;
    private final RestoreMarker restoreMarker;
    private TestHook<File> interruptHook;

    public NetworkBackup(InetSocketAddress serverSocket, int receiveBufferSize, File envDir, NameIdPair clientNameId, boolean retainLogfiles, int serverLoadThreshold, VLSN minVLSN, RepImpl repImpl, FileManager fileManager, LogManager logManager, DataChannelFactory channelFactory, Properties exceptionProperties) throws IllegalArgumentException {
        this.serverAddress = serverSocket;
        this.receiveBufferSize = receiveBufferSize;
        if (!envDir.exists()) {
            throw new IllegalArgumentException("Environment directory: " + envDir + " not found");
        }
        this.envDir = envDir;
        this.clientNameId = clientNameId;
        this.retainLogfiles = retainLogfiles;
        this.serverLoadThreshold = serverLoadThreshold;
        this.minVLSN = minVLSN;
        this.repImpl = repImpl;
        this.fileManager = fileManager;
        this.channelFactory = channelFactory;
        try {
            this.messageDigest = MessageDigest.getInstance("SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            throw EnvironmentFailureException.unexpectedException(e);
        }
        this.logger = LoggerUtils.getLoggerFixedPrefix(this.getClass(), clientNameId.toString(), repImpl);
        this.statistics = new StatGroup("NetworkBackup", "NetworkBackup statistics");
        this.backupFileCount = new AtomicIntStat(this.statistics, NetworkBackupStatDefinition.BACKUP_FILE_COUNT);
        this.disposedCount = new AtomicIntStat(this.statistics, NetworkBackupStatDefinition.DISPOSED_COUNT);
        this.fetchCount = new AtomicIntStat(this.statistics, NetworkBackupStatDefinition.FETCH_COUNT);
        this.skipCount = new AtomicIntStat(this.statistics, NetworkBackupStatDefinition.SKIP_COUNT);
        this.expectedBytes = new AtomicLongStat(this.statistics, NetworkBackupStatDefinition.EXPECTED_BYTES);
        this.transferredBytes = new AtomicLongStat(this.statistics, NetworkBackupStatDefinition.TRANSFERRED_BYTES);
        this.transferRate = new LongAvgRateStat(this.statistics, NetworkBackupStatDefinition.TRANSFER_RATE, 10000L, TimeUnit.MINUTES);
        this.exceptionProperties = exceptionProperties;
        this.restoreMarker = new RestoreMarker(fileManager, logManager);
    }

    public NetworkBackup(InetSocketAddress serverSocket, File envDir, NameIdPair clientNameId, boolean retainLogfiles, FileManager fileManager, LogManager logManager, DataChannelFactory channelFactory) throws DatabaseException {
        this(serverSocket, 0, envDir, clientNameId, retainLogfiles, Integer.MAX_VALUE, VLSN.NULL_VLSN, null, fileManager, logManager, channelFactory, new Properties());
    }

    public NetworkBackupStats getStats() {
        return new NetworkBackupStats(this.statistics.cloneGroup(false));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String[] execute() throws IOException, DatabaseException, ServiceDispatcher.ServiceConnectFailedException, LoadThresholdExceededException, InsufficientVLSNRangeException {
        try {
            this.channel = this.channelFactory.connect(this.serverAddress, new DataChannelFactory.ConnectOptions().setTcpNoDelay(true).setReceiveBufferSize(this.receiveBufferSize).setOpenTimeout(10000).setReadTimeout(10000));
            ServiceDispatcher.doServiceHandshake(this.channel, "LogFileFeeder");
            this.protocol = this.checkProtocol(new Protocol(this.clientNameId, 2, this.repImpl));
            this.checkServer();
            String[] fileNames = this.getFileList();
            LoggerUtils.info(this.logger, this.repImpl, "Restoring from:" + this.serverAddress + " Allocated network receive buffer size:" + this.channel.getSocketChannel().socket().getReceiveBufferSize() + "(" + this.receiveBufferSize + ")" + " candidate log file count:" + fileNames.length);
            this.getFiles(fileNames);
            this.cleanup(fileNames);
            assert (this.fileManager.listJDBFiles().length == fileNames.length) : "envDir=" + this.envDir + " list=" + Arrays.asList(this.fileManager.listJDBFiles()) + " fileNames=" + Arrays.asList(fileNames);
            long fileBegin = this.fileManager.getNumFromName(fileNames[0]);
            long fileEnd = this.fileManager.getNumFromName(fileNames[fileNames.length - 1]);
            String[] stringArray = this.fileManager.listFileNames(fileBegin, fileEnd);
            return stringArray;
        }
        finally {
            if (this.channel != null) {
                this.channel.close();
            }
            LoggerUtils.info(this.logger, this.repImpl, "Backup file total: " + this.backupFileCount.get() + ".  Files actually fetched: " + this.fetchCount.get() + ".  Files skipped(available locally): " + this.skipCount.get() + ".  Local files renamed/deleted: " + this.disposedCount.get());
        }
    }

    private void checkServer() throws IOException, BinaryProtocol.ProtocolException, LoadThresholdExceededException, InsufficientVLSNRangeException {
        this.protocol.write((BinaryProtocol.Message)new Protocol.FeederInfoReq(this.protocol), this.channel);
        Protocol.FeederInfoResp resp = this.protocol.read(this.channel, Protocol.FeederInfoResp.class);
        if (!(VLSN.NULL_VLSN.equals(this.minVLSN) || resp.getRangeFirst().compareTo(this.minVLSN) <= 0 && resp.getRangeLast().compareTo(this.minVLSN) >= 0)) {
            throw new InsufficientVLSNRangeException(this.minVLSN, resp.getRangeFirst(), resp.getRangeLast());
        }
        if (resp.getActiveFeeders() > this.serverLoadThreshold) {
            LoadThresholdExceededException exception = new LoadThresholdExceededException(this.serverLoadThreshold, resp.getActiveFeeders());
            throw exception;
        }
    }

    private void cleanup(String[] fileNames) throws IOException {
        LoggerUtils.fine(this.logger, this.repImpl, "Cleaning up");
        HashSet<String> logFileSet = new HashSet<String>(Arrays.asList(fileNames));
        for (File file : this.fileManager.listJDBFiles()) {
            if (logFileSet.contains(file.getName())) continue;
            this.disposeFile(file);
        }
        StringBuilder logFiles = new StringBuilder();
        for (String string : logFileSet) {
            File file;
            file = new File(this.fileManager.getFullFileName(string));
            if (!file.exists()) {
                throw EnvironmentFailureException.unexpectedState("Missing file: " + file);
            }
            logFiles.append(file.getCanonicalPath()).append(", ");
        }
        String names = logFiles.toString();
        if (names.length() > 0) {
            names = names.substring(0, names.length() - 2);
        }
        LoggerUtils.fine(this.logger, this.repImpl, "Log file set: " + names);
    }

    private void getFiles(String[] fileNames) throws IOException, DatabaseException {
        LoggerUtils.info(this.logger, this.repImpl, fileNames.length + " files in backup set");
        List<FileAndLength> fileTransferLengths = this.getFileTransferLengths(fileNames);
        block5: for (FileAndLength entry : fileTransferLengths) {
            if (this.testBarrier != null) {
                try {
                    this.testBarrier.await();
                }
                catch (InterruptedException interruptedException) {
                }
                catch (BrokenBarrierException e) {
                    throw EnvironmentFailureException.unexpectedException(e);
                }
            }
            for (int i = 0; i < 5; ++i) {
                try {
                    this.getFile(entry.file);
                    this.fetchCount.increment();
                    continue block5;
                }
                catch (DigestException e) {
                    if (i + 1 == 5) {
                        throw new IOException("Digest mismatch despite 5 attempts");
                    }
                    this.expectedBytes.add(entry.length);
                    continue;
                }
            }
        }
        this.restoreMarker.removeMarkerFile(this.fileManager);
        this.protocol.write((BinaryProtocol.Message)new Protocol.Done(this.protocol), this.channel);
    }

    private List<FileAndLength> getFileTransferLengths(String[] fileNames) throws IOException, DatabaseException {
        ArrayList<FileAndLength> fileTransferLengths = new ArrayList<FileAndLength>();
        for (String fileName : fileNames) {
            File file = new File(this.fileManager.getFullFileName(fileName));
            Protocol protocol = this.protocol;
            protocol.getClass();
            this.protocol.write((BinaryProtocol.Message)new Protocol.FileInfoReq(protocol, fileName, false), this.channel);
            Protocol.FileInfoResp statResp = this.protocol.read(this.channel, Protocol.FileInfoResp.class);
            long fileLength = statResp.getFileLength();
            if (file.exists() && fileLength == file.length()) {
                byte[] digest;
                if (statResp.getDigestSHA1().length == 0) {
                    Protocol protocol2 = this.protocol;
                    protocol2.getClass();
                    this.protocol.write((BinaryProtocol.Message)new Protocol.FileInfoReq(protocol2, fileName, true), this.channel);
                    statResp = this.protocol.read(this.channel, Protocol.FileInfoResp.class);
                }
                if (Arrays.equals(digest = LogFileFeeder.getSHA1Digest(file, fileLength).digest(), statResp.getDigestSHA1())) {
                    LoggerUtils.info(this.logger, this.repImpl, "File: " + file.getCanonicalPath() + " length: " + fileLength + " available with matching SHA1, copy skipped");
                    this.skipCount.increment();
                    continue;
                }
            }
            fileTransferLengths.add(new FileAndLength(file, fileLength));
            this.expectedBytes.add(fileLength);
        }
        return fileTransferLengths;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void getFile(File file) throws IOException, BinaryProtocol.ProtocolException, DigestException {
        boolean deleted;
        LoggerUtils.fine(this.logger, this.repImpl, "Requesting file: " + file);
        Protocol protocol = this.protocol;
        protocol.getClass();
        this.protocol.write((BinaryProtocol.Message)new Protocol.FileReq(protocol, file.getName()), this.channel);
        Protocol.FileStart fileResp = this.protocol.read(this.channel, Protocol.FileStart.class);
        File tmpFile = new File(this.fileManager.getFullFileName(file.getName()) + ".tmp");
        if (tmpFile.exists() && !(deleted = tmpFile.delete())) {
            throw EnvironmentFailureException.unexpectedState("Could not delete file: " + tmpFile);
        }
        ByteBuffer buffer = ByteBuffer.allocateDirect(8192);
        this.messageDigest.reset();
        FileOutputStream fileStream = new FileOutputStream(tmpFile);
        FileChannel fileChannel = fileStream.getChannel();
        try {
            int actualBytes;
            int rateInterval = 128;
            int count = 0;
            for (long bytes = fileResp.getFileLength(); bytes > 0L; bytes -= (long)actualBytes) {
                int readSize = (int)Math.min(8192L, bytes);
                buffer.clear();
                buffer.limit(readSize);
                actualBytes = this.channel.read(buffer);
                if (actualBytes == -1) {
                    throw new IOException("Premature EOF. Was expecting:" + readSize);
                }
                buffer.flip();
                fileChannel.write(buffer);
                buffer.rewind();
                this.messageDigest.update(buffer);
                this.transferredBytes.add(actualBytes);
                if (++count % 128 != 0 && bytes > 0L) continue;
                this.transferRate.add(this.transferredBytes.get(), System.currentTimeMillis());
            }
            if (this.logger.isLoggable(Level.INFO)) {
                LoggerUtils.info(this.logger, this.repImpl, String.format("Fetched log file: %s, size: %,d bytes, %s bytes, %s bytes, %s bytes/second", file.getName(), fileResp.getFileLength(), this.transferredBytes, this.expectedBytes, this.transferRate));
            }
        }
        finally {
            fileStream.close();
        }
        Protocol.FileEnd fileEnd = this.protocol.read(this.channel, Protocol.FileEnd.class);
        if (!Arrays.equals(this.messageDigest.digest(), fileEnd.getDigestSHA1())) {
            LoggerUtils.warning(this.logger, this.repImpl, "digest mismatch on file: " + file);
            throw new DigestException();
        }
        this.restoreMarker.createMarkerFile(RestoreRequired.FailureType.NETWORK_RESTORE, this.exceptionProperties);
        TestHookExecute.doHookIfSet(this.interruptHook, file);
        if (file.exists()) {
            this.disposeObsoleteFiles(file);
        }
        LoggerUtils.fine(this.logger, this.repImpl, "Renamed " + tmpFile + " to " + file);
        boolean renamed = tmpFile.renameTo(file);
        if (!renamed) {
            throw EnvironmentFailureException.unexpectedState("Rename from: " + tmpFile + " to " + file + " failed");
        }
        if (!file.setLastModified(fileResp.getLastModifiedTime())) {
            throw EnvironmentFailureException.unexpectedState("File.setlastModifiedTime() for:" + file + " and time " + new Date(fileResp.getLastModifiedTime()) + " failed.");
        }
    }

    private void disposeObsoleteFiles(File startFile) throws IOException {
        Object[] dirFiles = this.fileManager.listJDBFiles();
        Arrays.sort(dirFiles);
        for (int i = dirFiles.length - 1; i >= 0; --i) {
            Object file = dirFiles[i];
            if (((File)file).getName().equals(RestoreMarker.getMarkerFileName())) continue;
            this.disposeFile((File)file);
            if (startFile.equals(file)) break;
        }
    }

    private void disposeFile(File file) {
        this.disposedCount.increment();
        if (this.retainLogfiles) {
            long fileNumber = this.fileManager.getNumFromName(file.getName());
            boolean renamed = false;
            try {
                renamed = this.fileManager.renameFile(fileNumber, ".bup");
            }
            catch (IOException e) {
                throw EnvironmentFailureException.unexpectedState("Could not rename log file " + file.getPath() + " because of exception: " + e.getMessage());
            }
            if (!renamed) {
                throw EnvironmentFailureException.unexpectedState("Could not rename log file " + file.getPath());
            }
            LoggerUtils.fine(this.logger, this.repImpl, "Renamed log file: " + file.getPath());
        } else {
            boolean deleted = file.delete();
            if (!deleted) {
                throw EnvironmentFailureException.unexpectedState("Could not delete log file " + file.getPath());
            }
            LoggerUtils.fine(this.logger, this.repImpl, "deleted log file: " + file.getPath());
        }
    }

    private String[] getFileList() throws IOException, BinaryProtocol.ProtocolException {
        this.protocol.write((BinaryProtocol.Message)new Protocol.FileListReq(this.protocol), this.channel);
        Protocol.FileListResp fileListResp = this.protocol.read(this.channel, Protocol.FileListResp.class);
        Object[] fileList = fileListResp.getFileNames();
        Arrays.sort(fileList);
        this.backupFileCount.set(fileList.length);
        return fileList;
    }

    private Protocol checkProtocol(Protocol candidateProtocol) throws IOException, BinaryProtocol.ProtocolException {
        candidateProtocol.write((BinaryProtocol.Message)new BinaryProtocol.ClientVersion(candidateProtocol), this.channel);
        BinaryProtocol.ServerVersion serverVersion = candidateProtocol.read(this.channel, BinaryProtocol.ServerVersion.class);
        if (serverVersion.getVersion() != candidateProtocol.getVersion()) {
            String message = "Server requested protocol version:" + serverVersion.getVersion() + " but the client version is " + candidateProtocol.getVersion();
            LoggerUtils.info(this.logger, this.repImpl, message);
            throw new BinaryProtocol.ProtocolException(message);
        }
        return candidateProtocol;
    }

    public void setTestBarrier(CyclicBarrier testBarrier) {
        this.testBarrier = testBarrier;
    }

    public void setInterruptHook(TestHook<File> hook) {
        this.interruptHook = hook;
    }

    public static class LoadThresholdExceededException
    extends Exception {
        final int threshold;
        final int activeServers;

        LoadThresholdExceededException(int threshold, int activeServers) {
            assert (activeServers > threshold);
            this.threshold = threshold;
            this.activeServers = activeServers;
        }

        public int getActiveServers() {
            return this.activeServers;
        }

        public int getThreshold() {
            return this.threshold;
        }

        @Override
        public String getMessage() {
            return "Active server threshold: " + this.threshold + " exceeded. " + "Active servers: " + this.activeServers;
        }
    }

    public static class InsufficientVLSNRangeException
    extends Exception {
        private final VLSN minVLSN;
        private final VLSN rangeFirst;
        private final VLSN rangeLast;

        public InsufficientVLSNRangeException(VLSN minVLSN, VLSN rangeFirst, VLSN rangeLast) {
            this.minVLSN = minVLSN;
            this.rangeFirst = rangeFirst;
            this.rangeLast = rangeLast;
        }

        public VLSN getMinVLSN() {
            return this.minVLSN;
        }

        public VLSN getRangeFirst() {
            return this.rangeFirst;
        }

        @Override
        public String getMessage() {
            return "Insufficient VLSN range. Needed VLSN: " + this.minVLSN + " Available range: " + "[" + this.rangeFirst + ", " + this.rangeLast + "]";
        }
    }

    protected static class DigestException
    extends Exception {
        protected DigestException() {
        }
    }

    private static class FileAndLength {
        final File file;
        final long length;

        FileAndLength(File file, long length) {
            this.file = file;
            this.length = length;
        }
    }
}

