package com.dji.wpmzsdk.common.utils.kml


import android.util.Log
import com.dji.wpmzsdk.common.utils.kml.data.MissionImportHeightMode
import com.dji.industry.pilot.missionflight.library.MissionImportParams


import com.dji.wpmzsdk.common.utils.DJIGpsUtils
import com.dji.wpmzsdk.common.utils.kml.data.MissionType
import com.dji.wpmzsdk.common.utils.kml.mission.MissionInfoModel
import com.dji.wpmzsdk.common.utils.kml.model.*


import org.dom4j.Document
import org.dom4j.Element
import org.dom4j.io.SAXReader
import java.io.File

/**
 * Description :
 *
 * @filename : KMLCreateMissionUtil
 * @author : devin.xu
 * @time : 2022/12/1
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class KMLCreateMissionUtil {

    fun createMission(params: MissionImportParams, path: String): Any {
        val suffixIndex = path.lastIndexOf('.')
        if (suffixIndex < 0) {
            throw KMLException(KMLConstants.NON_KML_FILE)
        }

        val saxReader = SAXReader()
        var kmlPath = path
        var f = File(kmlPath)

        return if (KMLValueConverter.equals(kmlPath.substring(kmlPath.lastIndexOf('.')), KMLConstants.SUFFIX)) {
            var name = f.name.substring(0, f.name.lastIndexOf("."))
            name = name.replace(KMLUtil.ILLEGAL_EX.toRegex(), "")
            val type = params.missionType
            if (type == MissionType.Waypoint) {
                createWaypointMission(saxReader.read(f), name, params.heightMode)
            } else if (type == MissionType.Strip) {
                createStripMission(saxReader.read(f), name)
            } else {
                createMappingMission(saxReader.read(f), name, type!!)
            }
        } else {
            throw KMLException(KMLConstants.NON_KML_FILE)
        }
    }

    private fun createWaypointMission(kml: Document, name: String, heightMode: MissionImportHeightMode?): WaypointMissionModel {
        val document = kml.rootElement.element(KMLConstants.DOCUMENT)
        if (document == null || getWaylinePointsElement(document) == null) {
            throw KMLException(KMLConstants.MISSION_TYPE_ILLEGAL)
        }
        var missionName = name
        if (name.length > KMLConstants.MAX_MISSION_NAME_LENGTH) {
            missionName = name.substring(0, KMLConstants.MAX_MISSION_NAME_LENGTH)
        }

        val waypointMissionModel = SettingUtils.createWaypointMissionModel(missionName)
        val missionInfoModel = waypointMissionModel.missionInfo
        missionInfoModel.extInfo.isCreateByKml = true

        val waylineEle = getWaylinePointsElement(document)
        parseWaylineData(waylineEle, waypointMissionModel, heightMode)

        injectHeightMode(heightMode, waypointMissionModel)
        return waypointMissionModel

    }

    private fun getWaylinePointsElement(element: Element?): Element? {
        if (element == null) {
            return null
        }
        for (placeMarker in element.elements()) {
            if (placeMarker.name == KMLConstants.PLACE_MARK) {
                val lineEle = placeMarker.element(KMLConstants.LINE_STRING)
                if (lineEle != null) {
                    return placeMarker
                }
            }
        }
        return null
    }

    private fun parseWaylineData(element: Element?, waypointMissionModel: WaypointMissionModel, heightMode: MissionImportHeightMode?) {
        val waylineModel = waypointMissionModel.wayline
        val isEgmHeight = heightMode === MissionImportHeightMode.EGM96

        val points = waypointMissionModel.waypoints
        // 从线集解析
        val lineStringEle = element?.element(KMLConstants.LINE_STRING) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val coordinatesEle = lineStringEle.element(KMLConstants.COORDINATES) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val locations = coordinatesEle.textTrim.split(" ".toRegex())
        try {
            for (i in locations.indices) {
                var point = SettingUtils.createWaypointModel()
                val pointCoor = locations[i].split(",".toRegex())
                val lng = KMLValueConverter.string2Double(pointCoor[0].trim())
                val lat = KMLValueConverter.string2Double(pointCoor[1].trim())
                var alt = KMLValueConverter.string2Float(pointCoor[2].trim())
                if (DJIGpsUtils.isAvailable(lat, lng)) {
                    point.latitude = lat
                    point.longitude = lng
                    if (alt >= SettingUtils.MIN_WAYPOINT_HEIGHT && alt <= SettingUtils.MAX_WAYPOINT_HEIGHT) {
                        if (heightMode === MissionImportHeightMode.EGM96) {
                            point.altitude = alt
                            point.wgs84Altitude = GpsUtils.wgs84Altitude(alt.toDouble(), lat, lng).toFloat()
                            point.isUseWaylineAltitude = false
                        } else if (heightMode == MissionImportHeightMode.WGS84) {
                            point.wgs84Altitude = alt
                            point.altitude = GpsUtils.egm96Altitude(alt.toDouble(), lat, lng).toFloat()
                            point.isUseWaylineAltitude = false
                        } else {
                            point.altitude = alt
                            // 如果是原始 kml 数据，并且航点没有写高度值，设置为跟随航线
                            point.isUseWaylineAltitude = alt == 0f
                        }
                    } else {
                        point.altitude = SettingUtils.DEF_WAYLINE_HEIGHT
                    }
                }
                points.add(point)
            }
            waypointMissionModel.waypoints = points
        } catch (e: NumberFormatException) {
            Log.e("KMLCreateMission", KMLConstants.LOCATION_ILLEGAL)
        }

        val infoModel = waypointMissionModel.missionInfo
        if (points.isNotEmpty() && infoModel != null) {
            val latitude = points[0].latitude
            val longitude = points[0].longitude
            infoModel.latitude = latitude
            infoModel.longitude = longitude
            if (heightMode === MissionImportHeightMode.WGS84) {
                waylineModel.altitude = GpsUtils.egm96Altitude(waylineModel.altitude.toDouble(), latitude, longitude).toFloat()
            }
        }
    }

    private fun injectHeightMode(heightMode: MissionImportHeightMode?, waypointMissionModel: WaypointMissionModel) {
        if (heightMode != null) {
            waypointMissionModel.wayline.droneInfo.droneHeight.isUseAbsolute = isAbsoluteHeight(heightMode)
        }
    }

    private fun isAbsoluteHeight(heightMode: MissionImportHeightMode): Boolean {
        return when (heightMode) {
            MissionImportHeightMode.WGS84, MissionImportHeightMode.EGM96 -> true
            MissionImportHeightMode.RELATIVE -> false
            else -> false
        }
    }


    private fun createMappingMission(kml: Document, name: String, type: MissionType): MappingMissionModel {
        val document = kml.rootElement.element(KMLConstants.DOCUMENT) ?: throw KMLException(KMLConstants.PARAM_ILLEGAL)
        var missionName = name
        if (name.length > KMLConstants.MAX_MISSION_NAME_LENGTH) {
            missionName = name.substring(0, KMLConstants.MAX_MISSION_NAME_LENGTH)
        }
        val camera = MappingUtils.createMappingCameraModel(MappingCameraType.EP600_35MM)
        val mappingMissionModel = MappingUtils.createMappingModel(missionName, type, camera, null)
        val extraInfo = MissionInfoExtModel()
        extraInfo.isCreateByKml = true
        mappingMissionModel.missionInfo.extInfo = extraInfo
        val docElement = document.elements()
        for (element in docElement) {
            // Mission Info
            if (element.name == KMLConstants.PLACE_MARK) {
                // parse Points
                parseMappingPointsData(element, mappingMissionModel)
            }
        }
        if (mappingMissionModel.edgePoints == null || mappingMissionModel.edgePoints.isEmpty()) {
            throw KMLException(KMLConstants.PARAM_ILLEGAL)
        }
        return mappingMissionModel
    }

    private fun parseMappingPointsData(element: Element, model: MappingMissionModel) {
        val polygonEle = element.element(KMLConstants.POLYGON) ?: throw KMLException(KMLConstants.MISSION_TYPE_ILLEGAL)
        val outerBoundaryEle = polygonEle.element(KMLConstants.OUTER_BOUNDARY_IS) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val linearRingEle = outerBoundaryEle.element(KMLConstants.LINEAR_RING) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val coordinatesEle = linearRingEle.element(KMLConstants.COORDINATES) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val points: MutableList<EdgePointModel> = mutableListOf()
        val locations = coordinatesEle.textTrim.split(" ".toRegex())
        try {
            locations.forEach { location ->
                val point = EdgePointModel()
                val pointCoor = location.split(",".toRegex())
                val lng = KMLValueConverter.string2Double(pointCoor[0].trim())
                val lat = KMLValueConverter.string2Double(pointCoor[1].trim())
                if (DJIGpsUtils.isAvailable(lat, lng)) {
                    point.latitude = lat
                    point.longitude = lng
                    points.add(point)
                }
            }
            model.edgePoints = points
        } catch (e: java.lang.NumberFormatException) {
            Log.e("KMLCreateMission", KMLConstants.EDGE_POINTS_LOCATION_ILLEGAL)
        }
        val infoModel: MissionInfoModel = model.missionInfo
        if (points.isNotEmpty()) {
            infoModel.latitude = points[0]!!.latitude
            infoModel.longitude = points[0]!!.longitude
        }
    }

    private fun createStripMission(kml: Document, name: String): StripMissionModel {
        val document = kml.rootElement.element(KMLConstants.DOCUMENT) ?: throw KMLException(KMLConstants.PARAM_ILLEGAL)
        var missionName = name
        if (name.length > KMLConstants.MAX_MISSION_NAME_LENGTH) {
            missionName = name.substring(0, KMLConstants.MAX_MISSION_NAME_LENGTH)
        }

        val camera = StripUtils.createStripCameraModel(MappingCameraType.EP600_35MM)
        val stripMissionModel = StripUtils.createStripModel(missionName, MissionType.Strip, camera)
        val extraInfo = MissionInfoExtModel()
        extraInfo.isCreateByKml = true
        stripMissionModel.missionInfo.extInfo = extraInfo

        val docElement = document.elements()
        for (element in docElement) {
            if (element.name == KMLConstants.PLACE_MARK) {
                // parse Points
                parseStripPointsData(element, stripMissionModel)
            } else if (element.name == KMLConstants.EXTENDED_DATA) {
                val type = element.element(KMLConstants.MISSION_TYPE)
                if (type != null && KMLValueConverter.getMissionType(type.textTrim) != MissionType.Strip) {
                    throw KMLException(KMLConstants.MISSION_TYPE_ILLEGAL)
                }
            }
        }
        if (stripMissionModel.edgePoints == null || stripMissionModel.edgePoints.isEmpty()) {
            throw KMLException(KMLConstants.PARAM_ILLEGAL)
        }
        return stripMissionModel
    }

    private fun parseStripPointsData(element: Element, model: StripMissionModel) {
        // 从线集解析
        val lineStringEle = element.element(KMLConstants.LINE_STRING) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val coordinatesEle = lineStringEle.element(KMLConstants.COORDINATES) ?: throw KMLException(KMLConstants.LOCATION_ILLEGAL)
        val edgePoints: MutableList<StripEdgePointModel> = mutableListOf()
        val locations = coordinatesEle.textTrim.split(" ".toRegex())
        try {
            locations.forEach { location ->
                val point = StripEdgePointModel()
                val pointCoor = location.split(",".toRegex())
                val lng = KMLValueConverter.string2Double(pointCoor[0].trim())
                val lat = KMLValueConverter.string2Double(pointCoor[1].trim())
                if (DJIGpsUtils.isAvailable(lat, lng)) {
                    point.latitude = lat
                    point.longitude = lng
                    edgePoints.add(point)
                }
            }
            model.edgePoints = edgePoints
        } catch (e: java.lang.NumberFormatException) {
            Log.e("KMLCreateMission", KMLConstants.LOCATION_ILLEGAL)
        }
        val infoModel: MissionInfoModel? = model.missionInfo
        if (CollectionUtil.isNotEmpty(edgePoints) && infoModel != null) {
            infoModel.latitude = edgePoints[0]!!.latitude
            infoModel.longitude = edgePoints[0]!!.longitude
        }
    }

}