package com.dji.wpmzsdk.common.utils.kml;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;


import com.dji.industry.pilot.data.cache.model.EdgePointModelGreenDao;
import com.dji.industry.pilot.data.cache.model.MappingCameraModelGreenDao;
import com.dji.industry.pilot.data.cache.model.MappingMissionModelGreenDao;
import com.dji.industry.pilot.data.cache.model.MappingWaylineModelGreenDao;
import com.dji.industry.pilot.data.cache.model.StripMissionModelGreenDao;
import com.dji.industry.pilot.data.cache.model.StripWaylineModelGreenDao;
import com.dji.industry.pilot.data.cache.model.WaypointActionModelGreenDao;
import com.dji.wpmzsdk.common.utils.PPalGenerator;
import com.dji.wpmzsdk.common.utils.kml.converter.KMZPathWrapper;
import com.dji.wpmzsdk.common.utils.kml.converter.MissionGreenDaoTransform;
import com.dji.wpmzsdk.common.utils.kml.converter.WaylineConverter;
import com.dji.wpmzsdk.common.utils.kml.model.DroneCameraModel;
import com.dji.wpmzsdk.common.utils.kml.model.DroneHeightModel;
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraType;
import com.dji.wpmzsdk.common.utils.kml.model.MappingMissionModel;
import com.dji.wpmzsdk.common.utils.kml.model.MissionInfoExtModel;
import com.dji.wpmzsdk.common.utils.kml.model.PayloadConfigInfoModel;
import com.dji.wpmzsdk.common.utils.kml.model.PayloadInfoModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointActionType;
import com.dji.wpmzsdk.common.utils.kml.model.PreciseShotInfo;
import com.dji.wpmzsdk.common.utils.kml.model.StripMissionModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointActionModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointMissionModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointModelGreenDao;
import com.dji.wpmzsdk.common.utils.DJIGpsUtils;
import com.dji.wpmzsdk.common.utils.kml.data.DroneInfoModel;
import com.dji.wpmzsdk.common.utils.kml.mission.RTKReferenceStationSource;
import com.dji.wpmzsdk.common.utils.kml.mission.WaypointMissionGotoWaypointMode;
import com.dji.wpmzsdk.common.utils.kml.mission.WaypointMissionHeadingMode;
import com.dji.wpmzsdk.common.utils.kml.model.MissionInfoModelGreenDao;
import com.dji.wpmzsdk.common.utils.kml.model.WaylineModelGreenDao;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointMissionModelGreenDao;
import com.dji.wpmzsdk.common.utils.kml.data.MissionImportHeightMode;
import com.dji.industry.pilot.missionflight.library.MissionImportParams;

import com.dji.wpmzsdk.common.utils.FileUtils;
import com.dji.wpmzsdk.common.utils.KMLZipHelper;
import com.dji.wpmzsdk.common.utils.kml.data.MissionType;
import com.dji.wpmzsdk.common.utils.kml.transfrom.MissionTransformData;

import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;

import dji.sdk.wpmz.jni.JNIWPMZManager;
import dji.sdk.wpmz.value.mission.Wayline;
import dji.sdk.wpmz.value.mission.WaylineExecuteMissionConfig;
import dji.sdk.wpmz.value.mission.WaylineTemplate;


/**
 * KML 导入导出工具类
 *
 * @author bryan.jia
 */
public class KMLUtil {

    private final static String TAG = "KMLUtil";

    public static final float MAX_SPEED = 15;
    public static final int MIN_HEADING = -180;
    public static final int MAX_HEADING = 180;
    public static final String VERSION ="1.0.4";
    private static  Context context ;

    public static final String CACHE_PATH = "/sdcard/DJI/com.dji.wpmzsdk/CACHE_IMAGE";
    public static final String PATH = "/sdcard/DJI/com.dji.wpmzsdk/KML";
    //非法字符
    public final static String ILLEGAL_EX = "[ _`~!@#$%^&*()+=|{}':;',\\[\\]\\\\.<>/?~！@#￥%……&*（）\"——+|{}【】‘；：”“’。，、？]|\n|\r|\t";
    public static final Pattern ILLEGAL_PATTERN = Pattern.compile(ILLEGAL_EX);

    public static KMLFileParseInfo getKMLType(String path) {
        if (KMLValueConverter.equals(path.substring(path.lastIndexOf('.')), KMLConstants.SUFFIX)) {
            return parseKMLFile(path);

        } else if (KMLValueConverter.equals(path.substring(path.lastIndexOf('.')), KMLConstants.ZIP_SUFFIX)) {
            return new KMLFileParseInfo(KMLFileParseInfo.KMLFileType.MISSION);

        } else {
            return new KMLFileParseInfo(KMLFileParseInfo.KMLFileType.UNKNOWN);
        }
    }

    public static void setContext(Context ctx) {
        KMLUtil.context = ctx;
    }

    private static KMLFileParseInfo parseKMLFile(String path) {
        SAXReader saxReader = new SAXReader();
        File file = new File(path);
        Document kmlDocument;
        try {
            kmlDocument = saxReader.read(file);
        } catch (DocumentException e) {
            return new KMLFileParseInfo(KMLFileParseInfo.KMLFileType.UNKNOWN);
        }

        Element document = kmlDocument.getRootElement().element(KMLConstants.DOCUMENT);
        if (document == null) {
            return new KMLFileParseInfo(KMLFileParseInfo.KMLFileType.UNKNOWN);
        }

        List<Element> docElement = document.elements();
        for (Element element : docElement) {
            if (element.getName().equals(KMLConstants.EXTENDED_DATA)) {
                Element type = element.element(KMLConstants.MISSION_TYPE);
                if (type != null && KMLValueConverter.getMissionType(type.getTextTrim()) != null) {
                    KMLFileParseInfo parseInfo = new KMLFileParseInfo(KMLFileParseInfo.KMLFileType.MISSION);
                    parseInfo.setMissionType(Objects.requireNonNull(KMLValueConverter.getMissionType(type.getTextTrim())));
                    return parseInfo;
                }
            }
        }
        return new KMLFileParseInfo(KMLFileParseInfo.KMLFileType.ORIGINAL);
    }

    /**
     * 导入KML文件
     */
    public static Object importMission(MissionImportParams params, String path) throws KMLException, DocumentException {
        int suffixIndex = path.lastIndexOf('.');
        if (suffixIndex < 0) {
            throw new KMLException(KMLConstants.NON_KML_FILE);
        }

        SAXReader saxReader = new SAXReader();
        File f = new File(path);
        if (KMLValueConverter.equals(path.substring(path.lastIndexOf('.')), KMLConstants.ZIP_SUFFIX)) {
            File tempUnzipPath = new File(f.getParent(), KMLConstants.TEMP_FILE_DIR);
            if (tempUnzipPath.exists()) {
                FileUtils.delFile(tempUnzipPath);
            }
            tempUnzipPath.mkdirs();

            KMLZipHelper.unZipFolder(context, path, tempUnzipPath.getAbsolutePath(), true);
            File[] files = tempUnzipPath.listFiles();
            for (File file : files) {
                if (file.getName().endsWith(KMLConstants.SUFFIX)) {
                    f = file;
                    path = f.getAbsolutePath();
                } else if (file.getName().endsWith("jpg")) {
                    String dir= FileUtils.getDJIDirPath(context) + "/CACHE_PATH"; //CACHE_PATH
                    File newFile = new File(dir, file.getName());
                    try {
                        if (newFile.exists()) {
                            newFile.createNewFile();
                        }
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                    FileUtils.moveFile(file, newFile);
                }
            }
        }

        if (KMLValueConverter.equals(path.substring(path.lastIndexOf('.')), KMLConstants.SUFFIX)) {
            String name = f.getName().substring(0, f.getName().lastIndexOf("."));
            name = name.replaceAll(ILLEGAL_EX, "");
            MissionType type = params.getMissionType();
            if (type == MissionType.Waypoint) {
                return parseWaypointKML(saxReader.read(f), name, params.getHeightMode());
            } else if (type == MissionType.Strip) {
               return parseStripKML(saxReader.read(f), name);
            } else {
                return parseMappingKML(saxReader.read(f), name, type);
            }
        } else {
            throw new KMLException(KMLConstants.NON_KML_FILE);
        }

    }


    public static void saveKMLImportMission(Object missionModel, String currentPath){
        MissionGreenDaoTransform waylineConverter = new MissionGreenDaoTransform();
        MissionTransformData transformData = waylineConverter.generateGreenDaoMissionWayline(missionModel);
//        WaypointMissionModel transModel = new WaypointTemplateTransform().transTo(transformData);
//        transModel.setTransformData(transformData);
//        MissionInfoModel model = ((WaypointMissionModelGreenDao) missionModel).getMissionInfo().getMissionInfoModel();
//        transModel.setMissionInfo(model);



        List<WaylineTemplate> templateList = transformData.getTemplates();
        //executionConfig 参数
        WaylineExecuteMissionConfig executeMissionConfig =  transformData.getExecuteMissionConfig();
        //生成waylines
        PPalGenerator generator = new PPalGenerator();
        PPalGenerator.PPALParam param =
                new PPalGenerator.PPALParam(transformData.getMission()
                        , transformData.getMissionConfig(),
                        transformData.getExecuteMissionConfig(), templateList);
        List<Wayline> waylines = generator.getWaylines(param);
        //
        KMZPathWrapper pathWrapper;
        if (currentPath != null && !currentPath.isEmpty() && new File(currentPath).isDirectory()) {
            pathWrapper = KMZPathWrapper.Factory.create(currentPath, getName(currentPath));
            pathWrapper.clearOtherClipKmzFile();
        } else {
            String innerPath = "OutPath"; //UUID.randomUUID().toString().replace("-", "");
            String dir= FileUtils.getDJIDirPath(context) + "/KMZ"; //PATH
            pathWrapper = KMZPathWrapper.Factory.create(dir + "/" + innerPath, getName(currentPath));
        }
        //
        //生成kmz
        JNIWPMZManager.generateKMZFile(VERSION, transformData.getMission()
                , transformData.getMissionConfig(), templateList, waylines, executeMissionConfig,
                new ArrayList<>(), new ArrayList<>(), pathWrapper.getClipKmzFile().getPath());

    }


    public static String getName(String pathandname){

        int start=pathandname.lastIndexOf("/");
        int end=pathandname.lastIndexOf(".");
        if(start!=-1 && end!=-1){
            return pathandname.substring(start+1,end);
        }else{
            return "unknown";
        }

    }


    public static void saveLocalMission(Object missionModel , String kmlPath) {
        WaylineConverter waylineConverter = new WaylineConverter();
        if (missionModel instanceof MappingMissionModel) {
            //m todo
            MappingMissionModel mappingMissionModel = (MappingMissionModel) missionModel;
            MissionTransformData transformData = waylineConverter.generateKMLWayline(missionModel);
            mappingMissionModel.setTransformData(transformData);
            updateMappingMission(mappingMissionModel);

        } else if (missionModel instanceof StripMissionModel) {
            // m todo check
            StripMissionModel stripMissionModel = (StripMissionModel) missionModel;
            MissionTransformData transformData = waylineConverter.generateKMLWayline(missionModel);
            stripMissionModel.setTransformData(transformData);
            updateStripMission(stripMissionModel);

        } else {
            WaypointMissionModel waypointMissionModel = (WaypointMissionModel) missionModel;
            MissionTransformData transformData = waylineConverter.generateKMLWayline(missionModel);
            waypointMissionModel.setTransformData(transformData);
            updateWaypointMission(waypointMissionModel ,kmlPath);
        }


    }

    private static void updateWaypointMission(WaypointMissionModel waypointMissionModel , String currentPath){
        MissionTransformData transformData = waypointMissionModel.getTransformData();

        String localPath = generateWPMZFile(waypointMissionModel.getMissionInfo().getName(),
                currentPath,
                transformData,
                getWaypointPrecisePhotos(waypointMissionModel),
                new ArrayList<>() , new ArrayList<>());

        Log.d(TAG , "final path is " + localPath);

    }

    public static void  updateMappingMission(MappingMissionModel mappingMissionModel) {
        // saveMappingCamera(mappingMissionModel.getMappingWayline().getCamera()); todo check

        List<String> dsmFiles = new ArrayList<>();
        if (mappingMissionModel.getMappingWayline().getDsmPath() != null) {
            dsmFiles.addAll(mappingMissionModel.getMappingWayline().getDsmPath());
        }

        MissionTransformData transformData = mappingMissionModel.getTransformData();
        String currentPath = mappingMissionModel.getMissionInfo().getLocalWPMZPath();
        String localPath = generateWPMZFile(mappingMissionModel.getMissionInfo().getName(),
                currentPath,
                transformData,
                new ArrayList<>(),
                dsmFiles , new ArrayList<>());

        Log.d(TAG , "final path is " + localPath);
    }

    public static void  updateStripMission(StripMissionModel stripMissionModel) {
        // saveMappingCamera(stripMissionModel.getStripWayline().getCamera()); todo check

        MissionTransformData templateData = stripMissionModel.getTransformData();
        List<String> dsmFiles = new ArrayList<>();
        if (stripMissionModel.getStripWayline().getStripDsmPath() != null) {
            dsmFiles.addAll(stripMissionModel.getStripWayline().getStripDsmPath());
        }

        String currentPath = stripMissionModel.getMissionInfo().getLocalWPMZPath();
        String localPath =generateWPMZFile(
                stripMissionModel.getMissionInfo().getName(),
                currentPath,
                templateData,
                new ArrayList<>(),
                dsmFiles,
                new ArrayList<>()
        );
        Log.d(TAG , "final path is " + localPath);
    }



    private static List<String> getWaypointPrecisePhotos(WaypointMissionModel model) {
        List<WaypointModel> waypoints = model.getWaypoints();
        List<String> images = new ArrayList<>();

        for (WaypointModel item : waypoints) {
            List<WaypointActionModel> actions = item.getActions();
            if (actions != null) {
                for (WaypointActionModel actionItem : actions) {
                    PreciseShotInfo preciseInfo = actionItem.getPreciseShotInfo();
                    if (preciseInfo != null && preciseInfo.getCacheFilePath() != null) {
                        images.add(new File(preciseInfo.getCacheFilePath(), preciseInfo.getCacheFileName()).getAbsolutePath());
                    }
                }
            }
        }
        return images;
    }

    public static String generateWPMZFile(String name, String currentPath, MissionTransformData data,
                                          List<String> precisePhotoPath, List<String> dsmPath , List<String> audioFilePath) {
        KMZPathWrapper pathWrapper;
        if (currentPath != null && !currentPath.isEmpty() && new File(currentPath).isDirectory()) {
            pathWrapper = KMZPathWrapper.Factory.create(currentPath, name);
            pathWrapper.clearOtherClipKmzFile();
        } else {
            String innerPath = "OutPath"; //UUID.randomUUID().toString().replace("-", "");
            String dir= FileUtils.getDJIDirPath(context) + "/KMZ"; //PATH
            pathWrapper = KMZPathWrapper.Factory.create(dir + "/" + innerPath, name);
        }
        pathWrapper.refreshRes(precisePhotoPath, dsmPath , audioFilePath);

        JNIWPMZManager.generateKMZFile(VERSION, data.getMission(), data.getMissionConfig(), data.getTemplates(), data.getWayline(), data.getExecuteMissionConfig(),
                precisePhotoPath, dsmPath, currentPath);

        return pathWrapper.getDirPath();
    }




    public static WaypointMissionModelGreenDao parseWaypointKML(Document kml, String name, MissionImportHeightMode heightMode) throws KMLException {
        Element document = kml.getRootElement().element(KMLConstants.DOCUMENT);
        if (document == null || getWaylinePointsElement(document) == null) {
            throw new KMLException(KMLConstants.MISSION_TYPE_ILLEGAL);
        }
        if (name.length() > KMLConstants.MAX_MISSION_NAME_LENGTH) {
            name = name.substring(0, KMLConstants.MAX_MISSION_NAME_LENGTH);
        }
        // mission data to be set
        WaypointMissionModelGreenDao waypointMissionModel = KMLWaypointEntityCreateUtilKt.createWaypointMissionModel(name);
        MissionInfoModelGreenDao missionInfoModel = waypointMissionModel.getMissionInfo();
        missionInfoModel.getExtInfo().setCreateByKml(true);
        List<WaypointModelGreenDao> waypointModels = waypointMissionModel.getWaypoints();
        WaylineModelGreenDao waylineModel = waypointMissionModel.getWayline();

        List<Element> docElements = document.elements();
        for (Element element : docElements) {
            // MissionInfo 的扩展数据
            switch (element.getName()) {
                case KMLConstants.EXTENDED_DATA:
                    Element type = element.element(KMLConstants.MISSION_TYPE);
                    if (type != null && KMLValueConverter.getMissionType(type.getTextTrim()) != MissionType.Waypoint) {
                        throw new KMLException(KMLConstants.MISSION_TYPE_ILLEGAL);
                    }

                    parseMissionInfoExtendedData(element, missionInfoModel);
                    break;
                case KMLConstants.PLACE_MARK:
                    Element placeMarkDesc = element.element(KMLConstants.DESCRIPTION);
                    if (placeMarkDesc != null
                            && !TextUtils.isEmpty(placeMarkDesc.getTextTrim())
                            && placeMarkDesc.getTextTrim().equals(KMLConstants.POI_DESC)) {
                        // parse poi
                        parsePoiData(element, waylineModel);
                    }
                    break;
                case KMLConstants.FOLDER:
                    parsePointsData(element, waypointModels);
                    break;
                default:
            }
        }
        Element waylineEle = getWaylinePointsElement(document);
        parseWaylineData(waylineEle, waypointMissionModel, heightMode);
        // 若没有POI，则需要重新判断一下headingMode
        if (!DJIGpsUtils.isAvailable(waylineModel.getPoiLatitude(), waylineModel.getPoiLongitude())
                && waylineModel.getHeadingMode() == WaypointMissionHeadingMode.TOWARD_POINT_OF_INTEREST) {
            waylineModel.setHeadingMode(WaypointMissionHeadingMode.AUTO);
        }

        injectHeightMode(heightMode, waypointMissionModel);
        return waypointMissionModel;
    }

    /**
     * 设置用户选择的高度模式
     */
    private static void injectHeightMode(MissionImportHeightMode heightMode, WaypointMissionModelGreenDao waypointMissionModel) {
        if (heightMode != null) {
            waypointMissionModel.getWayline().getDroneInfo().getDroneHeight().setUseAbsolute(isAbsoluteHeight(heightMode));
        }
    }

    private static boolean isAbsoluteHeight(MissionImportHeightMode heightMode) {
        switch (heightMode) {
            case WGS84:
            case EGM96:
                return true;
            case RELATIVE:
            default:
                return false;
        }
    }

    private static Element getWaylinePointsElement(Element element) {
        if (element == null) {
            return null;
        }
        for (Element placeMarker : element.elements()) {
            if (placeMarker.getName().equals(KMLConstants.PLACE_MARK)) {
                Element lineEle = placeMarker.element(KMLConstants.LINE_STRING);
                if (lineEle != null) {
                    return placeMarker;
                }
            }
        }
        return null;
    }


    private static StripMissionModelGreenDao parseStripKML(Document kml, String name) throws KMLException {
        Element document = kml.getRootElement().element(KMLConstants.DOCUMENT);
        if (document == null) {
            throw new KMLException(KMLConstants.PARAM_ILLEGAL);
        }
        if (name.length() > KMLConstants.MAX_MISSION_NAME_LENGTH) {
            name = name.substring(0, KMLConstants.MAX_MISSION_NAME_LENGTH);
        }

        MappingCameraModelGreenDao camera = createMappingCameraModel(MappingCameraType.EP600_35MM);
        StripMissionModelGreenDao stripMissionModel = KMLStripGreenDaoCreateUtilKt.createStripModel(name, MissionType.Strip, camera);
        MissionInfoExtModel extraInfo = new MissionInfoExtModel();
        extraInfo.setCreateByKml(true);
        stripMissionModel.getMissionInfo().setExtInfo(extraInfo);

        List<Element> docElement = document.elements();
        for (Element element : docElement) {
            // Mission Info
            // 其它的扩展字段在此处添加解析

            if (element.getName().equals(KMLConstants.PLACE_MARK)) {
                // parse Field
                parseFieldData(element, stripMissionModel);
                // parse Points
                parseFieldPointsData(element, stripMissionModel);
            } else if (element.getName().equals(KMLConstants.EXTENDED_DATA)) {
                Element type = element.element(KMLConstants.MISSION_TYPE);
                if (type != null && KMLValueConverter.getMissionType(type.getTextTrim()) != MissionType.Strip) {
                    throw new KMLException(KMLConstants.MISSION_TYPE_ILLEGAL);
                }
            }
        }
        return stripMissionModel;
    }

    private static MappingMissionModelGreenDao parseMappingKML(Document kml, String name, MissionType missionType) throws KMLException {
        Element document = kml.getRootElement().element(KMLConstants.DOCUMENT);
        if (document == null) {
            throw new KMLException(KMLConstants.PARAM_ILLEGAL);
        }
        if (name.length() > KMLConstants.MAX_MISSION_NAME_LENGTH) {
            name = name.substring(0, KMLConstants.MAX_MISSION_NAME_LENGTH);
        }
        MappingCameraModelGreenDao camera = createMappingCameraModel(MappingCameraType.EP600_35MM);
        MappingMissionModelGreenDao mappingMissionModel = KMLMappingGreenDaoCreateUtilKt.createMappingModel(name, missionType, camera);
        MissionInfoExtModel extraInfo = new MissionInfoExtModel();
        extraInfo.setCreateByKml(true);
        mappingMissionModel.getMissionInfo().setExtInfo(extraInfo);
        List<Element> docElement = document.elements();
        for (Element element : docElement) {
            // Mission Info
            // 其它的扩展字段在此处添加解析

            if (element.getName().equals(KMLConstants.PLACE_MARK)) {
                // parse Field
                parseFieldData(element, mappingMissionModel);
                // parse Points
                parseFieldPointsData(element, mappingMissionModel);
            } else if (element.getName().equals(KMLConstants.EXTENDED_DATA)) {
                Element type = element.element(KMLConstants.MISSION_TYPE);
                if (type != null && KMLValueConverter.getMissionType(type.getTextTrim()) != missionType) {
                    throw new KMLException(KMLConstants.MISSION_TYPE_ILLEGAL);
                }
            }
        }
        return mappingMissionModel;
    }

    /**
     * 解析KML中 MissionInfo 扩展数据， 暂时只有Type字段，可扩展
     *
     * @param element   扩展数据字段，getName 应等于 ExtendedData
     * @param infoModel 航线信息数据
     */
    private static void parseMissionInfoExtendedData(Element element, MissionInfoModelGreenDao infoModel) {
        if (element == null) {
            return;
        }
        for (Element extendedData : element.elements()) {
            if (extendedData.getName().equals(KMLConstants.MISSION_TYPE)) {
                infoModel.setType(KMLValueConverter.getMissionType(extendedData.getTextTrim()));
            } else if (KMLConstants.STATION_TYPE.equals(extendedData.getName())) {
                String value = extendedData.getTextTrim();
                int v = NumberUtils.parseInt(value, RTKReferenceStationSource.NONE.value());
                infoModel.setRtkStation(RTKReferenceStationSource.find(v));
            }
            // 若有别的扩展数据，在这里加判断添加
        }
    }

    /**
     * 解析KML中 Wayline 扩展数据, 航线只需解析拿到扩展数据即可，坐标依照航点的
     *
     * @param element              扩展数据字段，getName 应等于 ExtendedData
     * @param waypointMissionModel missionModel数据
     * @param heightMode
     */
    private static void parseWaylineData(Element element, WaypointMissionModelGreenDao waypointMissionModel, MissionImportHeightMode heightMode) throws KMLException {
        WaylineModelGreenDao waylineModel = waypointMissionModel.getWayline();
        Element extendedEle = element.element(KMLConstants.EXTENDED_DATA);
        boolean hasWaylineAltitude = false;
        boolean isEgmHeight = heightMode == MissionImportHeightMode.EGM96;
        if (extendedEle != null && waylineModel != null) {
            for (Element extendedData : extendedEle.elements()) {
                switch (extendedData.getName()) {
                    case KMLConstants.ALTITUDE:
                        try {
                            float altitude = KMLValueConverter.string2Float(extendedData.getTextTrim());
                            if (altitude >= SettingUtils.MIN_WAYPOINT_HEIGHT && altitude <= SettingUtils.MAX_WAYPOINT_HEIGHT) {
                                waylineModel.setAltitude(altitude);
                            }
                        } catch (NumberFormatException e) {
                            // 若高度不合法则使用默认高度
                            Log.e(TAG, e.getMessage());
                        }
                        hasWaylineAltitude = true;
                        break;
                    case KMLConstants.AUTO_FLIGHT_SPEED:
                        try {
                            float speed = KMLValueConverter.string2Float(extendedData.getTextTrim());
                            if (speed >= 1 && speed <= MAX_SPEED) {
                                waylineModel.setAutoFlightSpeed(speed);
                            }
                        } catch (NumberFormatException e) {
                            // 若速度不合法则使用默认速度
                            Log.e(TAG, e.getMessage());
                        }
                        break;
                    case KMLConstants.ACTION_ON_FINISH:
                        waylineModel.setActionOnFinish(KMLValueConverter.getActionOnFinish(extendedData.getTextTrim()));
                        break;
                    case KMLConstants.HEADING_MODE:
                        waylineModel.setHeadingMode(KMLValueConverter.getHeadingMode(extendedData.getTextTrim()));
                        break;
                    case KMLConstants.GIMBAL_PITCH_MODE:
                        waylineModel.setGimbalPitchRotationEnable(KMLValueConverter.getGimbalPitchMode(extendedData.getTextTrim()));
                        break;
                    case KMLConstants.WAYLINE_WAYPOINT_TYPE:
                        waylineModel.setWaypointType(KMLValueConverter.getWaypointType(extendedData.getTextTrim()));
                        break;
                    case KMLConstants.POWER_SAVE_MODE:
                        waylineModel.setGotoFirstPointMode(Boolean.parseBoolean(extendedData.getTextTrim()) ?
                                WaypointMissionGotoWaypointMode.GotoWaypointPointToPoint : WaypointMissionGotoWaypointMode.GotoWaypointSafely);
                        break;
                    case KMLConstants.FLIGHT_CALI:
                        waylineModel.setFlightCali(Boolean.parseBoolean(extendedData.getTextTrim()));
                        break;
                    case KMLConstants.DRONE_INFO:
                        parseDroneInfo(extendedData, waylineModel.getDroneInfo());
                        break;
                    // 其他的扩展数据若作处理在这里添加
                    default:
                        break;
                }
            }
        }
        List<WaypointModelGreenDao> points = waypointMissionModel.getWaypoints();
        // 从线集解析
        Element lineStringEle = element.element(KMLConstants.LINE_STRING);
        if (lineStringEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }
        Element coordinatesEle = lineStringEle.element(KMLConstants.COORDINATES);
        if (coordinatesEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }
        boolean hasPoint = !points.isEmpty();

        String[] locations = coordinatesEle.getTextTrim().split(" ");
        try {
            for (int i = 0; i < locations.length; i++) {
                WaypointModelGreenDao point;
                if (hasPoint) {
                    point = points.get(i);
                } else {
                    point = KMLWaypointEntityCreateUtilKt.createWaypointModel();
                }
                String[] pointCoor = locations[i].split(",");
                double lng = KMLValueConverter.string2Double(pointCoor[0].trim());
                double lat = KMLValueConverter.string2Double(pointCoor[1].trim());
                float alt = KMLValueConverter.string2Float(pointCoor[2].trim());
                if (DJIGpsUtils.isAvailable(lat, lng)) {
                    point.setLatitude(lat);
                    point.setLongitude(lng);
                    if (alt >= SettingUtils.MIN_WAYPOINT_HEIGHT && alt <= SettingUtils.MAX_WAYPOINT_HEIGHT) {
                        if (isEgmHeight) {
                            alt = (float) GpsUtils.wgs84Altitude(alt, lat, lng);
                        }
                        point.setAltitude(alt);
                        // 如果是原始 kml 数据，并且航点没有写高度值，设置为跟随航线
                        if (!hasWaylineAltitude) {
                            point.setUseWaylineAltitude(alt == 0);
                        }
                    } else {
                        point.setAltitude(SettingUtils.DEF_WAYLINE_HEIGHT);
                    }
                }
                if (!hasPoint) {
                    points.add(point);
                }
            }
            waypointMissionModel.setWaypoints(points);
        } catch (NumberFormatException e) {
            Log.e(TAG, KMLConstants.LOCATION_ILLEGAL);
        }
        MissionInfoModelGreenDao infoModel = waypointMissionModel.getMissionInfo();
        if (CollectionUtil.isNotEmpty(points) && infoModel != null) {
            double latitude = points.get(0).getLatitude();
            infoModel.setLatitude(latitude);
            double longitude = points.get(0).getLongitude();
            infoModel.setLongitude(longitude);
            // 设置航点航线高度，为兼容 V3 版本，此高度后续直接用作椭球高，需要转换成椭球高
            // WaylineConverter.generateWaypointKMLWayline
            if (isEgmHeight) {
                waylineModel.setAltitude((float) GpsUtils.wgs84Altitude(waylineModel.getAltitude(), latitude, longitude));
            }
        }
    }

    private static void parseDroneInfo(Element element, DroneInfoModel droneInfoModel) {
        for (Element itemElement : element.elements()) {
            switch (itemElement.getName()) {
                case KMLConstants.DRONE_TYPE:
                    droneInfoModel.setDroneType(KMLValueConverter.getDroneType(itemElement.getTextTrim()));
                    break;
                case KMLConstants.DRONE_CAMERAS:
                    droneInfoModel.setDroneCamera(parseDroneCameras(itemElement));
                    break;
                case KMLConstants.DRONE_HEIGHT_MODE:
                    droneInfoModel.setDroneHeight(parseDroneHeightMode(itemElement));
                    break;
                default:
                    break;
            }
        }
        parsePayloadInfos(element, droneInfoModel);
    }

    private static List<DroneCameraModel> parseDroneCameras(Element element) {
        List<DroneCameraModel> cameraModels = new ArrayList<>();
        for (Element itemElement : element.elements()) {
            if (itemElement.getName().equals(KMLConstants.DRONE_CAMERA_INFO)) {
                DroneCameraModel cameraModel = new DroneCameraModel();
                for (Element cameraElement : itemElement.elements()) {
                    switch (cameraElement.getName()) {
                        case KMLConstants.DRONE_CAMERA_INDEX:
                            cameraModel.setCameraIndex(KMLValueConverter.string2Int(cameraElement.getTextTrim()));
                            break;
                        case KMLConstants.DRONE_CAMERA_NAME:
                            cameraModel.setCameraName(cameraElement.getTextTrim());
                            break;
                        case KMLConstants.DRONE_CAMERA_TYPE:
                            cameraModel.setCameraType(KMLValueConverter.string2Int(cameraElement.getTextTrim()));
                            break;
                        case KMLConstants.DRONE_PAYLOAD_CAMERA_TYPE:
                            cameraModel.setPayloadCameraType(KMLValueConverter.string2Int(cameraElement.getTextTrim()));
                            break;
                        case KMLConstants.DRONE_PAYLOAD_CAMERA_CONFIG:
                            cameraModel.setPayloadConfigInfo(parsePayloadCameraConfig(cameraElement));
                            break;
                        default:
                            break;
                    }
                }
                cameraModels.add(cameraModel);
            }
        }
        return cameraModels;
    }

    private static void setPayloadCameraInfo(DroneInfoModel droneInfoModel, PayloadInfoModel payloadInfoModel) {
        List<DroneCameraModel> droneCameras = droneInfoModel.getCameras();
        for (DroneCameraModel item : droneCameras) {
            if (item.isPSDKCamera() && item.getPayloadInfo() == null) {
                item.setPayloadInfo(payloadInfoModel);
                break;
            }
        }
    }

    private static void parsePayloadInfos(Element element, DroneInfoModel droneInfoModel) {
        for (Element itemElement : element.elements()) {
            switch (itemElement.getName()) {
                case KMLConstants.PAYLOAD_WIDGET:
                    PayloadInfoModel payloadInfoModel = parsePayloadInfo(itemElement);
                    setPayloadCameraInfo(droneInfoModel, payloadInfoModel);
                    break;
                default:
                    break;
            }
        }
    }

    private static PayloadInfoModel parsePayloadInfo(Element element) {
        PayloadInfoModel payloadInfo = new PayloadInfoModel();
        List<PayloadInfoModel.PayloadWidgetInfo> widgetInfos = new ArrayList<>();
        payloadInfo.setWidgetModels(widgetInfos);

        for (Element itemElement : element.elements()) {
            switch (itemElement.getName()) {
                case KMLConstants.PAYLOAD_NAME:
                    payloadInfo.setName(itemElement.getTextTrim());
                    break;
                case KMLConstants.PAYLOAD_WIDGET_INFO:
                    widgetInfos.add(parsePayloadWidget(itemElement));
                    break;
                default:
                    break;
            }
        }
        return payloadInfo;
    }

    private static PayloadInfoModel.PayloadWidgetInfo parsePayloadWidget(Element element) {
        PayloadInfoModel.PayloadWidgetInfo widgetInfo = new PayloadInfoModel.PayloadWidgetInfo();

        for (Element itemElement : element.elements()) {
            switch (itemElement.getName()) {
                case KMLConstants.PAYLOAD_WIDGET_NAME:
                    widgetInfo.setName(itemElement.getTextTrim());
                    break;
                case KMLConstants.PAYLOAD_WIDGET_INDEX:
                    widgetInfo.setIndex(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.PAYLOAD_WIDGET_TYPE:
                    widgetInfo.setType(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.PAYLOAD_WIDGET_VALUE:
                    widgetInfo.setValue(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.PAYLOAD_WIDGET_VALUE_MIN:
                    widgetInfo.setMinValue(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.PAYLOAD_WIDGET_VLAUE_MAX:
                    widgetInfo.setMaxValue(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                default:
                    break;
            }
        }
        return widgetInfo;
    }

    private static PayloadConfigInfoModel parsePayloadCameraConfig(Element element) {
        PayloadConfigInfoModel infoModel = new PayloadConfigInfoModel();
        for (Element itemElement : element.elements()) {
            switch (itemElement.getName()) {
                case KMLConstants.DEWARPING:
                    infoModel.setEnableDewarp(Boolean.parseBoolean(itemElement.getTextTrim()));
                    break;
                case KMLConstants.METERING_MODE:
                    infoModel.setMeteringMode(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.LIDAR_ECHO_MODE:
                    infoModel.setEchoMode(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.LIDAR_NEED_VARIEGATION_MODE:
                    infoModel.setNeedVariegation(Boolean.parseBoolean(itemElement.getTextTrim()));
                    break;
                case KMLConstants.LIDAR_SCAN_MODE:
                    infoModel.setScanMode(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                case KMLConstants.LIDAR_SAMPLE_RATE_MODE:
                    infoModel.setSampleRate(KMLValueConverter.string2Int(itemElement.getTextTrim()));
                    break;
                default:
                    break;
            }
        }
        return infoModel;
    }

    private static DroneHeightModel parseDroneHeightMode(Element element) {
        DroneHeightModel droneHeightModel = new DroneHeightModel();
        for (Element itemElement : element.elements()) {
            switch (itemElement.getName()) {
                case KMLConstants.DRONE_HEIGHT_USE_ABSOLUTE:
                    droneHeightModel.setUseAbsolute(Boolean.parseBoolean(itemElement.getTextTrim()));
                    break;
                case KMLConstants.DRONE_HEIGHT_HAS_TAKEOFF_HEIGHT:
                    droneHeightModel.setHasTakeoffHeight(Boolean.parseBoolean(itemElement.getTextTrim()));
                    break;
                case KMLConstants.DRONE_HEIGHT_TAKEOFF_HEIGHT:
                    droneHeightModel.setTakeoffHeight(KMLValueConverter.string2Float(itemElement.getTextTrim()));
                    break;
            }
        }
        return droneHeightModel;
    }

    private static List<PreciseShotInfo> parsePreciseShotInfo(Element element) {
        List<PreciseShotInfo> shotInfos = new ArrayList<>();
        for (Element itemElement : element.elements()) {
            if (itemElement.getName().equals(KMLConstants.PRECISE_INFO)) {
                PreciseShotInfo shotInfo = new PreciseShotInfo();
                String dir = FileUtils.getDJIDirPath(context) + "/CACHE_PATH";
                shotInfo.setCacheFilePath(dir); //CACHE_PATH
                for (Element shotInfoElement : itemElement.elements()) {
                    switch (shotInfoElement.getName()) {
                        case KMLConstants.PRECISE_IMG_WIDTH:
                            shotInfo.setImageWidth(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_IMG_HEIGHT:
                            shotInfo.setImageHeight(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CROP_X:
                            shotInfo.setCropPosX(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CROP_Y:
                            shotInfo.setCropPosY(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CROP_ANGLE:
                            shotInfo.setCropAngle(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CROP_WIDTH:
                            shotInfo.setCropWidth(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CROP_HEIGHT:
                            shotInfo.setCropHeight(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_FOCAL_DISTANCE:
                            shotInfo.setFocalDistance(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()) / 10d);
                            break;
                        case KMLConstants.PRECISE_AF_POSITION:
                            shotInfo.setAfPosition(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_GIMBAL_PITCH:
                            shotInfo.setGimbalPitch(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_GIMBAL_ROLL:
                            shotInfo.setGimbalRoll(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_GIMBAL_YAW:
                            shotInfo.setGimbalYaw(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CAMERA_TYPE:
                            shotInfo.setCameraType(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_GAMERA_INDEX:
                            shotInfo.setCameraIndex(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_GIMBAL_PORT:
                            shotInfo.setGimbalPort(KMLValueConverter.string2Int(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_FILE_SIZE:
                            shotInfo.setFileSize(Long.parseLong(shotInfoElement.getTextTrim()));
                            break;
                        case KMLConstants.PRECISE_CACHE_FILE_NAME:
                            shotInfo.setCacheFileName(shotInfoElement.getTextTrim());
                            break;
                        case KMLConstants.PRECISE_DRONE_YAW:
                            shotInfo.setDroneYaw(KMLValueConverter.string2Float(shotInfoElement.getTextTrim()));
                            break;
                        default:
                            break;
                    }
                }
                shotInfos.add(shotInfo);
            }
        }
        return shotInfos;
    }

    /**
     * 解析KML中 POI 数据
     *
     * @param element      POI Point字段，包括coordinate
     * @param waylineModel 航线model，用于设置poi数据
     */
    private static void parsePoiData(Element element, WaylineModelGreenDao waylineModel) {
        if (element == null) {
            return;
        }
        Element poiPoint = element.element(KMLConstants.POINT);
        if (poiPoint != null) {
            Element poiCoor = poiPoint.element(KMLConstants.COORDINATES);
            if (poiCoor != null && !TextUtils.isEmpty(poiCoor.getTextTrim())) {
                String[] coordinates = poiCoor.getTextTrim().split(",");
                if (coordinates.length == 3 || coordinates.length == 2) {
                    try {
                        double longitude = KMLValueConverter.string2Double(coordinates[0]);
                        double latitude = KMLValueConverter.string2Double(coordinates[1]);
                        if (DJIGpsUtils.isAvailable(latitude, longitude)) {
                            waylineModel.setPoiLongitude(longitude);
                            waylineModel.setPoiLatitude(latitude);
                        }
                    } catch (NumberFormatException e) {
                        Log.e(TAG, e.getMessage());
                    }
                }
            }
        }
    }

    /**
     * 解析航点数据，从Folder节点开始解析
     *
     * @param element        getName应等于Folder，所有航点数据存在于folder的节点中
     * @param waypointModels 航点
     */
    private static void parsePointsData(Element element, List<WaypointModelGreenDao> waypointModels) {
        // 解析 FOLDER 中的航点
        for (Element placeMark : element.elements()) {
            if (!placeMark.getName().equals(KMLConstants.PLACE_MARK)) {
                continue;
            }
            Element pointDesc = placeMark.element(KMLConstants.DESCRIPTION);
            // 当描述存在并且是 Waypoint 才去获取扩展字段, 以及获取坐标
            if (pointDesc != null
                    && !TextUtils.isEmpty(pointDesc.getTextTrim())
                    && pointDesc.getTextTrim().equals(KMLConstants.WAYPOINT_DESC)) {
                // Create a Waypoint
                WaypointModelGreenDao waypointModel = KMLWaypointEntityCreateUtilKt.createWaypointModel();
                List<WaypointActionModelGreenDao> actionModels = waypointModel.getActions();

                // parse ExtendedData
                Element pointExtended = placeMark.element(KMLConstants.EXTENDED_DATA);
                if (pointExtended == null) {
                    continue;
                }
                for (Element extendedData : pointExtended.elements()) {
                    switch (extendedData.getName()) {
                        case KMLConstants.USE_WAYLINE_ALTITUDE:
                            waypointModel.setUseWaylineAltitude(Boolean.parseBoolean(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.HEADING:
                            try {
                                int heading = KMLValueConverter.string2Int(extendedData.getTextTrim());
                                if (heading >= MIN_HEADING && heading <= MAX_HEADING) {
                                    waypointModel.setHeading(heading);
                                }
                            } catch (NumberFormatException e) {
                                // 若输入不合法，则不设置，使用创建时的默认值
                                Log.e(TAG, e.getMessage());
                            }
                            break;
                        case KMLConstants.TURN_MODE:
                            waypointModel.setTurnMode(KMLValueConverter.getWaypointTurnMode(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.GIMBAL_PITCH:
                            try {
                                float gimbalPitch = KMLValueConverter.string2Float(extendedData.getTextTrim());
                                waypointModel.setGimbalPitch(gimbalPitch);
                            } catch (NumberFormatException e) {
                                // 若输入不合法，则不设置，使用创建时的默认值
                                Log.e(TAG, e.getMessage());
                            }
                            break;
                        case KMLConstants.USE_WAYLINE_SPEED:
                            waypointModel.setUseWaylineSpeed(Boolean.parseBoolean(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.SPEED:
                            waypointModel.setSpeed(KMLValueConverter.string2Float(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.USE_WAYLINE_HEADING_MODE:
                            waypointModel.setUseWaylineHeadMode(Boolean.parseBoolean(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.USE_WAYLINE_POINT_TYPE:
                            waypointModel.setUseWaylinePointType(Boolean.parseBoolean(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.WAYPOINT_TYPE:
                            waypointModel.setWaypointType(KMLValueConverter.getWaypointType(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.CORNER_RADIUS:
                            waypointModel.setCornerRadius(KMLValueConverter.string2Float(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.HEADING_MODE:
                            waypointModel.setHeadingMode(KMLValueConverter.getHeadingMode(extendedData.getTextTrim()));
                            break;
                        case KMLConstants.WAYPOINT_PRECISE_INFO:
                            waypointModel.setPreciseShotInfo(parsePreciseShotInfo(extendedData));
                            break;
                        case KMLConstants.ACTIONS:
                            Attribute param = extendedData.attribute(KMLConstants.PARAM);
                            Attribute accuracy = extendedData.attribute(KMLConstants.ACTION_ACCURACY);
                            Attribute cameraIndex = extendedData.attribute(KMLConstants.ACTION_CAMERA_INDEX);
                            Attribute payloadType = extendedData.attribute(KMLConstants.ACTION_PAYLOAD_TYPE);
                            Attribute payloadIndex = extendedData.attribute(KMLConstants.ACTION_PAYLOAD_INDEX);
                            Attribute preciseInfoName = extendedData.attribute(KMLConstants.ACTION_PRECISE_INFO_NAME);
                            Attribute cameraPathName = extendedData.attribute(KMLConstants.ACTION_MEDIA_PATH_NAME);
                            WaypointActionType actionType = KMLValueConverter.getActionType(extendedData.getTextTrim());

                            WaypointActionModelGreenDao actionModel = new WaypointActionModelGreenDao();

                            if (actionType != null) {
                                if (accuracy != null) {
                                    actionModel.setAccuracy(KMLValueConverter.string2Int(accuracy.getValue()));
                                }
                                if (cameraIndex != null) {
                                    actionModel.setCameraIndex(KMLValueConverter.string2Int(cameraIndex.getValue()));
                                }
                                boolean isValid = true;
                                if (param != null) {
                                    int value = 0;
                                    try {
                                        value = KMLValueConverter.string2Int(param.getValue());
                                    } catch (NumberFormatException ex) {
                                        isValid = false;
                                    }
                                    if (isValid) {
                                        actionModel.setParam(value);

                                        if (actionType == WaypointActionType.STAY) {
                                            isValid = (value >= 0 && value <= 30 * 1000);

                                        } else if (actionType == WaypointActionType.ROTATE_AIRCRAFT) {
                                            isValid = (value >= MIN_HEADING && value <= MAX_HEADING);

                                        } else if (actionType == WaypointActionType.GIMBAL_PITCH) {
                                            isValid = (actionModel.getRealParam() >= -120 && actionModel.getRealParam() <= 120);
                                        }
                                    }
                                }

                                if (payloadIndex != null) {
                                    actionModel.setPayloadIndex(KMLValueConverter.string2Int(payloadIndex.getValue()));
                                }
                                if (payloadType != null) {
                                    actionModel.setPayloadType(KMLValueConverter.string2Int(payloadType.getValue()));
                                }
                                if (preciseInfoName != null) {
                                    actionModel.setPreciseInfoName(preciseInfoName.getValue());
                                }
                                if (cameraPathName != null) {
                                    actionModel.setMediaPathName(cameraPathName.getValue());
                                }

                                if (isValid) {
                                    actionModel.setActionType(actionType);
                                    actionModels.add(actionModel);
                                }
                            }
                            break;
                        default:
                            break;
                    }
                }
                waypointModels.add(waypointModel);
            }
        }
    }


    /**
     * parse FieldData, including wayline model & edge points
     */
    private static void parseFieldData(Element placemark, StripMissionModelGreenDao model) {
        if (placemark == null) {
            return;
        }
        Element extendedDataEle = placemark.element(KMLConstants.EXTENDED_DATA);
        if (extendedDataEle != null) {
            StripWaylineModelGreenDao waylineModel = model.getStripWayline();
            Element altEle = extendedDataEle.element(KMLConstants.ALTITUDE);
            if (altEle != null) {
                try {
                    int alt = KMLValueConverter.string2Int(altEle.getTextTrim());
                    if (alt >= MappingUtils.MIN_ALTITUDE && alt <= MappingUtils.MAX_ALTITUDE) {
                        waylineModel.setAltitude(alt);
                    }
                } catch (NumberFormatException e) {
                    Log.e(TAG, KMLConstants.ALTITUDE_PARAM_ILLEGAL);
                }
            }
            // Camera
            parseCameraData(extendedDataEle, waylineModel);
            // 2D
            parseStripBasicData(extendedDataEle, waylineModel);
        }
    }


    /**
     * parse FieldData, including wayline model & edge points
     */
    private static void parseFieldData(Element placemark, MappingMissionModelGreenDao model) {
        if (placemark == null) {
            return;
        }
        Element extendedDataEle = placemark.element(KMLConstants.EXTENDED_DATA);
        if (extendedDataEle != null) {
            MappingWaylineModelGreenDao waylineModel = model.getMappingWayline();
            // Altitude 高度决定了Margin的范围，先要去取高度值
            Element altEle = extendedDataEle.element(KMLConstants.ALTITUDE);
            if (altEle != null) {
                try {
                    int alt = KMLValueConverter.string2Int(altEle.getTextTrim());
                    if (alt >= MappingUtils.MIN_ALTITUDE && alt <= MappingUtils.MAX_ALTITUDE) {
                        waylineModel.setAltitude(alt);
                    }
                } catch (NumberFormatException e) {
                    Log.e(TAG, KMLConstants.ALTITUDE_PARAM_ILLEGAL);
                }
            }
            // Camera
            parseCameraData(extendedDataEle, waylineModel);
            // 3D
            if (waylineModel.getType() == MissionType.Mapping3D) {
                parse3DData(extendedDataEle, waylineModel);
            }
            // 2D
            parseMappingBasicData(extendedDataEle, waylineModel);
        }
    }

    /**
     * 解析航带边界点数据， 默认最后一个点不计算
     */
    private static void parseFieldPointsData(Element element, StripMissionModelGreenDao model) throws KMLException {
        // 从线集解析
        Element lineStringEle = element.element(KMLConstants.LINE_STRING);
        if (lineStringEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }
        Element coordinatesEle = lineStringEle.element(KMLConstants.COORDINATES);
        if (coordinatesEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }

        List<EdgePointModelGreenDao> edgePoints = new ArrayList<>();

        String[] locations = coordinatesEle.getTextTrim().split(" ");
        try {
            for (int i = 0; i < locations.length; i++) {
                EdgePointModelGreenDao point = new EdgePointModelGreenDao();
                String[] pointCoor = locations[i].split(",");
                double lng = KMLValueConverter.string2Double(pointCoor[0].trim());
                double lat = KMLValueConverter.string2Double(pointCoor[1].trim());
                if (DJIGpsUtils.isAvailable(lat, lng)) {
                    point.setLatitude(lat);
                    point.setLongitude(lng);
                    edgePoints.add(point);
                }
            }
            model.setEdgePoints(edgePoints);
        } catch (NumberFormatException e) {
            Log.e(TAG, KMLConstants.LOCATION_ILLEGAL);
        }
        MissionInfoModelGreenDao infoModel = model.getMissionInfo();
        if (CollectionUtil.isNotEmpty(edgePoints) && infoModel != null) {
            infoModel.setLatitude(edgePoints.get(0).getLatitude());
            infoModel.setLongitude(edgePoints.get(0).getLongitude());
        }
    }

    /**
     * 解析边界点数据， 默认最后一个点不计算
     */
    private static void parseFieldPointsData(Element element, MappingMissionModelGreenDao model) throws KMLException {
        Element polygonEle = element.element(KMLConstants.POLYGON);
        if (polygonEle == null) {
            throw new KMLException(KMLConstants.MISSION_TYPE_ILLEGAL);
        }
        Element outerBoundaryEle = polygonEle.element(KMLConstants.OUTER_BOUNDARY_IS);
        if (outerBoundaryEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }
        Element linearRingEle = outerBoundaryEle.element(KMLConstants.LINEAR_RING);
        if (linearRingEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }
        Element coordinatesEle = linearRingEle.element(KMLConstants.COORDINATES);
        if (coordinatesEle == null) {
            throw new KMLException(KMLConstants.LOCATION_ILLEGAL);
        }
        List<EdgePointModelGreenDao> points = new ArrayList<>();

        String[] locations = coordinatesEle.getTextTrim().split(" ");
        try {
            for (int i = 0; i < locations.length - 1; i++) {
                EdgePointModelGreenDao point = new EdgePointModelGreenDao();
                String[] pointCoor = locations[i].split(",");
                double lng = KMLValueConverter.string2Double(pointCoor[0].trim());
                double lat = KMLValueConverter.string2Double(pointCoor[1].trim());
                if (DJIGpsUtils.isAvailable(lat, lng)) {
                    point.setLatitude(lat);
                    point.setLongitude(lng);
                    points.add(point);
                }
            }
            model.setEdgePoints(points);
        } catch (NumberFormatException e) {
            Log.e(TAG, KMLConstants.EDGE_POINTS_LOCATION_ILLEGAL);
        }
        MissionInfoModelGreenDao infoModel = model.getMissionInfo();
        if (CollectionUtil.isNotEmpty(points) && infoModel != null) {
            infoModel.setLatitude(points.get(0).getLatitude());
            infoModel.setLongitude(points.get(0).getLongitude());
        }
    }

    /**
     * 解析航带基础设置
     */
    private static void parseStripBasicData(Element element, StripWaylineModelGreenDao waylineModel) {
        for (Element ele : element.elements()) {
            switch (ele.getName()) {
                case KMLConstants.OVERLAP_H:
                    try {
                        int overlapH = KMLValueConverter.string2Int(ele.getTextTrim());
                        if (overlapH >= MappingUtils.MIN_OVERLAP && overlapH <= MappingUtils.MAX_OVERLAP) {
                            waylineModel.setOverlapH(overlapH);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.OVERLAP_H_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.OVERLAP_W:
                    try {
                        int overlapW = KMLValueConverter.string2Int(ele.getTextTrim());
                        if (overlapW >= MappingUtils.MIN_OVERLAP && overlapW <= MappingUtils.MAX_OVERLAP) {
                            waylineModel.setOverlapW(overlapW);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.OVERLAP_W_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.AUTO_FLIGHT_SPEED:
                    try {
                        float speed = KMLValueConverter.string2Float(ele.getTextTrim());
                        if (speed >= MappingUtils.MIN_SPEED && speed <= MappingUtils.MAX_SPEED) {
                            waylineModel.setSpeed(speed);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.AUTO_FLIGHT_SPEED_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.TAKEOFF_SPEED:
                    try {
                        float speed = KMLValueConverter.string2Float(ele.getTextTrim());
                        if (speed >= MappingUtils.MIN_SPEED && speed <= MappingUtils.MAX_SPEED) {
                            waylineModel.setTakeoffSpeed(speed);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.AUTO_FLIGHT_SPEED_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.ACTION_ON_FINISH:
                    waylineModel.setActionOnFinish(KMLValueConverter.getActionOnFinish(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ACTION_ON_FINISH + ele.getTextTrim());
                    break;
                case KMLConstants.PHOTO_MODE:
                    try {
                        int photoMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setPhotoMode(photoMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.PHOTO_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.LEFT_EXTAND_DISTANCE:
                    try {
                        int leftExtandDistance = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setLeftExtend(leftExtandDistance);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LEFT_EXTAND_DISTANCE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.RIGHT_EXTAND_DISTANCE:
                    try {
                        int rightExtandDistance = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setRightExtend(rightExtandDistance);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.RIGHT_EXTAND_DISTANCE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.CUT_DISTANCE:
                    try {
                        int cutDistance = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setCutLength(cutDistance);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.CUT_DISTANCE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.INCLUDE_CENTER_LINE:
                    try {
                        int includeCenterLine = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setHasCenterLine(includeCenterLine == 1);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.INCLUDE_CENTER_LINE + ele.getTextTrim());
                    }

                    break;
                case KMLConstants.PLAN_MODE:
                    try {
                        int planMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setPlanMode(planMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.PLAN_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.DEWARPING:
                    waylineModel.setEnableDewarping(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.DEWARPING + ele.getTextTrim());
                    break;
                case KMLConstants.ENABLE_SINGLE_LINE:
                    waylineModel.setEnableSingleLine(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ENABLE_SINGLE_LINE + ele.getTextTrim());
                    break;
                case KMLConstants.FOCUS_MODE:
                    try {
                        int focusMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setFocusMode(focusMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.FOCUS_MODE + ele.getTextTrim());
                    }
                    break;
                 case KMLConstants.LIDAR_ECHO_MODE:
                    try {
                        int echoMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setEchoMode(echoMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LIDAR_ECHO_MODE + ele.getTextTrim());
                    }
                    break;
                 case KMLConstants.LIDAR_SAMPLE_RATE_MODE:
                    try {
                        int sampleRateMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setSampleRate(sampleRateMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LIDAR_SAMPLE_RATE_MODE + ele.getTextTrim());
                    }
                    break;
                 case KMLConstants.LIDAR_SCAN_MODE:
                    try {
                        int scanMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setScanMode(scanMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LIDAR_SCAN_MODE + ele.getTextTrim());
                    }
                    break;
                 case KMLConstants.LIDAR_NEED_VARIEGATION_MODE:
                     waylineModel.setNeedVariegation(Boolean.parseBoolean(ele.getTextTrim()));
                     Log.e(TAG, KMLConstants.LIDAR_NEED_VARIEGATION_MODE + ele.getTextTrim());
                    break;
                case KMLConstants.MAPPING_ALTITUDE_MODE:
                    try {
                        int altitudeMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setAltitudeMode(altitudeMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.MAPPING_ALTITUDE_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.RELATIVE_DISTANCE:
                    try {
                        int relativeDistance = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setRelativeDistance(relativeDistance);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.RELATIVE_DISTANCE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.ENABLE_CALIBRATE:
                    waylineModel.setEnableCalibrate(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ENABLE_CALIBRATE + ele.getTextTrim());
                    break;
                default:
                    break;
            }
        }
    }


    /**
     * 解析测绘基础设置项 (2D)
     */
    private static void parseMappingBasicData(Element element, MappingWaylineModelGreenDao waylineModel) {
        for (Element ele : element.elements()) {
            switch (ele.getName()) {
                case KMLConstants.DIRECTION:
                    try {
                        int direction = KMLValueConverter.string2Int(ele.getTextTrim());
                        if (direction >= MappingUtils.MIN_DIRECTION && direction <= MappingUtils.MAX_DIRECTION) {
                            waylineModel.setDirection(direction);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.DIRECTION_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.MARGIN:
                    try {
                        float margin = KMLValueConverter.string2Float(ele.getTextTrim());
                        float max = waylineModel.getAltitude() * MappingUtils.MARGIN_COE;
                        if (margin >= -max && margin <= max) {
                            waylineModel.setMargin(margin);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.MARGIN_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.OVERLAP_H:
                    try {
                        int overlapH = KMLValueConverter.string2Int(ele.getTextTrim());
                        if (overlapH >= MappingUtils.MIN_OVERLAP && overlapH <= MappingUtils.MAX_OVERLAP) {
                            waylineModel.setOverlapH(overlapH);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.OVERLAP_H_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.OVERLAP_W:
                    try {
                        int overlapW = KMLValueConverter.string2Int(ele.getTextTrim());
                        if (overlapW >= MappingUtils.MIN_OVERLAP && overlapW <= MappingUtils.MAX_OVERLAP) {
                            waylineModel.setOverlapW(overlapW);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.OVERLAP_W_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.AUTO_FLIGHT_SPEED:
                    try {
                        float speed = KMLValueConverter.string2Float(ele.getTextTrim());
                        if (speed >= MappingUtils.MIN_SPEED && speed <= MappingUtils.MAX_SPEED) {
                            waylineModel.setSpeed(speed);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.AUTO_FLIGHT_SPEED_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.TAKEOFF_SPEED:
                    try {
                        float speed = KMLValueConverter.string2Float(ele.getTextTrim());
                        if (speed >= MappingUtils.MIN_SPEED && speed <= MappingUtils.MAX_SPEED) {
                            waylineModel.setTakeoffSpeed(speed);
                        }
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.AUTO_FLIGHT_SPEED_PARAM_ILLEGAL + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.ACTION_ON_FINISH:
                    waylineModel.setActionOnFinish(KMLValueConverter.getActionOnFinish(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ACTION_ON_FINISH + ele.getTextTrim());
                    break;
                case KMLConstants.ELEVATION_OPTI:
                    waylineModel.setElevationOptimize(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ELEVATION_OPTI + ele.getTextTrim());
                    break;
                case KMLConstants.ENABLE_FIVE_WAY_POSE:
                    waylineModel.setFiveWayPose(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ENABLE_FIVE_WAY_POSE + ele.getTextTrim());
                    break;
                case KMLConstants.FIVE_WAY_POSE_WITH_GIMBAL_PITCH:
                    try {
                        int fiveWayPoseWithGimbalPitch = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setFiveWayPoseWithGimbalPitch(fiveWayPoseWithGimbalPitch);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.FIVE_WAY_POSE_WITH_GIMBAL_PITCH + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.PHOTO_MODE:
                    try {
                        int photoMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setPhotoMode(photoMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.PHOTO_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.DEWARPING:
                    waylineModel.setEnableDewarping(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.DEWARPING + ele.getTextTrim());
                    break;
                case KMLConstants.FOCUS_MODE:
                    try {
                        int focusMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setFocusMode(focusMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.FOCUS_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.LIDAR_ECHO_MODE:
                    try {
                        int echoMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setEchoMode(echoMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LIDAR_ECHO_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.LIDAR_SAMPLE_RATE_MODE:
                    try {
                        int sampleRateMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setSampleRate(sampleRateMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LIDAR_SAMPLE_RATE_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.LIDAR_SCAN_MODE:
                    try {
                        int scanMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setScanMode(scanMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.LIDAR_SCAN_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.LIDAR_NEED_VARIEGATION_MODE:
                    waylineModel.setNeedVariegation(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.LIDAR_NEED_VARIEGATION_MODE + ele.getTextTrim());
                    break;
                case KMLConstants.MAPPING_ALTITUDE_MODE:
                    try {
                        int altitudeMode = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setAltitudeMode(altitudeMode);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.MAPPING_ALTITUDE_MODE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.RELATIVE_DISTANCE:
                    try {
                        int relativeDistance = KMLValueConverter.string2Int(ele.getTextTrim());
                        waylineModel.setRelativeDistance(relativeDistance);
                    } catch (NumberFormatException ex) {
                        Log.e(TAG, KMLConstants.RELATIVE_DISTANCE + ele.getTextTrim());
                    }
                    break;
                case KMLConstants.ENABLE_CALIBRATE:
                    waylineModel.setEnableCalibrate(Boolean.parseBoolean(ele.getTextTrim()));
                    Log.e(TAG, KMLConstants.ENABLE_CALIBRATE + ele.getTextTrim());
                    break;
            }
        }
    }

    /**
     * 解析相机相关参数
     */
    private static void parseCameraData(Element element, StripWaylineModelGreenDao waylineModel) {
        Element cameraEle = element.element(KMLConstants.CAMERA_TYPE);
        if (cameraEle != null
                && KMLValueConverter.isKnownCamera(cameraEle.getTextTrim())
                && MappingCameraType.find(cameraEle.getTextTrim()) != MappingCameraType.OTHER) {
            waylineModel.setCamera(createMappingCameraModel(MappingCameraType.find(cameraEle.getTextTrim())));
        } else {
            MappingCameraModelGreenDao cameraModel = new MappingCameraModelGreenDao();
            cameraModel.setName(MappingCameraType.OTHER.getNameStr());
            cameraModel.setSensorHeight(StripUtils.DEF_CAMERA_SENSOR_HEIGHT);
            cameraModel.setSensorWidth(StripUtils.DEF_CAMERA_SENSOR_WIDTH);
            cameraModel.setImageHeight(StripUtils.DEF_CAMERA_IMAGE_HEIGHT);
            cameraModel.setImageWidth(StripUtils.DEF_CAMERA_IMAGE_WIDTH);
            cameraModel.setFocalLength(StripUtils.DEF_CAMERA_FOCAL_LENGTH);
            cameraModel.setShotInterval(StripUtils.DEF_CAMERA_SHOT_INTERVAL);
            try {
                Element focalLengthEle = element.element(KMLConstants.FOCAL_LENGTH);
                if (focalLengthEle != null) {
                    cameraModel.setFocalLength(KMLValueConverter.string2Float(focalLengthEle.getTextTrim()));
                }
                Element sensorHEle = element.element(KMLConstants.SENSOR_H);
                if (sensorHEle != null) {
                    cameraModel.setSensorHeight(KMLValueConverter.string2Float(sensorHEle.getTextTrim()));
                }
                Element sensorWEle = element.element(KMLConstants.SENSOR_W);
                if (sensorWEle != null) {
                    cameraModel.setSensorWidth(KMLValueConverter.string2Float(sensorWEle.getTextTrim()));
                }
                Element imageWEle = element.element(KMLConstants.IMAGE_W);
                if (imageWEle != null) {
                    cameraModel.setImageWidth(KMLValueConverter.string2Int(imageWEle.getTextTrim()));
                }
                Element imageHEle = element.element(KMLConstants.IMAGE_H);
                if (imageHEle != null) {
                    cameraModel.setImageHeight(KMLValueConverter.string2Int(imageHEle.getTextTrim()));
                }
                Element shotIntervalEle = element.element(KMLConstants.SHOT_INTERVAL);
                if (shotIntervalEle != null) {
                    cameraModel.setShotInterval(KMLValueConverter.string2Float(shotIntervalEle.getTextTrim()));
                }
                waylineModel.setCamera(cameraModel);
            } catch (NumberFormatException e) {
                Log.e(TAG, KMLConstants.CAMERA_PARAM_ILLEGAL);
            }
        }
    }

    /**
     * 解析相机相关参数
     */
    private static void parseCameraData(Element element, MappingWaylineModelGreenDao waylineModel) {
        Element cameraEle = element.element(KMLConstants.CAMERA_TYPE);
        if (cameraEle != null
                && KMLValueConverter.isKnownCamera(cameraEle.getTextTrim())
                && MappingCameraType.find(cameraEle.getTextTrim()) != MappingCameraType.OTHER) {
            waylineModel.setCamera(createMappingCameraModel(MappingCameraType.find(cameraEle.getTextTrim())));
        } else {
            MappingCameraModelGreenDao cameraModel = new MappingCameraModelGreenDao();
            cameraModel.setName("Custom Camera");
            cameraModel.setSensorHeight(MappingUtils.DEF_CAMERA_SENSOR_HEIGHT);
            cameraModel.setSensorWidth(MappingUtils.DEF_CAMERA_SENSOR_WIDTH);
            cameraModel.setImageHeight(MappingUtils.DEF_CAMERA_IMAGE_HEIGHT);
            cameraModel.setImageWidth(MappingUtils.DEF_CAMERA_IMAGE_WIDTH);
            cameraModel.setFocalLength(MappingUtils.DEF_CAMERA_FOCAL_LENGTH);
            cameraModel.setShotInterval(MappingUtils.DEF_CAMERA_SHOT_INTERVAL);
            try {
                Element focalLengthEle = element.element(KMLConstants.FOCAL_LENGTH);
                if (focalLengthEle != null) {
                    cameraModel.setFocalLength(KMLValueConverter.string2Float(focalLengthEle.getTextTrim()));
                }
                Element sensorHEle = element.element(KMLConstants.SENSOR_H);
                if (sensorHEle != null) {
                    cameraModel.setSensorHeight(KMLValueConverter.string2Float(sensorHEle.getTextTrim()));
                }
                Element sensorWEle = element.element(KMLConstants.SENSOR_W);
                if (sensorWEle != null) {
                    cameraModel.setSensorWidth(KMLValueConverter.string2Float(sensorWEle.getTextTrim()));
                }
                Element imageWEle = element.element(KMLConstants.IMAGE_W);
                if (imageWEle != null) {
                    cameraModel.setImageWidth(KMLValueConverter.string2Int(imageWEle.getTextTrim()));
                }
                Element imageHEle = element.element(KMLConstants.IMAGE_H);
                if (imageHEle != null) {
                    cameraModel.setImageHeight(KMLValueConverter.string2Int(imageHEle.getTextTrim()));
                }
                Element shotIntervalEle = element.element(KMLConstants.SHOT_INTERVAL);
                if (shotIntervalEle != null) {
                    cameraModel.setShotInterval(KMLValueConverter.string2Float(shotIntervalEle.getTextTrim()));
                }
                waylineModel.setCamera(cameraModel);
            } catch (NumberFormatException e) {
                Log.e(TAG, KMLConstants.CAMERA_PARAM_ILLEGAL);
            }
        }
    }

    private static MappingCameraModelGreenDao createMappingCameraModel(MappingCameraType cameraType) {
        if (cameraType != MappingCameraType.OTHER) {
            MappingCameraModelGreenDao cameraModel = new MappingCameraModelGreenDao();
            cameraModel.setName(cameraType.getNameStr());
            cameraModel.setSensorWidth(cameraType.getSensorW());
            cameraModel.setSensorHeight(cameraType.getSensorH());
            cameraModel.setFocalLength(cameraType.getFocalLength());
            cameraModel.setImageHeight(cameraType.getImageHeight());
            cameraModel.setImageWidth(cameraType.getImageWidth());
            cameraModel.setShotInterval(MappingUtils.getCameraShotInterval(cameraType, false));
            return cameraModel;
        }
        return null;
    }

    /**
     * 若MissionType为3D类型，则解析3D扩展数据
     */
    private static void parse3DData(Element element, MappingWaylineModelGreenDao waylineModel) {
        Element gimbalPitchEle = element.element(KMLConstants.GIMBAL_PITCH);
        if (gimbalPitchEle != null) {
            try {
                int gimbalPitch = KMLValueConverter.string2Int(gimbalPitchEle.getTextTrim());
                if (gimbalPitch >= MappingUtils.MIN_GIMBAL_PITCH && gimbalPitch <= MappingUtils.MAX_GIMBAL_PITCH) {
                    waylineModel.setGimbalPitch(gimbalPitch);
                }
            } catch (NumberFormatException ex) {
                Log.e(TAG, KMLConstants.DIRECTION_PARAM_ILLEGAL + gimbalPitchEle.getTextTrim());
            }
        }
        Element inclineSpeedEle = element.element(KMLConstants.INCLINE_SPEED);
        if (inclineSpeedEle != null) {
            try {
                float inclineSpeed = KMLValueConverter.string2Float(inclineSpeedEle.getTextTrim());
                if (inclineSpeed >= MappingUtils.MIN_SPEED && inclineSpeed <= MappingUtils.MAX_SPEED) {
                    waylineModel.setInclineSpeed(inclineSpeed);
                }
            } catch (NumberFormatException ex) {
                Log.e(TAG, KMLConstants.INCLINE_SPEED_PARAM_ILLEGAL + inclineSpeedEle.getTextTrim());
            }
        }
        Element inclineOverlapWEle = element.element(KMLConstants.INCLINE_OVERLAP_W);
        if (inclineOverlapWEle != null) {
            try {
                int inclineOverlapW = KMLValueConverter.string2Int(inclineOverlapWEle.getTextTrim());
                if (inclineOverlapW >= MappingUtils.MIN_OVERLAP && inclineOverlapW <= MappingUtils.MAX_OVERLAP) {
                    waylineModel.setInclineOverlapW(inclineOverlapW);
                }
            } catch (NumberFormatException ex) {
                Log.e(TAG, KMLConstants.INCLINE_OVERLAP_W_PARAM_ILLEGAL + inclineOverlapWEle.getTextTrim());
            }
        }
        Element inclineOverlapHEle = element.element(KMLConstants.INCLINE_OVERLAP_H);
        if (inclineOverlapHEle != null) {
            try {
                int inclineOverlapH = KMLValueConverter.string2Int(inclineOverlapHEle.getTextTrim());
                if (inclineOverlapH >= MappingUtils.MIN_OVERLAP && inclineOverlapH <= MappingUtils.MAX_OVERLAP) {
                    waylineModel.setInclineOverlapH(inclineOverlapH);
                }
            } catch (NumberFormatException ex) {
                Log.e(TAG, KMLConstants.INCLINE_OVERLAP_H_PARAM_ILLEGAL + inclineOverlapHEle.getTextTrim());
            }
        }
    }

}
