package com.dji.wpmzsdk.common.utils.kml;
/*
 *   WWWWWW||WWWWWW
 *    W W W||W W W
 *         ||
 *       ( OO )__________
 *        /  |           \
 *       /o o|    DJI     \
 *       \___/||_||__||_|| **
 *            || ||  || ||
 *           _||_|| _||_||
 *          (__|__|(__|__|
 *
 * Copyright (c) 2017, DJI All Rights Reserved.
 */

import android.content.Context;
import android.location.Location;

import androidx.annotation.Nullable;


import com.dji.wpmzsdk.common.utils.kml.data.MissionType;
import com.dji.wpmzsdk.common.utils.kml.mission.MissionInfoModel;
import com.dji.wpmzsdk.common.utils.kml.mission.WaypointMissionFinishedAction;
import com.dji.wpmzsdk.common.utils.kml.model.CameraType;
import com.dji.wpmzsdk.common.utils.kml.model.DJILatLng;
import com.dji.wpmzsdk.common.utils.kml.model.EdgePointModel;
import com.dji.wpmzsdk.common.utils.kml.model.Mapping2DWaylineModel;
import com.dji.wpmzsdk.common.utils.kml.model.Mapping3DWaylineModel;
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraModel;
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraType;
import com.dji.wpmzsdk.common.utils.kml.model.MappingMissionModel;
import com.dji.wpmzsdk.common.utils.kml.model.MissionInfoExtModel;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import dji.sdk.wpmz.value.mission.WaylineExitOnRCLostAction;

/**
 * <p>Description:</p>
 *
 * @author create at 2018/12/26 6:06 PM by daniel for dji-pilot
 * @version v1.0
 */
public class MappingUtils {

    public static final int EP800_CALI_POINTS_NUM = 32;
    // Mapping Param Range
    // relative alt mode
    public static final int MAX_RELATIVE_ALTITUDE = 1500;
    public static final int MIN_RELATIVE_ALTITUDE = 12;
    public static final int DEF_RELATIVE_ALTITUDE = 100; // 被摄面相对于起飞点
    public static final int MIN_RELATIVE_TO_TAKEOFF_DISTANCE = -200; // 被摄面相对于起飞点
    public static final int MAX_RELATIVE_TO_TAKEOFF_DISTANCE = 1500; // 被摄面相对于起飞点
    public static final int DEF_RELATIVE_TO_TAKEOFF_DISTANCE = 0; // 被摄面相对于起飞点
    // absolute alt mode
    public static final int MAX_ABSOLUTE_ALTITUDE = 10000;
    public static final int MIN_ABSOLUTE_ALTITUDE = -10000;
    public static final int DEF_ABSOLUTE_ALTITUDE = 0;
    public static final int MAX_RELATIVE_TO_TARGET_ALTITUDE = 1500; // 航线相对被摄面
    public static final int MIN_RELATIVE_TO_TARGET_ALTITUDE = 0;
    public static final int DEF_RELATIVE_TO_TARGET_ALTITUDE = 100;

    // Altitude
    public static final int MAX_ALTITUDE = 10000;
    public static final int MIN_ALTITUDE = -10000;
    // Direction
    public static final int MAX_DIRECTION = 359;
    public static final int MIN_DIRECTION = 0;
    // Margin
    public static final float MARGIN_COE = 1F;
    // Gimbal Pitch
    public static final int MAX_GIMBAL_PITCH = -40;
    public static final int MIN_GIMBAL_PITCH = -85;
    // Overlap
    public static final int MAX_OVERLAP = 90;
    public static final int MIN_OVERLAP = 10;
    // Speed
    public static final int MAX_SPEED = 15;
    public static final int MIN_SPEED = 1;

    // Camera Interval
    public static final float MIN_SHOT_INTERVAL = 1f;
    public static final float MAX_SHOT_INTERVAL = 10f;
    public static final float DEF_CAMERA_SHOT_INTERVAL = 2.5f;
    public static final float DEF_GD610_CAMERA_SHOT_INTERVAL = 3.0f;
    public static final float DEF_EP600_CAMERA_SHOT_INTERVAL_WITH_FIVE_WAY_POSE = 1.0f * 5;
    public static final float DEF_EP600_CAMERA_SHOT_INTERVAL_WITH_SINGLE_WAY_POSE = 1.0f; // 摆拍中单拍一张的时间
    public static final int DEF_CAMERA_IMAGE_WIDTH = 3000;
    public static final int DEF_CAMERA_IMAGE_HEIGHT = 3000;
    public static final float DEF_CAMERA_SENSOR_WIDTH = 10f;
    public static final float DEF_CAMERA_SENSOR_HEIGHT = 10f;
    public static final float DEF_CAMERA_FOCAL_LENGTH = 10f;

    public static final float DEF_MAPPING_SPEED = 15;
    public static final float DEF_MAPPING_TAKEOFF_SPEED = 10;
    public static final int DEF_MAPPING_HEIGHT = 100;
    public static final float DEF_MAPPING_MARGIN = 0f;
    public static final float DEF_MAPPING_MARGIN_VALUE = 0.25f;
    public static final float DEF_MAPPING_MARGIN_MAX = 1.0f;
    public static final float DEF_MAPPING_MARGIN_STEP = 0.01f;
    public static final int DEF_MAPPING_LIDAR_OVERLAP_W = 20;
    public static final int DEF_MAPPING_LIDAR_OVERLAP_H = 70;
    public static final int DEF_MAPPING_OVERLAP_W = 70;
    public static final int DEF_MAPPING_OVERLAP_H = 80;
    public static final int DEF_MAPPING_INCLINE_OVERLAP_W = 60;
    public static final int DEF_MAPPING_INCLINE_OVERLAP_H = 70;
    public static final int DEF_MAPPING_GIMBAL_PITCH = -45;
    public static final int DEF_MAPPING_DSM_HEIGHT = 150;
    public static final int DEF_MAPPING_MIN_DSM_HEIGHT = 25;
    public static final int DEF_MAPPING_MAX_DSM_HEIGHT = 1500;
    public static final int DEF_MAPPING_MIN_REALTIME_DSM_HEIGHT = 30;
    public static final int DEF_MAPPING_MAX_REALTIME_DSM_HEIGHT = 200;

    public static final double LIDAR_COEFF_K = 0.875d;
    public static final double LIDAR_COEFF_B = 0.125d;
    public static final double LIDAR_VARIEGATION_OVERLAP_H = 70d;
    public static final int LIDAR_SCANMODE_NON_REPEAT = 0;
    public static final int LIDAR_SCANMODE_REPEAT = 1;

    public static final int MAPPING_CALI_DRAW_TYPE_NONE = 0;
    public static final int MAPPING_CALI_DRAW_TYPE_NORMAL = 1;
    public static final int MAPPING_CALI_DRAW_TYPE_ELEVATION = 2;

    public static final int MAPPING_METER_TO_CENTIMETER = 100;
    public static final int MAPPING_GSD_FACTOR = 1000;
    public static final double MAX_AREA = 1000000000.0;
    public static final float DEF_SECURE_TAKEOFF_HEIGHT = 20F;
    public static final float MIN_SECURE_TAKEOFF_HEIGHT = 2F;

//    public static void applySettings(Mapping2DWaylineModel model, MappingBasic2DModel viewModel) {
//        model.setAltitude(viewModel.getAltitude());
//        model.setDirection(viewModel.getDirection());
//        model.setWgs84Altitude(viewModel.getWgs84Altitude());
//        model.setTakeoffSpeed(viewModel.getTakeoffSpeed());
//        model.setSpeed(viewModel.getSpeed());
//        model.setActionOnFinish(viewModel.getCompletion().getValue());
//        model.setElevationOptimize(viewModel.isElevationOptimize());
//        model.setFiveWayPose(viewModel.isFiveWayPose());
//        model.setAltitudeMode(viewModel.getAltitudeMode());
//        model.setRelativeDistance(viewModel.getRelativeDistance());
//        model.setEnableCalibrate(viewModel.isEnableCalibrate());
//        model.setDirection(viewModel.getDirection());
//        model.setSecureTakeoffHeight(viewModel.getSecureTakeoffHeight());
//        model.setMappingInclineClimbEnable(viewModel.isInclineClimbEnable());
//        if(viewModel.isFiveWayPose()){
//            model.setFiveWayPoseWithGimbalPitch(viewModel.getFiveWayPoseWithGimbalPitch());
//        }else{
//            model.setFiveWayPoseWithGimbalPitch(DEF_MAPPING_GIMBAL_PITCH);
//        }
//        model.setEnableDsm(viewModel.isEnableDsm());
//        model.setDsmAltitude(viewModel.getDsmAltitude());
//        model.setRealtimeSurfaceFollow(viewModel.isRealtimeSurfaceFollow());
//        if (viewModel.getCameraType() == MappingCameraType.OTHER) {
//            model.setCamera(viewModel.getMappingCameraModel());
//        } else {
//            model.setCamera(createMappingCameraModel(viewModel.getCameraType()));
//        }
//        if (model instanceof Mapping3DWaylineModel && viewModel instanceof MappingBasic3DModel) {
//            MappingBasic3DModel basic3DModel = (MappingBasic3DModel) viewModel;
//            ((Mapping3DWaylineModel) model).setInclineSpeed(basic3DModel.getInclineSpeed());
//            ((Mapping3DWaylineModel) model).setGimbalPitch(basic3DModel.getGimbalPitch());
//        }
//    }
//
//    public static void applySettings(Mapping2DWaylineModel model, MappingAdvanced2DModel viewModel) {
//        model.setOverlapH(viewModel.getOverlapH());
//        model.setOverlapW(viewModel.getOverlapW());
//        model.setMargin(viewModel.getMargin());
//        model.setPhotoMode(viewModel.getPhotoMode());
//        model.setYawParamEnable(viewModel.isMappingYawParamEnable());
//        model.setYawParamMode(viewModel.getMappingYawParamMode());
//        model.setYawParamAngle(viewModel.getMappingYawParamAngle());
//        model.setYawParamGimbalPitchMode(viewModel.getMappingYawParamGimbalPitchMode());
//        model.setYawParamGimbalPitchAngle(viewModel.getMappingYawParamGimbalPitchAngle());
//
//        if (model instanceof Mapping3DWaylineModel && viewModel instanceof MappingAdvanced3DModel) {
//            MappingAdvanced3DModel advanced3DModel = (MappingAdvanced3DModel) viewModel;
//            ((Mapping3DWaylineModel) model).setInclineOverlapW(advanced3DModel.getInclineOverlapW());
//            ((Mapping3DWaylineModel) model).setInclineOverlapH(advanced3DModel.getInclineOverlapH());
//        }
//    }

//    public static void applySettings(Mapping2DWaylineModel model, CameraAdvancedModel viewModel) {
//        model.setEnableDewarping(viewModel.isDewarpEnable());
//        model.setPhotoFormat(viewModel.getPhotoFormat());
//        model.setFocusMode(viewModel.getWaylineFocusMode());
//        model.setEchoMode(viewModel.getEchoMode());
//        model.setSampleRate(viewModel.getSampleRate());
//        model.setScanMode(viewModel.getScanMode());
//        model.setNeedVariegation(viewModel.isNeedVariegation());
//    }

    public static MappingMissionModel createMappingModel(String name, MissionType type, MappingCameraModel cameraModel, List<? extends DJILatLng> initPoints) {
        MappingMissionModel mappingModel = new MappingMissionModel();
        mappingModel.setMissionInfo(createMissionInfoMode(name, type));
        mappingModel.setMappingWayline(type == MissionType.Mapping3D ?
                createMapping3DWaylineModel(cameraModel) : createMapping2DWaylineModel(cameraModel));
        addInitPoints(mappingModel, initPoints);
        return mappingModel;
    }

    private static void addInitPoints(MappingMissionModel mappingModel, List<? extends DJILatLng> initPoints) {
        if (initPoints == null) {
            return;
        }
        List<EdgePointModel> points = new ArrayList<>();
        for (DJILatLng initPoint : initPoints) {
            points.add(new EdgePointModel(initPoint.latitude, initPoint.longitude));
        }
        mappingModel.setEdgePoints(points);
    }

    private static MissionInfoModel createMissionInfoMode(String name, MissionType type) {
        MissionInfoModel info = new MissionInfoModel();
        info.setName(name);
        info.setType(type);
        info.setCreateTime(System.currentTimeMillis());
        info.setUpdateTime(System.currentTimeMillis());
        info.setCollected(false);
        info.setUuid(UUID.randomUUID().toString().replace("-", "_"));
        info.setExtInfo(new MissionInfoExtModel());
        return info;
    }

    private static Mapping2DWaylineModel createMapping2DWaylineModel(MappingCameraModel cameraModel) {
        Mapping2DWaylineModel waylineModel = new Mapping2DWaylineModel();
        waylineModel.setActionOnFinish(WaypointMissionFinishedAction.GO_HOME);
        waylineModel.setAltitude(DEF_MAPPING_HEIGHT);
        waylineModel.setTakeoffSpeed(DEF_MAPPING_TAKEOFF_SPEED);
        waylineModel.setSpeed(DEF_MAPPING_SPEED);
        waylineModel.setMargin(DEF_MAPPING_MARGIN);
        waylineModel.setOverlapW(DEF_MAPPING_OVERLAP_W);
        waylineModel.setOverlapH(DEF_MAPPING_OVERLAP_H);
        waylineModel.setDirection(0);
        waylineModel.setCamera(cameraModel);
        waylineModel.setDsmAltitude(DEF_MAPPING_DSM_HEIGHT);
        waylineModel.setFiveWayPoseWithGimbalPitch(DEF_MAPPING_GIMBAL_PITCH);
        waylineModel.setScanMode(LIDAR_SCANMODE_REPEAT);
        waylineModel.setNeedVariegation(true); // 默认打开点云上色
        waylineModel.setLostAction(WaylineExitOnRCLostAction.GO_BACK);
        waylineModel.setSecureTakeoffHeight(DEF_SECURE_TAKEOFF_HEIGHT);
        waylineModel.setMappingInclineClimbEnable(false);
        waylineModel.setElevationOptimize(true);
        return waylineModel;
    }

//    public static void saveSelectedCameraName(MappingCameraModel model) {
//        SharedPreferencesUtil.putString(Constant.MAPPING_CAMERA_SELECTED, model.getName(), false);
//    }

    public static MappingCameraModel createMappingCameraModel(MappingCameraType cameraType) {
        if (cameraType != MappingCameraType.OTHER) {
            MappingCameraModel cameraModel = new MappingCameraModel();
            cameraModel.setName(cameraType.getNameStr());
            cameraModel.setSensorWidth(cameraType.getSensorW());
            cameraModel.setSensorHeight(cameraType.getSensorH());
            cameraModel.setFocalLength(cameraType.getFocalLength());
            cameraModel.setImageHeight(cameraType.getImageHeight());
            cameraModel.setImageWidth(cameraType.getImageWidth());
            cameraModel.setShotInterval(getCameraShotInterval(cameraType, false));
            return cameraModel;
        }
        return null;
    }

    public static Mapping3DWaylineModel createMapping3DWaylineModel(MappingCameraModel cameraModel) {
        Mapping3DWaylineModel waylineModel = new Mapping3DWaylineModel();
        waylineModel.setActionOnFinish(WaypointMissionFinishedAction.GO_HOME);
        waylineModel.setAltitude(DEF_MAPPING_HEIGHT);
        waylineModel.setTakeoffSpeed(DEF_MAPPING_TAKEOFF_SPEED);
        waylineModel.setSpeed(DEF_MAPPING_SPEED);
        waylineModel.setMargin(DEF_MAPPING_MARGIN);
        waylineModel.setOverlapW(DEF_MAPPING_OVERLAP_W);
        waylineModel.setOverlapH(DEF_MAPPING_OVERLAP_H);
        waylineModel.setDirection(0);
        waylineModel.setCamera(cameraModel);
        waylineModel.setInclineOverlapH(DEF_MAPPING_INCLINE_OVERLAP_H);
        waylineModel.setInclineOverlapW(DEF_MAPPING_INCLINE_OVERLAP_W);
        waylineModel.setInclineSpeed(DEF_MAPPING_SPEED);
        waylineModel.setGimbalPitch(DEF_MAPPING_GIMBAL_PITCH);
        waylineModel.setScanMode(LIDAR_SCANMODE_REPEAT);
        waylineModel.setNeedVariegation(true); // 默认打开点云上色
        waylineModel.setLostAction(WaylineExitOnRCLostAction.GO_BACK);
        waylineModel.setSecureTakeoffHeight(DEF_SECURE_TAKEOFF_HEIGHT);
        waylineModel.setMappingInclineClimbEnable(false);
        return waylineModel;
    }

//    public static MappingBasic2DModel createBasicSetting(MissionType type, Mapping2DWaylineModel waylineModel) {
//        MappingBasic2DModel model = null;
//        switch (type) {
//            case Mapping2D:
//                model = new MappingBasic2DModel();
//                break;
//            case Mapping3D:
//                Mapping3DWaylineModel model3D = (Mapping3DWaylineModel) waylineModel;
//                model = new MappingBasic3DModel();
//                ((MappingBasic3DModel) model).setInclineSpeed(model3D.getInclineSpeed());
//                ((MappingBasic3DModel) model).setGimbalPitch(model3D.getGimbalPitch());
//                break;
//            default:
//                break;
//        }
//        if (model != null) {
//            model.setDirection(waylineModel.getDirection());
//            model.setAltitude(waylineModel.getAltitude());
//            model.setTakeoffSpeed(waylineModel.getTakeoffSpeed());
//            initBasicSettingCameraType(waylineModel, model);
//            model.setSpeed(waylineModel.getSpeed());
//            model.getCompletion().setValue(waylineModel.getActionOnFinish());
//            model.setElevationOptimize(waylineModel.isElevationOptimize());
//            model.setFiveWayPose(waylineModel.isFiveWayPose());
//            model.setFiveWayPoseWithGimbalPitch(waylineModel.getFiveWayPoseWithGimbalPitch());
//            model.setEnableDsm(waylineModel.isEnableDsm());
//            model.setDsmAltitude(waylineModel.getDsmAltitude());
//            model.setRealtimeSurfaceFollow(waylineModel.isRealtimeSurfaceFollow());
//            model.setAltitudeMode(waylineModel.getAltitudeMode());
//            model.setRelativeDistance(waylineModel.getRelativeDistance());
//            model.setEnableCalibrate(waylineModel.isEnableCalibrate());
//            model.setSecureTakeoffHeight(waylineModel.getSecureTakeoffHeight());
//            model.setInclineClimbEnable(waylineModel.isMappingInclineClimbEnable());
//        }
//        return model;
//    }
//
//    private static void initBasicSettingCameraType(Mapping2DWaylineModel waylineModel, MappingBasic2DModel model) {
//        if (waylineModel == null || waylineModel.getCamera() == null) {
//            return;
//        }
//        String name = waylineModel.getCamera().getName();
//        model.setCameraType(null);
//        for (MappingCameraType type : MappingCameraType.values()) {
//            if (type.getNameStr().equals(name) && type != MappingCameraType.OTHER) {
//                model.setCameraType(type);
//                break;
//            }
//        }
//        if (model.getCameraType() == null) {
//            model.setCameraType(MappingCameraType.OTHER);
//            model.setMappingCameraModel(waylineModel.getCamera());
//        }
//    }
//
//    public static MappingAdvanced2DModel createAdvanceSetting(MissionType type, Mapping2DWaylineModel waylineModel) {
//        MappingAdvanced2DModel model = null;
//        switch (type) {
//            case Mapping2D:
//                model = new MappingAdvanced2DModel();
//                break;
//            case Mapping3D:
//                model = new MappingAdvanced3DModel();
//                ((MappingAdvanced3DModel) model).setInclineOverlapH(((Mapping3DWaylineModel) waylineModel).getInclineOverlapH());
//                ((MappingAdvanced3DModel) model).setInclineOverlapW(((Mapping3DWaylineModel) waylineModel).getInclineOverlapW());
//                break;
//            default:
//                break;
//        }
//        if (model != null) {
//            model.setMargin(waylineModel.getMargin());
//            model.setOverlapH(waylineModel.getOverlapH());
//            model.setOverlapW(waylineModel.getOverlapW());
//            model.setPhotoMode(waylineModel.getPhotoMode());
//            model.setSupportMappingYawParam(!waylineModel.isEnableDsm() && !waylineModel.isFiveWayPose());
//            model.setMappingYawParamEnable(waylineModel.isYawParamEnable());
//            model.setMappingYawParamMode(waylineModel.getYawParamMode());
//            model.setMappingYawParamAngle(waylineModel.getYawParamAngle());
//            model.setMappingYawParamGimbalPitchMode(waylineModel.getYawParamGimbalPitchMode());
//            model.setMappingYawParamGimbalPitchAngle(waylineModel.getYawParamGimbalPitchAngle());
//        }
//        return model;
//    }

//    public static CameraAdvancedModel createCameraAdvanceSetting(Mapping2DWaylineModel waylineModel) {
//        final CameraAdvancedModel cameraAdvancedModel = new CameraAdvancedModel();
//        cameraAdvancedModel.setDewarpEnable(waylineModel.isEnableDewarping());
//        cameraAdvancedModel.setPhotoFormat(waylineModel.getPhotoFormat());
//        cameraAdvancedModel.setWaylineFocusMode(waylineModel.getFocusMode());
//        cameraAdvancedModel.setEchoMode(waylineModel.getEchoMode());
//        cameraAdvancedModel.setSampleRate(waylineModel.getSampleRate());
//        cameraAdvancedModel.setScanMode(waylineModel.getScanMode());
//        cameraAdvancedModel.setNeedVariegation(waylineModel.isNeedVariegation());
//        return cameraAdvancedModel;
//    }

    public static void createDsmSetting(List<String> models, Mapping2DWaylineModel waylineModel) {
        final List<String> paths = waylineModel.getDsmPath();
        if (paths == null) {
            return;
        }
        models.addAll(paths);
    }


    //航线库中copy操作，航线name，creatTime，updateTime等会改变。
//    public static MappingMissionModel copyMappingMission(MappingMissionModel mappingMissionModel, String name) {
//        MappingMissionModel model = mappingMissionModel.clone();
//
//        if (model != null) {
//            model.setId(null);
//            handleCopiedMappingWayline(model.getMappingWayline());
//            String diagram = mappingMissionModel.getMissionInfo().getDiagram();
//            WaypointUtils.handleCopiedMissionInfo(model.getMissionInfo(), name, diagram);
//        }
//        return model;
//    }

    private static void handleCopiedMappingWayline(Mapping2DWaylineModel waylineModel) {
        if (waylineModel != null) {
            waylineModel.setId(null);
        }
    }


    public static float getMaxSpeedValue(double distance, double shotInterval) {
        //航向拍照距离  除以 相机的最小拍照间隔 就是 测绘飞行中的最大速度阈值
        return (float) (distance / shotInterval);
    }

    public static MappingCameraType getMappingCameraType(CameraType cameraType) {
        MappingCameraType type = MappingCameraType.OTHER;
        if (cameraType == null) {
            return type;
        }
        switch (cameraType) {
            case DJICameraTypeFC330X:
                type = MappingCameraType.P4;
                break;
            case DJICameraTypeFC300X:
                type = MappingCameraType.P4P;
                break;
            case DJICameraTypeFC6310A:
                type = MappingCameraType.P4A;
                break;
            case P4P_V2_CAMERA:
                type = MappingCameraType.P4PV2;
                break;
            default:
                type = MappingCameraType.P4;
                break;
        }
        return type;
    }


    /**
     * 获取区域中心点
     * from GpsUtils.getCenter()
     *
     * @param edgePoints
     * @return
     */
    @Nullable
    public static DJILatLng getCenter(List<EdgePointModel> edgePoints) {
        if (edgePoints == null || edgePoints.size() == 0) {
            return null;
        }

        double minLat = edgePoints.get(0).getLatitude();
        double minLng = edgePoints.get(0).getLongitude();
        double maxLat = edgePoints.get(0).getLatitude();
        double maxLng = edgePoints.get(0).getLongitude();
        for(int i = 0; i != edgePoints.size(); ++i) {
            double curLat = edgePoints.get(i).getLatitude();
            double curLng = edgePoints.get(i).getLongitude();
            if(curLat > maxLat) {
                maxLat = curLat;
            }
            if(curLat < minLat) {
                minLat =  curLat;
            }
            if(curLng > maxLng) {
                maxLng = curLng;
            }
            if(curLng < minLng) {
                minLng = curLng;
            }

        }

        return new DJILatLng((maxLat + minLat) / 2, (maxLng + minLng) / 2);
    }

    /**
     * 获取拍照间隔时间
     * @param cameraType 相机类型
     * @param needFivePose 是否摆拍
     */
    public static float getCameraShotInterval(MappingCameraType cameraType, boolean needFivePose) {
        if (MappingCameraType.isGD610Series(cameraType)) {  // 610 3s
            return DEF_GD610_CAMERA_SHOT_INTERVAL;
        } else if (cameraType == MappingCameraType.SHARE_102S) { // 赛尔相机间隔1s
            return 1.0f;
        } else if (MappingCameraType.isEP600(cameraType)) {
            if (needFivePose) { // 摆拍一个方向为1s， 1 * 5 = 5s
                return DEF_EP600_CAMERA_SHOT_INTERVAL_WITH_FIVE_WAY_POSE;
            } else { // EP600 正射拍照间隔 1s
                return 1f;
            }
        }
        return DEF_CAMERA_SHOT_INTERVAL;
    }

    /**
     * 测绘系列相机： EP600 & EP800
     * 1. 相机支持高级设置
     * 2. 航线支持仿地飞行
     */
    public static boolean isMappingCamera(MappingCameraType type) {
        return MappingCameraType.isEP600(type) || MappingCameraType.isEP800(type);
    }

//    public static int getMappingFiveWayPoseExecutionTime(List<DJILatLng> points, double stepH) {
//        if (points == null || points.isEmpty()) {
//            return 0;
//        }
//
//        float time = 0f;
//        for (int i = 0; i < points.size() - 1; i++) {
//            DJILatLng cur = points.get(i);
//            DJILatLng next = points.get(i + 1);
//
//            double speed = 1;
//            final float costTime = SmartObliqueCaptureDirection.costTime(cur.tag);
//            if (SmartObliqueCaptureDirection.none(cur.tag) || costTime == 0) {
//                speed = 15.0;
//            } else {
//                final double vMax = stepH / costTime;
//                if (vMax >= 15) {
//                    speed = 15.0;
//                } else {
//                    speed = vMax;
//                }
//            }
//            double d = LocationUtils.getDistanceInMeterFromTwoGPSLocations(cur.latitude, cur.longitude, next.latitude, next.longitude);
//            time += WaypointUtils.getFlyTime(d, speed, 3.5f);
//        }
//        return (int) Math.ceil(time);
//    }
//
//    public static String getLidarVisibleOverlapText(int overlapW) {
//        return String.format(Locale.US, "%d%%", LidarOverlapWTransformKt.getVisibleOverlapWValue(overlapW));
//    }

    /**
     * calculate GSD by public method (X7 && custom camera use this method)
     */
    public static float calculateGsdValueWithImageSize(float height, MappingCameraType cameraType, MappingCameraModel cameraModel) {
        if (cameraType == MappingCameraType.NULL) {
            return 0f;
        }
        if (cameraType != MappingCameraType.OTHER) {
            // 激光雷达相机计算GSD时的相机参数需要用可见光的参数去计算。雷达激光的参数用于计算航线，满足重叠率的
            if (cameraType == MappingCameraType.EP800_LIDAR) {
                return Math.max(
                        (100 * height * MappingCameraType.EP800_VISUAL.getSensorH())
                                / (MappingCameraType.EP800_VISUAL.getFocalLength() * MappingCameraType.EP800_VISUAL.getImageHeight()),
                        (100 * height * MappingCameraType.EP800_VISUAL.getSensorW())
                                / (MappingCameraType.EP800_VISUAL.getFocalLength() * MappingCameraType.EP800_VISUAL.getImageWidth())
                );
            } else {
                return Math.max(
                        (100 * height * cameraType.getSensorH()) / (cameraType.getFocalLength() * cameraType.getImageHeight()),
                        (100 * height * cameraType.getSensorW()) / (cameraType.getFocalLength() * cameraType.getImageWidth())
                );
            }
        } else if (cameraModel != null) {
            if (cameraModel.getImageHeight() == 0 || cameraModel.getImageWidth() == 0) {
                return calculateGsdValueWithGsdBase(height, cameraType, cameraModel);
            }
            return Math.max(
                    (100 * height * cameraType.getSensorH()) / (cameraType.getFocalLength() * cameraType.getImageHeight()),
                    (100 * height * cameraType.getSensorW()) / (cameraType.getFocalLength() * cameraType.getImageWidth())
            );
        }
        return 0;
    }

    public static float calculateTerraGsdValue(float height, MappingCameraType cameraType, boolean terraMappingUseIrLens) {
        // 快速建图默认预览图 960*720
        int imageWidth = 960;
        int imageHeight = 720;

        boolean isIrCamera = cameraType == MappingCameraType.PM320_IR
                || cameraType == MappingCameraType.GD610_IR
                || cameraType == MappingCameraType.GD612_IR;
        // 红外预览图
        if (isIrCamera || terraMappingUseIrLens) {
            imageWidth = 640;
            imageHeight = 512;
        }
        return Math.max(
                (100 * height * cameraType.getSensorH()) / (cameraType.getFocalLength() * imageHeight),
                (100 * height * cameraType.getSensorW()) / (cameraType.getFocalLength() * imageWidth)
        );

    }

    /**
     * 获取测区中心点经纬度坐标
     * @param areaPoints 测区点经纬度坐标列表
     * @return
     */
    public static DJILatLng calculateCenterLocation(@NotNull List<DJILatLng> areaPoints) {
        Location centerLocation = GpsUtils.getCenter(CollectionUtil.transform(areaPoints,
                                                                              GpsUtils::convertToLocation));
        return new DJILatLng(centerLocation.getLatitude(),
                                      centerLocation.getLongitude(),
                                      centerLocation.getAltitude(),
                                      centerLocation.getAccuracy());
    }

    /**
     * GSD = survey.height / gsdBase
     * gsdBase = 影像的宽度 * 35mm等效焦距 / 35, 这个公式算可能会有点偏差，需要进一步修正，回头有新的设备让专门的童鞋来算吧
     *
     * @param height 飞行器高度
     *               <p>
     *               MAPPING_GSD_FACTOR : 根据计算公式和P4P的数据（sensor_w: 13.13, sensor_h: 8.76, focal_length: 8.8, gsdBase: 3650.0）推导得出。
     * @return gsd value
     */
    public static float calculateGsdValueWithGsdBase(float height, MappingCameraType cameraType, MappingCameraModel cameraModel) {
        double gsdBase;
        if (cameraType != MappingCameraType.OTHER) {
            return height * MAPPING_METER_TO_CENTIMETER / cameraType.getGsdBase();
        } else if (cameraModel != null) {
            gsdBase = MAPPING_GSD_FACTOR * cameraModel.getSensorWidth() * cameraModel.getFocalLength() / 35;
            if (gsdBase != 0) {
                return height * MAPPING_METER_TO_CENTIMETER / (float) gsdBase;
            }
        }
        return 0f;
    }


}
