package com.dji.wpmzsdk.common.utils.kml.converter


import android.content.Context
import com.dji.wpmzsdk.common.utils.FileUtils
import com.dji.wpmzsdk.common.utils.KMLZipHelper
import com.dji.wpmzsdk.common.utils.kml.EncryptUtil
import java.io.File
import java.io.IOException
import java.util.*
import kotlin.math.abs

/**
 * Description :
 * wpml kmz 文件缓存包装，优化大量资源文件下解压和压缩耗时
 *
 * 内部缓存结构 xxx.kmz（除资源外的文件打包，包含 template.kml 和 waylines.wpml）; res/image/ ; res/dsm
 * @filename : KMZPathWrapper
 * @author : devin.xu
 * @time : 2022/5/20
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class KMZPathWrapper(val dirPath: String, val name: String) {

    private val TAG = "KMZPathWrapper"

    init {
        File(dirPath).mkdirs()
    }

    fun getClipKmzFile(): File {
        return File(dirPath, name + SUFFIX)
    }

    fun getDsmResFolder(): File {
        return File(dirPath, RES_DSM)
    }

    fun getImageResFolder(): File {
        return File(dirPath, RES_IMAGE)
    }

    fun getAudioResFolder(): File {
        return File(dirPath, RES_AUDIO)
    }

    fun getWaylineFile(): File {
        return File(dirPath, "$KMZ_ROOT_PATH/$WAYLINE_FILE")
    }

    fun copyClipKmzFile(name: String): File {
        val localFile =  getClipKmzFile()
        val cacheFolder = File(dirPath, CACHE_PATH)
        cacheFolder.mkdirs()
        val renameFile = File(cacheFolder, name)
        FileUtils.copyFileByChannel(localFile.path, renameFile.path)
        return renameFile
    }

    @Throws(IOException::class) fun repackClipKmzFile(context: Context){
        clearOtherClipKmzFile()

        val rootFolder = File(dirPath, KMZ_ROOT_PATH)
        val outputFile = getClipKmzFile()
        outputFile.createNewFile()
        KMLZipHelper.zipFiles(context, mutableListOf(rootFolder.path), outputFile.path)
    }

    fun refreshRes(imageFiles: List<String>?, dsmFiles: List<String>?, audioFiles: List<String>?) {
        refreshRes(RES_IMAGE, imageFiles)
        refreshRes(RES_DSM, dsmFiles)
        refreshRes(RES_AUDIO, audioFiles)
    }

    fun clearCache() {
        FileUtils.delFile(dirPath)
    }

    fun clearOtherClipKmzFile() {
        val childFile = File(dirPath).listFiles { _: File?, name: String -> name.endsWith(SUFFIX) }
        if (childFile != null && childFile.isNotEmpty()) {
            childFile.forEach { it.delete() }
        }
    }

    fun unzipFullKmz(context: Context ,kmzPath: String) {
        if (!checkKmzNeedUnzip(context ,kmzPath)) {
            return
        }
        val templateRelativePath = KMLZipHelper.getSelectFilePath(context, kmzPath, TEMPLATE_FILE) ?: return

        FileUtils.delFile(dirPath)
        File(dirPath).mkdirs()
        val unzipFolder = File(dirPath, "temp")
        KMLZipHelper.unZipFolder(context, kmzPath, unzipFolder.path, false)

        val rootFolder = File(dirPath, KMZ_ROOT_PATH)
        File(unzipFolder, templateRelativePath).parentFile?.renameTo(rootFolder)

        try {
            val zipFiles = mutableListOf<String>()
            val templateFile = File(rootFolder, TEMPLATE_FILE)
            val cacheFolder = File(dirPath, "$CACHE_PATH/$KMZ_ROOT_PATH")
            cacheFolder.mkdirs()

            if (templateFile.exists()) {
                val copyTemp = File(cacheFolder, TEMPLATE_FILE)
                FileUtils.copyFileByChannel(templateFile.path, copyTemp.path)
            }
            val waylineFile = File(rootFolder, WAYLINE_FILE)
            if (waylineFile.exists()) {
                val copyWayline = File(cacheFolder, WAYLINE_FILE)
                FileUtils.copyFileByChannel(waylineFile.path, copyWayline.path)
            }

            if (templateFile.exists() || waylineFile.exists()) {
                zipFiles.add(cacheFolder.path)
                getClipKmzFile().createNewFile()
                KMLZipHelper.zipFiles(context, zipFiles, getClipKmzFile().path)
            }
        } catch (e: Exception) {
            // ignore
        }
    }

    fun zipFullKmz(context: Context): File {
        return try {
            KMLZipHelper.unZipFolder(context, getClipKmzFile().path, dirPath, false)

            val rootFolder = File(dirPath, KMZ_ROOT_PATH)
            val cacheFolder = File(dirPath, CACHE_PATH)
            cacheFolder.mkdirs()
            val outputFile = File(cacheFolder, OUTPUT_KMZ_FILE)
            outputFile.createNewFile()
            KMLZipHelper.zipFiles(context, mutableListOf(rootFolder.path), outputFile.path)
            outputFile
        } catch (e: Exception) {
            getClipKmzFile()
        }
    }

    /**
     * 比较 template 文件是否一致，如果一致则认为已解压过，不需要重复解压
     * @param origKmz
     * @return
     */
    private fun checkKmzNeedUnzip(context: Context ,origKmz: String?): Boolean {
        val clipKmzFile = getClipKmzFile()
        if (clipKmzFile.exists() && clipKmzFile.length() > 0) {
            val rootFolder = File(dirPath, KMZ_ROOT_PATH)
            rootFolder.mkdirs()
            val origTemplateFilePath = File(rootFolder, TEMPLATE_FILE).path

            val cacheFolder = File(dirPath, CACHE_PATH)
            cacheFolder.mkdirs()
            val clipTemplateFilePath = File(cacheFolder, "$TEMPLATE_FILE.temp").path

            val origTemplateFile = KMLZipHelper.unZipFile(context, origKmz, origTemplateFilePath, TEMPLATE_FILE)
            val clipTemplateFile = KMLZipHelper.unZipFile(context, clipKmzFile.path, clipTemplateFilePath, TEMPLATE_FILE)

            if (origTemplateFile != null && clipTemplateFile != null) {
                return !Objects.equals(FileUtils.getFileMD5(origTemplateFile), FileUtils.getFileMD5(clipTemplateFile))
            }
        }
        return true
    }

    private fun refreshRes(storePath: String, resFiles: List<String>?) {
        val resDir = File(dirPath, storePath)
        if (resFiles != null && resFiles.isNotEmpty()) {
            if (!resDir.exists()) {
                resDir.mkdirs()
            }

            val resFileNames = resFiles.map { File(it).name }
            resDir.listFiles()?.forEach {
                if (!resFileNames.contains(it.name)) {
                    it.delete()
                }
            }

            resFiles.forEach {
                val src = File(it)
                val dst = File(resDir, src.name)
                if (!dst.exists()) {
                    FileUtils.copyFileByChannel(it, dst.path)
                }
            }

        } else if (resDir.exists()) {
            FileUtils.delFile(resDir)
        }
    }

    companion object Factory {

        private const val KMZ_ROOT_PATH = "wpmz"
        private const val CACHE_PATH = "cache"
        private const val OUTPUT_KMZ_FILE = "output.kmz"

        const val SUFFIX = ".kmz"
        const val TEMPLATE_FILE = "template.kml"
        const val WAYLINE_FILE = "waylines.wpml"
        const val RES_IMAGE = "$KMZ_ROOT_PATH/res/image"
        const val RES_DSM = "$KMZ_ROOT_PATH/res/dsm"
        const val RES_AUDIO = "$KMZ_ROOT_PATH/res/audio"

        fun create(path: String, name: String): KMZPathWrapper {
            val cachePath = File(path)
            return KMZPathWrapper(cachePath.path, name)
        }

        fun parse(context: Context ,rootPath: String, kmzPath: String): KMZPathWrapper {
            val file = File(kmzPath)
            return if (file.isFile) {
                val kmzName = file.name.replace(SUFFIX, "")
                // 这里路径上不能有中文，否则上传图片的时候 CSDK 会失败
                var md5 = EncryptUtil.stringToMD5(kmzName)
                if (md5 == null) {
                    md5 = abs(kmzName.hashCode()).toString()
                }
                val cachePath = File("$rootPath/temp", md5)
                val pathWrapper = create(cachePath.path, kmzName)
                pathWrapper.unzipFullKmz(context ,kmzPath)
                pathWrapper

            } else {
                var kmzName = ""
                val childFile = file.listFiles { _: File?, name: String -> name.endsWith(SUFFIX) }
                if (childFile != null && childFile.isNotEmpty()) {
                    kmzName = childFile[0].name.replace(SUFFIX, "")
                }
                KMZPathWrapper(kmzPath, kmzName)
            }
        }
    }

}