package com.dji.wpmzsdk.common.utils.kml.transfrom

import com.dji.industry.pilot.data.cache.model.EdgePointModelGreenDao

import com.dji.industry.pilot.data.cache.model.StripMissionModelGreenDao
import com.dji.industry.pilot.data.cache.model.StripWaylineModelGreenDao

import com.dji.industry.pilot.mission.transfrom.*
import com.dji.wpmzsdk.common.utils.kml.GpsUtils
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraType
import com.dji.wpmzsdk.common.utils.kml.model.MissionInfoModelGreenDao

import dji.sdk.wpmz.value.mission.*

/**
 * Description :
 *
 * @filename : StripMissionEntityTransform
 * @author : devin.xu
 * @time : 2022/11/30
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class StripMissionEntityTransform {

    private val mFinishActionTransform = WaylineFinishActionTransform()
    private val mAltitudeModeTransform = MappingAltitudeModeTransform()
    private val mShootTypeTransform = MappingShootTypeTransform()
    private val mStripPlanModeTransform = StripPlanModeTransform()
    private val mFocusModeTransform = WaylineMappingFocusModeTransform()
    private val mEchoModeTransform = WaylineEchoModeTransform()
    private val mWaylineScanModeTransform = WaylineScanModeTransform()
    private val mMappingCameraTransform = MappingPayloadInfoTransform()
    private val mWaylinePaylodParamTransform = WaylinePaylodParamTransform()

    fun transFrom(model: StripMissionModelGreenDao): MissionTransformData {
        val stripWaylineModel = model.stripWayline!!
        val edgePointModels = model.edgePoints!!
        val template = WaylineTemplate()
        template.autoFlightSpeed = stripWaylineModel.speed.toDouble()
        template.transitionalSpeed = stripWaylineModel.takeoffSpeed.toDouble()
        template.coordinateParam = transCoordinateParamFrom(stripWaylineModel)
        template.payloadParam = transPayloadParamsFrom(stripWaylineModel)
        template.templateType = WaylineTemplateType.MAPPINGSTRIP
        template.mappingStripInfo = transStripTemplateFrom(stripWaylineModel, edgePointModels)
        heightModeCheck(model, template.mappingStripInfo)

        val missionConfig: WaylineMissionConfig = transConfigFrom(stripWaylineModel)
        val transformData = MissionTransformData()
        transformData.mission = createWaylineMission(model.missionInfo!!)
        transformData.setTemplate(template)
        transformData.missionConfig = transConfigFrom(stripWaylineModel)
        transformData.executeMissionConfig = createExecuteConfig(missionConfig)
        return transformData
    }

    private fun heightModeCheck(model: StripMissionModelGreenDao, template: WaylineTemplateMappingStripInfo) {
        template.ellipsoidHeight = model.stripWayline!!.altitude.toDouble()
        if (model.stripWayline!!.altitudeMode == 1 /*AltitudeMode.ABSOLUTE*/ && model.edgePoints != null && model.edgePoints!!.isNotEmpty()) {
            val firstPoint = model.edgePoints!![0]
            template.height = GpsUtils.egm96Altitude(model.stripWayline!!.altitude.toDouble(), firstPoint.latitude, firstPoint.longitude)
        }
        if (MappingCameraType.find(model.stripWayline!!.camera!!.name) != MappingCameraType.EP800_LIDAR) {
            template.caliFlightEnable = false
        }
    }

    private fun createWaylineMission(infoModel: MissionInfoModelGreenDao): WaylineMission {
        val mission = WaylineMission()
        mission.updateTime = infoModel.updateTime.toDouble()
        mission.createTime = infoModel.createTime.toDouble()
        return mission
    }

    private fun transStripTemplateFrom(
        stripWaylineModel: StripWaylineModelGreenDao,
        edgePointModels: List<EdgePointModelGreenDao>
    ): WaylineTemplateMappingStripInfo {
        val stripInfoTemplate = WaylineTemplateMappingStripInfo()
        stripInfoTemplate.boundaryOptimEnable =
            mStripPlanModeTransform.transFrom(stripWaylineModel.planMode)
        stripInfoTemplate.coordinates = transEdgePoint(edgePointModels)
        stripInfoTemplate.cuttingDistance = stripWaylineModel.cutLength.toDouble()
        stripInfoTemplate.includeCenterEnable = stripWaylineModel.hasCenterLine
        stripInfoTemplate.leftExtend = stripWaylineModel.leftExtend
        stripInfoTemplate.rightExtend = stripWaylineModel.rightExtend
        stripInfoTemplate.elevationOptimizeEnable = false
        stripInfoTemplate.height = stripWaylineModel.altitude.toDouble()
        stripInfoTemplate.ellipsoidHeight = stripWaylineModel.altitude.toDouble()
        val overlap = WaylineOverlap()
        overlap.isOrthoCameraOverlapWSet = true
        var overlapW = stripWaylineModel.overlapW
        if (isLidarCamera(stripWaylineModel.camera)) {
            overlapW = getVisibleOverlapWValue(stripWaylineModel.overlapW)
        }
        overlap.orthoCameraOverlapW = overlapW
        overlap.isOrthoCameraOverlapHSet = true
        overlap.orthoCameraOverlapH = stripWaylineModel.overlapH
        overlap.isOrthoLidarOverlapHSet = true
        overlap.orthoLidarOverlapH = stripWaylineModel.overlapH
        overlap.isOrthoLidarOverlapWSet = true
        overlap.orthoLidarOverlapW = stripWaylineModel.overlapW
        stripInfoTemplate.overlap = overlap
        stripInfoTemplate.boundaryOptimEnable =
            mStripPlanModeTransform.transFrom(stripWaylineModel.planMode)
        stripInfoTemplate.singleLineEnable = stripWaylineModel.enableSingleLine
        stripInfoTemplate.shootType = mShootTypeTransform.transFrom(stripWaylineModel.photoMode)
        stripInfoTemplate.isShootTypeSet = true
        stripInfoTemplate.caliFlightEnable = stripWaylineModel.enableCalibrate
        return stripInfoTemplate
    }

    fun transEdgePoint(points: List<EdgePointModelGreenDao>): List<WaylineLocationCoordinate3D> {
        val edgePoint: MutableList<WaylineLocationCoordinate3D> = java.util.ArrayList()
        for (item in points) {
            val location = WaylineLocationCoordinate3D()
            location.latitude = item.latitude
            location.longitude = item.longitude
            edgePoint.add(location)
        }
        return edgePoint
    }

    private fun transConfigFrom(waylineModel: StripWaylineModelGreenDao): WaylineMissionConfig {
        val config = WaylineMissionConfig()
        config.flyToWaylineMode = WaylineFlyToWaylineMode.SAFELY
        config.exitOnRCLostBehavior = WaylineExitOnRCLostBehavior.EXCUTE_RC_LOST_ACTION
        config.exitOnRCLostType = WaylineExitOnRCLostAction.GO_BACK
        config.globalTransitionalSpeed = waylineModel.takeoffSpeed.toDouble()
        config.isTakeOffPositionRefSet = false
        config.isGlobalRTHHeightSet = false
        config.finishAction = mFinishActionTransform.transFrom(waylineModel.actionOnFinish)

        val payloadInfos: MutableList<WaylinePayloadInfo> = ArrayList()
        if (waylineModel.camera != null) {
            val cameraType = MappingCameraType.find(waylineModel.camera!!.name)
            config.droneInfo = getDroneInfoFromCamera(cameraType)
            payloadInfos.add(mMappingCameraTransform.transPayloadInfoFrom(waylineModel.camera!!))
        }
        config.payloadInfo = payloadInfos
        return config
    }

    private fun transCoordinateParamFrom(waylineModel: StripWaylineModelGreenDao): WaylineCoordinateParam {
        val coordinateParam = WaylineCoordinateParam()
        coordinateParam.coordinateMode = WaylineCoordinateMode.WGS84
        coordinateParam.isWaylinePositioningTypeSet = false
        val isAbsolute = waylineModel.altitudeMode == MappingAltitudeModeTransform.ABSOLUTE
        val globalShootHeight = if (isAbsolute) waylineModel.relativeDistance.toDouble() else (waylineModel.altitude - waylineModel.relativeDistance).toDouble()
        coordinateParam.globalShootHeight = globalShootHeight
        coordinateParam.isGlobalShootHeightSet = true
        coordinateParam.altitudeMode = mAltitudeModeTransform.transFrom(waylineModel.altitudeMode)
        return coordinateParam
    }

    private fun transPayloadParamsFrom(waylineModel: StripWaylineModelGreenDao): List<WaylinePayloadParam> {
        val payloadParams: MutableList<WaylinePayloadParam> = ArrayList()
        val payloadParam = WaylinePayloadParam()
        payloadParam.focusMode = mFocusModeTransform.transFrom(waylineModel.focusMode)
        payloadParam.isDewarpingEnableSet = true
        payloadParam.dewarpingEnable = waylineModel.enableDewarping
        payloadParam.isModelColoringEnableSet = true
        payloadParam.modelColoringEnable = waylineModel.needVariegation
        payloadParam.returnMode = mEchoModeTransform.transFrom(waylineModel.echoMode)
        payloadParam.samplingRate =
            mWaylinePaylodParamTransform.transSampleRateFrom(
                waylineModel.sampleRate,
                payloadParam.returnMode
            )
        payloadParam.scanningMode = mWaylineScanModeTransform.transFrom(waylineModel.scanMode)
        if (waylineModel.camera != null) {
            val cameraType = MappingCameraType.find(waylineModel.camera!!.name)
            payloadParam.imageFormat = mWaylinePaylodParamTransform.transImageFormatFrom(cameraType)
            if (!MappingCameraType.isEP600(cameraType)) {
                payloadParam.focusMode = WaylinePayloadFocusMode.UNKNOWN
            }
        }
        payloadParams.add(payloadParam)
        return payloadParams
    }



}