/*******************************************************************************
 * (C) Copyright 2015 Haifeng Li
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/

package smile.sequence

import smile.data.Attribute
import smile.sequence.CRF.Trainer
import smile.util._

/** High level sequence annotation operators.
  *
  * @author Haifeng Li
  */
trait Operators {

  /** First-order Hidden Markov Model. A hidden Markov model (HMM) is a statistical
    * Markov model in which the system being modeled is assumed to be a Markov
    * process with unobserved (hidden) states. An HMM can be considered as the
    * simplest dynamic Bayesian network. <p> In a regular Markov model, the state
    * is directly visible to the observer, and therefore the state transition
    * probabilities are the only parameters. In a hidden Markov model, the state is
    * not directly visible, but output, dependent on the state, is visible. Each
    * state has a probability distribution over the possible output tokens.
    * Therefore the sequence of tokens generated by an HMM gives some information
    * about the sequence of states.
    *
    * @param observations the observation sequences, of which symbols take
    *                     values in [0, n), where n is the number of unique symbols.
    * @param labels the state labels of observations, of which states take
    *               values in [0, p), where p is the number of hidden states.
    */
  def hmm(observations: Array[Array[Int]], labels: Array[Array[Int]]): HMM[Int] = {
    time {
      new HMM[Int](observations, labels)
    }
  }

  /** Trains a first-order Hidden Markov Model.
    *
    * @param observations the observation sequences, of which symbols take
    *                     values in [0, n), where n is the number of unique symbols.
    * @param labels the state labels of observations, of which states take
    *               values in [0, p), where p is the number of hidden states.
    */
  def hmm[T <: Object](observations: Array[Array[T]], labels: Array[Array[Int]]): HMM[T] = {
    time {
      new HMM[T](observations, labels)
    }
  }

  /** First-order linear conditional random field. A conditional random field is a
    * type of discriminative undirected probabilistic graphical model. It is most
    * often used for labeling or parsing of sequential data.
    *
    * A CRF is a Markov random field that was trained discriminatively. Therefore it is not necessary
    * to model the distribution over always observed variables, which makes it
    * possible to include arbitrarily complicated features of the observed
    * variables into the model.
    *
    * ====References:====
    *  - J. Lafferty, A. McCallum and F. Pereira. Conditional random fields: Probabilistic models for segmenting and labeling sequence data. ICML, 2001.</li>
    *  - Thomas G. Dietterich, Guohua Hao, and Adam Ashenfelter. Gradient Tree Boosting for Training Conditional Random Fields. JMLR, 2008.
    *
    * @param sequences the observation attribute sequences.
    * @param labels sequence labels.
    * @param attributes the feature attributes.
    * @param k the number of classes.
    * @param eta the learning rate of potential function.
    * @param ntrees the number of trees/iterations.
    * @param maxNodes the maximum number of leaf nodes in the tree.
    */
  def crf(sequences: Array[Array[Array[Double]]], labels: Array[Array[Int]], attributes: Array[Attribute], k: Int, eta: Double = 1.0, ntrees: Int = 100, maxNodes: Int = 100): CRF = {
    time {
      val trainer = new Trainer(attributes, k)
      trainer.setLearningRate(eta)
        .setMaxNodes(maxNodes)
        .setNumTrees(ntrees)
      trainer.train(sequences, labels)
    }
  }
}