/*
 * Decompiled with CFR 0.152.
 */
package com.kdgregory.log4j.aws.internal.cloudwatch;

import com.amazonaws.services.logs.AWSLogsClient;
import com.amazonaws.services.logs.model.CreateLogGroupRequest;
import com.amazonaws.services.logs.model.CreateLogStreamRequest;
import com.amazonaws.services.logs.model.DescribeLogGroupsRequest;
import com.amazonaws.services.logs.model.DescribeLogGroupsResult;
import com.amazonaws.services.logs.model.DescribeLogStreamsRequest;
import com.amazonaws.services.logs.model.DescribeLogStreamsResult;
import com.amazonaws.services.logs.model.InputLogEvent;
import com.amazonaws.services.logs.model.LogGroup;
import com.amazonaws.services.logs.model.LogStream;
import com.amazonaws.services.logs.model.PutLogEventsRequest;
import com.kdgregory.log4j.aws.internal.shared.LogMessage;
import com.kdgregory.log4j.aws.internal.shared.LogWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import org.apache.log4j.helpers.LogLog;

public class CloudWatchLogWriter
implements LogWriter {
    private static final int RETRY_LIMIT = 3;
    private String groupName;
    private String streamName;
    private long batchDelay;
    private Thread dispatchThread;
    private AWSLogsClient client;
    private volatile Long shutdownTime;
    private volatile int batchCount;
    private LinkedBlockingDeque<LogMessage> messageQueue = new LinkedBlockingDeque();

    public CloudWatchLogWriter(String logGroup, String logStream, long batchDelay) {
        this.groupName = logGroup;
        this.streamName = logStream;
        this.batchDelay = batchDelay;
    }

    @Override
    public void addMessage(LogMessage message) {
        this.messageQueue.add(message);
    }

    @Override
    public void setBatchDelay(long value) {
        this.batchDelay = value;
    }

    @Override
    public void stop() {
        this.shutdownTime = new Long(System.currentTimeMillis() + this.batchDelay);
        if (this.dispatchThread != null) {
            this.dispatchThread.interrupt();
        }
    }

    @Override
    public void run() {
        this.client = new AWSLogsClient();
        if (!this.ensureGroupAndStreamAvailable()) {
            return;
        }
        this.dispatchThread = Thread.currentThread();
        do {
            List<LogMessage> currentBatch = this.buildBatch();
            this.attemptToSend(currentBatch);
        } while (this.keepRunning());
    }

    public long getBatchDelay() {
        return this.batchDelay;
    }

    public int getBatchCount() {
        return this.batchCount;
    }

    private boolean keepRunning() {
        return this.shutdownTime == null ? true : this.shutdownTime > System.currentTimeMillis() && this.messageQueue.peek() == null;
    }

    private List<LogMessage> buildBatch() {
        ArrayList<LogMessage> batch = new ArrayList<LogMessage>(512);
        long firstMessageTimeout = this.shutdownTime != null ? this.shutdownTime : Long.MAX_VALUE;
        LogMessage message = this.waitForMessage(firstMessageTimeout);
        if (message == null) {
            return batch;
        }
        long batchTimeout = System.currentTimeMillis() + this.batchDelay;
        int batchBytes = 0;
        int batchMsgs = 0;
        while (message != null) {
            if ((batchBytes += message.size() + 26) >= 0x100000 || ++batchMsgs == 10000) {
                this.messageQueue.addFirst(message);
                break;
            }
            batch.add(message);
            message = this.waitForMessage(batchTimeout);
        }
        return batch;
    }

    private LogMessage waitForMessage(long waitUntil) {
        try {
            long waitTime = waitUntil - System.currentTimeMillis();
            if (waitTime < 0L) {
                waitTime = 1L;
            }
            return this.messageQueue.poll(waitTime, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ex) {
            return null;
        }
    }

    private void attemptToSend(List<LogMessage> batch) {
        if (batch.isEmpty()) {
            return;
        }
        ++this.batchCount;
        PutLogEventsRequest request = new PutLogEventsRequest().withLogGroupName(this.groupName).withLogStreamName(this.streamName).withLogEvents(this.constructLogEvents(batch));
        Exception lastException = null;
        for (int attempt = 0; attempt < 3; ++attempt) {
            try {
                LogStream stream = this.findLogStream();
                request.setSequenceToken(stream.getUploadSequenceToken());
                this.client.putLogEvents(request);
                return;
            }
            catch (Exception ex) {
                lastException = ex;
                this.sleepQuietly(250 * (attempt + 1));
                continue;
            }
        }
        LogLog.error((String)"failed to send batch after 3 retries", lastException);
    }

    private List<InputLogEvent> constructLogEvents(List<LogMessage> batch) {
        Collections.sort(batch);
        ArrayList<InputLogEvent> result = new ArrayList<InputLogEvent>(batch.size());
        for (LogMessage msg : batch) {
            InputLogEvent event = new InputLogEvent().withTimestamp(Long.valueOf(msg.getTimestamp())).withMessage(msg.getMessage());
            result.add(event);
        }
        return result;
    }

    private boolean ensureGroupAndStreamAvailable() {
        LogLog.debug((String)"making log group and stream (this is first connection to AWS)");
        try {
            LogStream logStream;
            LogGroup logGroup = this.findLogGroup();
            if (logGroup == null) {
                CreateLogGroupRequest request = new CreateLogGroupRequest().withLogGroupName(this.groupName);
                this.client.createLogGroup(request);
            }
            if ((logStream = this.findLogStream()) == null) {
                CreateLogStreamRequest request = new CreateLogStreamRequest().withLogGroupName(this.groupName).withLogStreamName(this.streamName);
                this.client.createLogStream(request);
            }
            return true;
        }
        catch (Exception ex) {
            LogLog.error((String)"unable to configure log group/stream", (Throwable)ex);
            return false;
        }
    }

    private LogGroup findLogGroup() {
        DescribeLogGroupsRequest request = new DescribeLogGroupsRequest().withLogGroupNamePrefix(this.groupName);
        DescribeLogGroupsResult result = this.client.describeLogGroups(request);
        for (LogGroup group : result.getLogGroups()) {
            if (!group.getLogGroupName().equals(this.groupName)) continue;
            return group;
        }
        return null;
    }

    private LogStream findLogStream() {
        DescribeLogStreamsRequest request = new DescribeLogStreamsRequest().withLogGroupName(this.groupName).withLogStreamNamePrefix(this.streamName);
        DescribeLogStreamsResult result = this.client.describeLogStreams(request);
        for (LogStream stream : result.getLogStreams()) {
            if (!stream.getLogStreamName().equals(this.streamName)) continue;
            return stream;
        }
        return null;
    }

    private void sleepQuietly(long time) {
        try {
            Thread.sleep(time);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }
}

