//[okhttp](../../../index.md)/[okhttp3](../index.md)/[Authenticator](index.md)

# Authenticator

[jvm]\
fun interface [Authenticator](index.md)

Performs either **preemptive** authentication before connecting to a proxy server, or **reactive** authentication after receiving a challenge from either an origin web server or proxy server.

##  Preemptive Authentication

To make HTTPS calls using an HTTP proxy server OkHttp must first negotiate a connection with the proxy. This proxy connection is called a &quot;TLS Tunnel&quot; and is specified by [RFC 2817](https://tools.ietf.org/html/rfc2817). The HTTP CONNECT request that creates this tunnel connection is special: it does not participate in any [interceptors](../-interceptor/index.md) or [event listeners](../-event-listener/index.md). It doesn't include the motivating request's HTTP headers or even its full URL; only the target server's hostname is sent to the proxy.

Prior to sending any CONNECT request OkHttp always calls the proxy authenticator so that it may prepare preemptive authentication. OkHttp will call [authenticate](authenticate.md) with a fake `HTTP/1.1 407 Proxy Authentication Required` response that has a `Proxy-Authenticate: OkHttp-Preemptive` challenge. The proxy authenticator may return either an authenticated request, or null to connect without authentication.

```java
for (Challenge challenge : response.challenges()) {
  // If this is preemptive auth, use a preemptive credential.
  if (challenge.scheme().equalsIgnoreCase("OkHttp-Preemptive")) {
    return response.request().newBuilder()
        .header("Proxy-Authorization", "secret")
        .build();
  }
}
return null; // Didn't find a preemptive auth scheme.
```

##  Reactive Authentication

Implementations authenticate by returning a follow-up request that includes an authorization header, or they may decline the challenge by returning null. In this case the unauthenticated response will be returned to the caller that triggered it.

Implementations should check if the initial request already included an attempt to authenticate. If so it is likely that further attempts will not be useful and the authenticator should give up.

When reactive authentication is requested by an origin web server, the response code is 401 and the implementation should respond with a new request that sets the &quot;Authorization&quot; header.

```java
if (response.request().header("Authorization") != null) {
  return null; // Give up, we've already failed to authenticate.
}

String credential = Credentials.basic(...)
return response.request().newBuilder()
    .header("Authorization", credential)
    .build();
```

When reactive authentication is requested by a proxy server, the response code is 407 and the implementation should respond with a new request that sets the &quot;Proxy-Authorization&quot; header.

```java
if (response.request().header("Proxy-Authorization") != null) {
  return null; // Give up, we've already failed to authenticate.
}

String credential = Credentials.basic(...)
return response.request().newBuilder()
    .header("Proxy-Authorization", credential)
    .build();
```

The proxy authenticator may implement preemptive authentication, reactive authentication, or both.

Applications may configure OkHttp with an authenticator for origin servers, or proxy servers, or both.

##  Authentication Retries

If your authentication may be flaky and requires retries you should apply some policy to limit the retries by the class of errors and number of attempts.  To get the number of attempts to the current point use this function.

```java
private int responseCount(Response response) {
  int result = 1;
  while ((response = response.priorResponse()) != null) {
    result++;
  }
  return result;
}
```

## Types

| Name | Summary |
|---|---|
| [Companion](-companion/index.md) | [jvm]<br>object [Companion](-companion/index.md) |

## Functions

| Name | Summary |
|---|---|
| [authenticate](authenticate.md) | [jvm]<br>abstract fun [authenticate](authenticate.md)(route: [Route](../-route/index.md)?, response: [Response](../-response/index.md)): [Request](../-request/index.md)?<br>Returns a request that includes a credential to satisfy an authentication challenge in [response](authenticate.md). Returns null if the challenge cannot be satisfied. |
