package com.vaadin.pro.licensechecker;

import java.time.Instant;
import java.time.Period;
import java.time.temporal.TemporalAmount;
import java.util.prefs.Preferences;

import org.slf4j.Logger;

public class History {

    private static final String CHECK_PREFIX = "lastcheck-";

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    static boolean isRecentlyValidated(Product product, BuildType buildType) {
        return isRecentlyValidated(product, Period.ofDays(1), buildType);
    }

    static boolean isRecentlyValidated(Product product, TemporalAmount period, BuildType buildType) {
        getLogger().debug("Checking if license for " + product +
                " has recently been checked");
        Instant lastCheck = getLastCheckTime(product, buildType);
        if (lastCheck == null) {
            return false;
        }
        Instant now = Instant.now();
        if (lastCheck.isAfter(now)) {
            // Invalid last check value
            return false;
        }

        Instant nextCheck = lastCheck.plus(period);
        return now.isBefore(nextCheck);
    }

    public static Instant getLastCheckTime(Product product, BuildType buildType) {
        String lastCheckKey = getLastCheckKey(product, buildType);
        long lastCheck = getPreferences().getLong(lastCheckKey, -1);
        if (lastCheck == -1) {
            getLogger()
                    .debug("License for " + product + " has never been checked");
            return null;
        } else {
            Instant lastCheckInstant = Instant.ofEpochMilli(lastCheck);
            getLogger().debug("Last check for " + product + " was on " +
                    lastCheckInstant);
            return lastCheckInstant;
        }
    }

    public static String getLastSubscription(Product product) {
        String lastSubscriptionKey = getLastSubscriptionKey(product);
        return getPreferences().get(lastSubscriptionKey, "");
    }

    static long setLastCheckTimeNow(Product product, BuildType buildType) {
        getLogger().debug("Marking license for " + product + " as checked now for buildType " + buildType);
        return setLastCheck(product, Instant.now(), buildType);
    }

    static long setLastCheck(Product product, Instant instant, BuildType buildType) {
        getLogger()
                .debug("Marking license for " + product + " as checked on " + instant + " for buildType " + buildType);

        long timeMillis = instant.toEpochMilli();
        getPreferences().putLong(getLastCheckKey(product, buildType), timeMillis);
        return timeMillis;
    }

    public static String setLastSubscription(Product product,
            String subscription) {
        getLogger().debug("Storing subscription for " + product);

        getPreferences().put(getLastSubscriptionKey(product), subscription);
        return subscription;
    }

    private static String getLastCheckKey(Product product, BuildType buildType) {
        String base = CHECK_PREFIX + product.getName() + "-" + product.getVersion();
        if (buildType != null) {
            return base + "-" + buildType.getKey();
        } else {
            return base;
        }
    }

    private static String getLastSubscriptionKey(Product product) {
        return CHECK_PREFIX + product.getName() + "-" + product.getVersion() + "-subscription";
    }

    private static Preferences getPreferences() {
        return Preferences.userNodeForPackage(OnlineKeyValidator.class);
    }

    /**
     * Clears the local check history so that all products are checked again.
     */
    public static void clearAll() throws Exception {
        Preferences preferences = getPreferences();
        for (String key : preferences.keys()) {
            if (key.startsWith(CHECK_PREFIX)) {
                preferences.remove(key);
            }
        }
    }

}
