
package de.flapdoodle.embed.process.transitions;

import de.flapdoodle.embed.process.config.store.Package;
import de.flapdoodle.embed.process.distribution.Distribution;
import de.flapdoodle.embed.process.distribution.PackageResolver;
import de.flapdoodle.reverse.StateID;
import de.flapdoodle.reverse.naming.HasLabel;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link PackageOfDistribution}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePackageOfDistribution.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutablePackageOfDistribution
    extends PackageOfDistribution {
  private final String transitionLabel;
  private final PackageResolver packageResolver;
  private final StateID<Distribution> distribution;
  private final StateID<Package> destination;

  private ImmutablePackageOfDistribution(ImmutablePackageOfDistribution.Builder builder) {
    this.packageResolver = builder.packageResolver;
    if (builder.transitionLabel != null) {
      initShim.transitionLabel(builder.transitionLabel);
    }
    if (builder.distribution != null) {
      initShim.distribution(builder.distribution);
    }
    if (builder.destination != null) {
      initShim.destination(builder.destination);
    }
    this.transitionLabel = initShim.transitionLabel();
    this.distribution = initShim.distribution();
    this.destination = initShim.destination();
    this.initShim = null;
  }

  private ImmutablePackageOfDistribution(
      String transitionLabel,
      PackageResolver packageResolver,
      StateID<Distribution> distribution,
      StateID<Package> destination) {
    this.transitionLabel = transitionLabel;
    this.packageResolver = packageResolver;
    this.distribution = distribution;
    this.destination = destination;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte transitionLabelBuildStage = STAGE_UNINITIALIZED;
    private String transitionLabel;

    String transitionLabel() {
      if (transitionLabelBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (transitionLabelBuildStage == STAGE_UNINITIALIZED) {
        transitionLabelBuildStage = STAGE_INITIALIZING;
        this.transitionLabel = Objects.requireNonNull(ImmutablePackageOfDistribution.super.transitionLabel(), "transitionLabel");
        transitionLabelBuildStage = STAGE_INITIALIZED;
      }
      return this.transitionLabel;
    }

    void transitionLabel(String transitionLabel) {
      this.transitionLabel = transitionLabel;
      transitionLabelBuildStage = STAGE_INITIALIZED;
    }

    private byte distributionBuildStage = STAGE_UNINITIALIZED;
    private StateID<Distribution> distribution;

    StateID<Distribution> distribution() {
      if (distributionBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distributionBuildStage == STAGE_UNINITIALIZED) {
        distributionBuildStage = STAGE_INITIALIZING;
        this.distribution = Objects.requireNonNull(ImmutablePackageOfDistribution.super.distribution(), "distribution");
        distributionBuildStage = STAGE_INITIALIZED;
      }
      return this.distribution;
    }

    void distribution(StateID<Distribution> distribution) {
      this.distribution = distribution;
      distributionBuildStage = STAGE_INITIALIZED;
    }

    private byte destinationBuildStage = STAGE_UNINITIALIZED;
    private StateID<Package> destination;

    StateID<Package> destination() {
      if (destinationBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (destinationBuildStage == STAGE_UNINITIALIZED) {
        destinationBuildStage = STAGE_INITIALIZING;
        this.destination = Objects.requireNonNull(ImmutablePackageOfDistribution.super.destination(), "destination");
        destinationBuildStage = STAGE_INITIALIZED;
      }
      return this.destination;
    }

    void destination(StateID<Package> destination) {
      this.destination = destination;
      destinationBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (transitionLabelBuildStage == STAGE_INITIALIZING) attributes.add("transitionLabel");
      if (distributionBuildStage == STAGE_INITIALIZING) attributes.add("distribution");
      if (destinationBuildStage == STAGE_INITIALIZING) attributes.add("destination");
      return "Cannot build PackageOfDistribution, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code transitionLabel} attribute
   */
  @Override
  public String transitionLabel() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.transitionLabel()
        : this.transitionLabel;
  }

  /**
   * @return The value of the {@code packageResolver} attribute
   */
  @Override
  protected PackageResolver packageResolver() {
    return packageResolver;
  }

  /**
   * @return The value of the {@code distribution} attribute
   */
  @Override
  protected StateID<Distribution> distribution() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distribution()
        : this.distribution;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<Package> destination() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.destination()
        : this.destination;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfDistribution#transitionLabel() transitionLabel} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for transitionLabel
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePackageOfDistribution withTransitionLabel(String value) {
    String newValue = Objects.requireNonNull(value, "transitionLabel");
    if (this.transitionLabel.equals(newValue)) return this;
    return new ImmutablePackageOfDistribution(newValue, this.packageResolver, this.distribution, this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfDistribution#packageResolver() packageResolver} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for packageResolver
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePackageOfDistribution withPackageResolver(PackageResolver value) {
    if (this.packageResolver == value) return this;
    PackageResolver newValue = Objects.requireNonNull(value, "packageResolver");
    return new ImmutablePackageOfDistribution(this.transitionLabel, newValue, this.distribution, this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfDistribution#distribution() distribution} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distribution
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePackageOfDistribution withDistribution(StateID<Distribution> value) {
    if (this.distribution == value) return this;
    StateID<Distribution> newValue = Objects.requireNonNull(value, "distribution");
    return new ImmutablePackageOfDistribution(this.transitionLabel, this.packageResolver, newValue, this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfDistribution#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePackageOfDistribution withDestination(StateID<Package> value) {
    if (this.destination == value) return this;
    StateID<Package> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutablePackageOfDistribution(this.transitionLabel, this.packageResolver, this.distribution, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePackageOfDistribution} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePackageOfDistribution
        && equalsByValue((ImmutablePackageOfDistribution) another);
  }

  private boolean equalsByValue(ImmutablePackageOfDistribution another) {
    return transitionLabel.equals(another.transitionLabel)
        && packageResolver.equals(another.packageResolver)
        && distribution.equals(another.distribution)
        && destination.equals(another.destination);
  }

  /**
   * Computes a hash code from attributes: {@code transitionLabel}, {@code packageResolver}, {@code distribution}, {@code destination}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + transitionLabel.hashCode();
    h += (h << 5) + packageResolver.hashCode();
    h += (h << 5) + distribution.hashCode();
    h += (h << 5) + destination.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code PackageOfDistribution} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "PackageOfDistribution{"
        + "transitionLabel=" + transitionLabel
        + ", packageResolver=" + packageResolver
        + ", distribution=" + distribution
        + ", destination=" + destination
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link PackageOfDistribution} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable PackageOfDistribution instance
   */
  public static ImmutablePackageOfDistribution copyOf(PackageOfDistribution instance) {
    if (instance instanceof ImmutablePackageOfDistribution) {
      return (ImmutablePackageOfDistribution) instance;
    }
    return ImmutablePackageOfDistribution.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePackageOfDistribution ImmutablePackageOfDistribution}.
   * <pre>
   * ImmutablePackageOfDistribution.builder()
   *    .transitionLabel(String) // optional {@link PackageOfDistribution#transitionLabel() transitionLabel}
   *    .packageResolver(de.flapdoodle.embed.process.distribution.PackageResolver) // required {@link PackageOfDistribution#packageResolver() packageResolver}
   *    .distribution(de.flapdoodle.reverse.StateID&lt;de.flapdoodle.embed.process.distribution.Distribution&gt;) // optional {@link PackageOfDistribution#distribution() distribution}
   *    .destination(de.flapdoodle.reverse.StateID&lt;de.flapdoodle.embed.process.config.store.Package&gt;) // optional {@link PackageOfDistribution#destination() destination}
   *    .build();
   * </pre>
   * @return A new ImmutablePackageOfDistribution builder
   */
  public static ImmutablePackageOfDistribution.Builder builder() {
    return new ImmutablePackageOfDistribution.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePackageOfDistribution ImmutablePackageOfDistribution}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_PACKAGE_RESOLVER = 0x1L;
    private long initBits = 0x1L;

    private String transitionLabel;
    private PackageResolver packageResolver;
    private StateID<Distribution> distribution;
    private StateID<Package> destination;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.process.transitions.PackageOfDistribution} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(PackageOfDistribution instance) {
      Objects.requireNonNull(instance, "instance");
      mergeInternal(instance);
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.reverse.naming.HasLabel} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(HasLabel instance) {
      Objects.requireNonNull(instance, "instance");
      mergeInternal(instance);
      return this;
    }

    private void mergeInternal(Object object) {
      long bits = 0;
      if (object instanceof PackageOfDistribution) {
        PackageOfDistribution instance = (PackageOfDistribution) object;
        this.destination(instance.destination());
        this.distribution(instance.distribution());
        this.packageResolver(instance.packageResolver());
        if ((bits & 0x1L) == 0) {
          this.transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
      }
      if (object instanceof HasLabel) {
        HasLabel instance = (HasLabel) object;
        if ((bits & 0x1L) == 0) {
          this.transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
      }
    }

    /**
     * Initializes the value for the {@link PackageOfDistribution#transitionLabel() transitionLabel} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfDistribution#transitionLabel() transitionLabel}.</em>
     * @param transitionLabel The value for transitionLabel 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder transitionLabel(String transitionLabel) {
      this.transitionLabel = Objects.requireNonNull(transitionLabel, "transitionLabel");
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfDistribution#packageResolver() packageResolver} attribute.
     * @param packageResolver The value for packageResolver 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder packageResolver(PackageResolver packageResolver) {
      this.packageResolver = Objects.requireNonNull(packageResolver, "packageResolver");
      initBits &= ~INIT_BIT_PACKAGE_RESOLVER;
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfDistribution#distribution() distribution} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfDistribution#distribution() distribution}.</em>
     * @param distribution The value for distribution 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distribution(StateID<Distribution> distribution) {
      this.distribution = Objects.requireNonNull(distribution, "distribution");
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfDistribution#destination() destination} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfDistribution#destination() destination}.</em>
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(StateID<Package> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      return this;
    }

    /**
     * Builds a new {@link ImmutablePackageOfDistribution ImmutablePackageOfDistribution}.
     * @return An immutable instance of PackageOfDistribution
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePackageOfDistribution build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutablePackageOfDistribution(this);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_PACKAGE_RESOLVER) != 0) attributes.add("packageResolver");
      return "Cannot build PackageOfDistribution, some of required attributes are not set " + attributes;
    }
  }
}
