/*
 * Copyright (C) 2020 ActiveJ LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.activej.async.function;

import io.activej.common.function.RunnableEx;
import io.activej.promise.Promise;

import static io.activej.common.exception.FatalErrorHandler.handleError;

@FunctionalInterface
public interface AsyncRunnable extends AsyncRunnableEx {
	@Override
	Promise<Void> run();

	/**
	 * Wraps a {@link RunnableEx} interface.
	 *
	 * @param runnable a {@link RunnableEx}
	 * @return {@link AsyncRunnable} that works on top of {@link RunnableEx} interface
	 */
	static AsyncRunnable of(RunnableEx runnable) {
		return () -> {
			try {
				runnable.run();
			} catch (Exception e) {
				handleError(e, runnable);
				return Promise.ofException(e);
			}
			return Promise.complete();
		};
	}

	static AsyncRunnable sanitize(AsyncRunnableEx runnable) {
		return () -> {
			try {
				return runnable.run();
			} catch (Exception e) {
				handleError(e, runnable);
				return Promise.ofException(e);
			}
		};
	}
}
