/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */
package io.camunda.identity.sdk.impl;

import static io.camunda.identity.sdk.utility.UrlUtility.combinePaths;

import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.authorizations.Authorizations;
import io.camunda.identity.sdk.authorizations.dto.AssignAuthorization;
import io.camunda.identity.sdk.authorizations.dto.Authorization;
import io.camunda.identity.sdk.authorizations.dto.UpdateAuthorizations;
import io.camunda.identity.sdk.impl.rest.RestClient;
import io.camunda.identity.sdk.impl.rest.request.AssignAuthorizationRequest;
import io.camunda.identity.sdk.impl.rest.request.AuthorizationRequest;
import io.camunda.identity.sdk.impl.rest.request.UpdateAuthorizationsRequest;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class AuthorizationsImpl implements Authorizations {
  static final String AUTHORIZATIONS_PATH = "/api/authorizations";
  static final String FOR_TOKEN_PATH = "/for-token";

  private final String baseUrl;
  private final RestClient restClient;

  public AuthorizationsImpl(
      final IdentityConfiguration identityConfiguration
  ) {
    this(identityConfiguration.getBaseUrl());
  }

  private AuthorizationsImpl(
      final String baseUrl
  ) {
    this.baseUrl = baseUrl;
    this.restClient = new RestClient();
  }

  @Override
  public List<Authorization> forToken(final String token) {
    return restClient.request(
        new AuthorizationRequest(
            combinePaths(baseUrl, AUTHORIZATIONS_PATH, FOR_TOKEN_PATH),
            token,
            Collections.emptyMap()
        )
    );
  }

  @Override
  public List<Authorization> forToken(final String token, final String organizationId) {
    return restClient.request(
        new AuthorizationRequest(
            combinePaths(baseUrl, AUTHORIZATIONS_PATH, FOR_TOKEN_PATH),
            token,
            Map.of("organizationId", organizationId)
        )
    );
  }

  @Override
  public void assignAuthorization(
      final String token,
      final String entityId,
      final String entityType,
      final String organizationId,
      final String resourceKey,
      final String resourceType,
      final Set<String> requestedPermissions
  ) {
    requestedPermissions
        .forEach(permission -> {
          restClient.request(
              new AssignAuthorizationRequest(
                  combinePaths(baseUrl, AUTHORIZATIONS_PATH),
                  token,
                  new AssignAuthorization(
                      entityId,
                      entityType,
                      organizationId,
                      resourceKey,
                      resourceType,
                      permission
                  )
              )
          );
        });
  }

  @Override
  public void update(final String token, final UpdateAuthorizations updateAuthorizations) {
    restClient.request(
        new UpdateAuthorizationsRequest(
            combinePaths(baseUrl, AUTHORIZATIONS_PATH),
            token,
            updateAuthorizations
        )
    );
  }
}
