/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.metrics;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.TimerTask;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.DelegatingConfiguration;
import org.apache.flink.metrics.Metric;
import org.apache.flink.metrics.MetricConfig;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.metrics.reporter.MetricReporter;
import org.apache.flink.metrics.reporter.Scheduled;
import org.apache.flink.runtime.metrics.scope.ScopeFormat;
import org.apache.flink.runtime.metrics.scope.ScopeFormats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricRegistry {
    static final Logger LOG = LoggerFactory.getLogger(MetricRegistry.class);
    private List<MetricReporter> reporters;
    private ScheduledExecutorService executor;
    private final ScopeFormats scopeFormats;
    private final char delimiter;

    public MetricRegistry(Configuration config) {
        int delim;
        ScopeFormats scopeFormats;
        try {
            scopeFormats = MetricRegistry.createScopeConfig(config);
        }
        catch (Exception e) {
            LOG.warn("Failed to parse scope format, using default scope formats", (Throwable)e);
            scopeFormats = new ScopeFormats();
        }
        this.scopeFormats = scopeFormats;
        try {
            delim = config.getString("metrics.scope.delimiter", ".").charAt(0);
        }
        catch (Exception e) {
            LOG.warn("Failed to parse delimiter, using default delimiter.", (Throwable)e);
            delim = 46;
        }
        this.delimiter = (char)delim;
        this.reporters = new ArrayList<MetricReporter>();
        String definedReporters = config.getString("metrics.reporters", null);
        if (definedReporters == null) {
            LOG.info("No metrics reporter configured, no metrics will be exposed/reported.");
            this.executor = null;
        } else {
            String[] namedReporters;
            for (String namedReporter : namedReporters = definedReporters.split("\\s*,\\s*")) {
                DelegatingConfiguration reporterConfig = new DelegatingConfiguration(config, "metrics.reporter." + namedReporter + ".");
                String className = reporterConfig.getString("class", null);
                if (className == null) {
                    LOG.error("No reporter class set for reporter " + namedReporter + ". Metrics might not be exposed/reported.");
                    continue;
                }
                try {
                    String configuredPeriod = reporterConfig.getString("interval", null);
                    TimeUnit timeunit = TimeUnit.SECONDS;
                    long period = 10L;
                    if (configuredPeriod != null) {
                        try {
                            String[] interval = configuredPeriod.split(" ");
                            period = Long.parseLong(interval[0]);
                            timeunit = TimeUnit.valueOf(interval[1]);
                        }
                        catch (Exception e) {
                            LOG.error("Cannot parse report interval from config: " + configuredPeriod + " - please use values like '10 SECONDS' or '500 MILLISECONDS'. Using default reporting interval.");
                        }
                    }
                    Class<?> reporterClass = Class.forName(className);
                    MetricReporter reporterInstance = (MetricReporter)reporterClass.newInstance();
                    MetricConfig metricConfig = new MetricConfig();
                    reporterConfig.addAllToProperties((Properties)metricConfig);
                    reporterInstance.open(metricConfig);
                    if (reporterInstance instanceof Scheduled) {
                        if (this.executor == null) {
                            this.executor = Executors.newSingleThreadScheduledExecutor();
                        }
                        LOG.info("Periodically reporting metrics in intervals of {} {} for reporter {} of type {}.", new Object[]{period, timeunit.name(), namedReporter, className});
                        this.executor.scheduleWithFixedDelay(new ReporterTask((Scheduled)reporterInstance), period, period, timeunit);
                    }
                    this.reporters.add(reporterInstance);
                }
                catch (Throwable t) {
                    this.shutdownExecutor();
                    LOG.error("Could not instantiate metrics reporter" + namedReporter + ". Metrics might not be exposed/reported.", t);
                }
            }
        }
    }

    public char getDelimiter() {
        return this.delimiter;
    }

    public List<MetricReporter> getReporters() {
        return this.reporters;
    }

    public void shutdown() {
        if (this.reporters != null) {
            for (MetricReporter reporter : this.reporters) {
                try {
                    reporter.close();
                }
                catch (Throwable t) {
                    LOG.warn("Metrics reporter did not shut down cleanly", t);
                }
            }
            this.reporters = null;
        }
        this.shutdownExecutor();
    }

    private void shutdownExecutor() {
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(1L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
            }
        }
    }

    public ScopeFormats getScopeFormats() {
        return this.scopeFormats;
    }

    public void register(Metric metric, String metricName, MetricGroup group) {
        try {
            if (this.reporters != null) {
                for (MetricReporter reporter : this.reporters) {
                    if (reporter == null) continue;
                    reporter.notifyOfAddedMetric(metric, metricName, group);
                }
            }
        }
        catch (Exception e) {
            LOG.error("Error while registering metric.", (Throwable)e);
        }
    }

    public void unregister(Metric metric, String metricName, MetricGroup group) {
        try {
            if (this.reporters != null) {
                for (MetricReporter reporter : this.reporters) {
                    if (reporter == null) continue;
                    reporter.notifyOfRemovedMetric(metric, metricName, group);
                }
            }
        }
        catch (Exception e) {
            LOG.error("Error while registering metric.", (Throwable)e);
        }
    }

    static ScopeFormats createScopeConfig(Configuration config) {
        String jmFormat = config.getString("metrics.scope.jm", ScopeFormat.DEFAULT_SCOPE_JOBMANAGER_GROUP);
        String jmJobFormat = config.getString("metrics.scope.jm.job", ScopeFormat.DEFAULT_SCOPE_JOBMANAGER_JOB_GROUP);
        String tmFormat = config.getString("metrics.scope.tm", ScopeFormat.DEFAULT_SCOPE_TASKMANAGER_GROUP);
        String tmJobFormat = config.getString("metrics.scope.tm.job", ScopeFormat.DEFAULT_SCOPE_TASKMANAGER_JOB_GROUP);
        String taskFormat = config.getString("metrics.scope.task", ScopeFormat.DEFAULT_SCOPE_TASK_GROUP);
        String operatorFormat = config.getString("metrics.scope.operator", ScopeFormat.DEFAULT_SCOPE_OPERATOR_GROUP);
        return new ScopeFormats(jmFormat, jmJobFormat, tmFormat, tmJobFormat, taskFormat, operatorFormat);
    }

    private static final class ReporterTask
    extends TimerTask {
        private final Scheduled reporter;

        private ReporterTask(Scheduled reporter) {
            this.reporter = reporter;
        }

        @Override
        public void run() {
            try {
                this.reporter.report();
            }
            catch (Throwable t) {
                LOG.warn("Error while reporting metrics", t);
            }
        }
    }
}

