/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.jobgraph.JobStatus;
import org.apache.flink.runtime.state.RetrievableStateHandle;
import org.apache.flink.runtime.zookeeper.RetrievableStateStorageHelper;
import org.apache.flink.runtime.zookeeper.ZooKeeperStateHandleStore;
import org.apache.flink.shaded.org.apache.curator.framework.CuratorFramework;
import org.apache.flink.shaded.org.apache.curator.framework.api.BackgroundCallback;
import org.apache.flink.shaded.org.apache.curator.framework.api.CuratorEvent;
import org.apache.flink.shaded.org.apache.curator.framework.api.CuratorEventType;
import org.apache.flink.shaded.org.apache.curator.utils.ZKPaths;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooKeeperCompletedCheckpointStore
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(ZooKeeperCompletedCheckpointStore.class);
    private final CuratorFramework client;
    private final ZooKeeperStateHandleStore<CompletedCheckpoint> checkpointsInZooKeeper;
    private final int maxNumberOfCheckpointsToRetain;
    private final ArrayDeque<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> checkpointStateHandles;

    public ZooKeeperCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, CuratorFramework client, String checkpointsPath, RetrievableStateStorageHelper<CompletedCheckpoint> stateStorage, Executor executor) throws Exception {
        Preconditions.checkArgument((maxNumberOfCheckpointsToRetain >= 1 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint.");
        Preconditions.checkNotNull(stateStorage, (String)"State storage");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        Preconditions.checkNotNull((Object)client, (String)"Curator client");
        Preconditions.checkNotNull((Object)checkpointsPath, (String)"Checkpoints path");
        client.newNamespaceAwareEnsurePath(checkpointsPath).ensure(client.getZookeeperClient());
        this.client = client.usingNamespace(client.getNamespace() + checkpointsPath);
        this.checkpointsInZooKeeper = new ZooKeeperStateHandleStore<CompletedCheckpoint>(this.client, stateStorage, executor);
        this.checkpointStateHandles = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        LOG.info("Initialized in '{}'.", (Object)checkpointsPath);
    }

    @Override
    public void recover() throws Exception {
        List<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> initialCheckpoints;
        LOG.info("Recovering checkpoints from ZooKeeper.");
        this.checkpointStateHandles.clear();
        while (true) {
            try {
                initialCheckpoints = this.checkpointsInZooKeeper.getAllSortedByName();
            }
            catch (ConcurrentModificationException e) {
                LOG.warn("Concurrent modification while reading from ZooKeeper. Retrying.");
                continue;
            }
            break;
        }
        int numberOfInitialCheckpoints = initialCheckpoints.size();
        LOG.info("Found {} checkpoints in ZooKeeper.", (Object)numberOfInitialCheckpoints);
        for (Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> checkpoint : initialCheckpoints) {
            this.checkpointStateHandles.add(checkpoint);
        }
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint) throws Exception {
        Preconditions.checkNotNull((Object)checkpoint, (String)"Checkpoint");
        String path = ZooKeeperCompletedCheckpointStore.checkpointIdToPath(checkpoint.getCheckpointID());
        RetrievableStateHandle<CompletedCheckpoint> stateHandle = this.checkpointsInZooKeeper.add(path, checkpoint);
        this.checkpointStateHandles.addLast((Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>)new Tuple2(stateHandle, (Object)path));
        while (this.checkpointStateHandles.size() > this.maxNumberOfCheckpointsToRetain) {
            this.removeSubsumed(this.checkpointStateHandles.removeFirst());
        }
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    @Override
    public CompletedCheckpoint getLatestCheckpoint() {
        if (this.checkpointStateHandles.isEmpty()) {
            return null;
        }
        while (!this.checkpointStateHandles.isEmpty()) {
            Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> checkpointStateHandle = this.checkpointStateHandles.peekLast();
            try {
                return ZooKeeperCompletedCheckpointStore.retrieveCompletedCheckpoint(checkpointStateHandle);
            }
            catch (Exception e) {
                LOG.warn("Could not retrieve latest checkpoint. Removing it from the completed checkpoint store.", (Throwable)e);
                try {
                    this.removeBrokenStateHandle(this.checkpointStateHandles.pollLast());
                }
                catch (Exception removeException) {
                    LOG.warn("Could not remove the latest checkpoint with a broken state handle.", (Throwable)removeException);
                }
            }
        }
        return null;
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() throws Exception {
        ArrayList<CompletedCheckpoint> checkpoints = new ArrayList<CompletedCheckpoint>(this.checkpointStateHandles.size());
        Iterator<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> stateHandleIterator = this.checkpointStateHandles.iterator();
        while (stateHandleIterator.hasNext()) {
            Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandlePath = stateHandleIterator.next();
            try {
                checkpoints.add(ZooKeeperCompletedCheckpointStore.retrieveCompletedCheckpoint(stateHandlePath));
            }
            catch (Exception e) {
                LOG.warn("Could not retrieve checkpoint. Removing it from the completed checkpoint store.", (Throwable)e);
                stateHandleIterator.remove();
                this.removeBrokenStateHandle(stateHandlePath);
            }
        }
        return checkpoints;
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.checkpointStateHandles.size();
    }

    @Override
    public void shutdown(JobStatus jobStatus) throws Exception {
        if (jobStatus.isGloballyTerminalState()) {
            LOG.info("Shutting down");
            for (Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> checkpoint : this.checkpointStateHandles) {
                try {
                    this.removeShutdown(checkpoint, jobStatus);
                }
                catch (Exception e) {
                    LOG.error("Failed to discard checkpoint.", (Throwable)e);
                }
            }
            this.checkpointStateHandles.clear();
            String path = "/" + this.client.getNamespace();
            LOG.info("Removing {} from ZooKeeper", (Object)path);
            ZKPaths.deleteChildren(this.client.getZookeeperClient().getZooKeeper(), path, true);
        } else {
            LOG.info("Suspending");
            this.checkpointStateHandles.clear();
        }
    }

    private void removeSubsumed(final Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandleAndPath) throws Exception {
        Callable<Void> action = new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                ((CompletedCheckpoint)((RetrievableStateHandle)stateHandleAndPath.f0).retrieveState()).subsume();
                return null;
            }
        };
        this.remove(stateHandleAndPath, action);
    }

    private void removeShutdown(final Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandleAndPath, final JobStatus jobStatus) throws Exception {
        Callable<Void> action = new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                CompletedCheckpoint checkpoint = (CompletedCheckpoint)((RetrievableStateHandle)stateHandleAndPath.f0).retrieveState();
                checkpoint.discard(jobStatus);
                return null;
            }
        };
        this.remove(stateHandleAndPath, action);
    }

    private void removeBrokenStateHandle(Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandleAndPath) throws Exception {
        this.remove(stateHandleAndPath, null);
    }

    private void remove(final Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandleAndPath, final Callable<Void> action) throws Exception {
        BackgroundCallback callback = new BackgroundCallback(){

            @Override
            public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
                block11: {
                    long checkpointId = ZooKeeperCompletedCheckpointStore.pathToCheckpointId((String)stateHandleAndPath.f1);
                    try {
                        if (event.getType() == CuratorEventType.DELETE) {
                            if (event.getResultCode() == 0) {
                                Exception exception = null;
                                if (null != action) {
                                    try {
                                        action.call();
                                    }
                                    catch (Exception e) {
                                        exception = new Exception("Could not execute callable action for checkpoint " + checkpointId + '.', e);
                                    }
                                }
                                try {
                                    ((RetrievableStateHandle)stateHandleAndPath.f0).discardState();
                                }
                                catch (Exception e) {
                                    Exception newException = new Exception("Could not discard meta data for completed checkpoint " + checkpointId + '.', e);
                                    if (exception == null) {
                                        exception = newException;
                                    }
                                    exception.addSuppressed(newException);
                                }
                                if (exception != null) {
                                    throw exception;
                                }
                                break block11;
                            }
                            throw new IllegalStateException("Unexpected result code " + event.getResultCode() + " in '" + event + "' callback.");
                        }
                        throw new IllegalStateException("Unexpected event type " + (Object)((Object)event.getType()) + " in '" + event + "' callback.");
                    }
                    catch (Exception e) {
                        LOG.warn("Failed to discard checkpoint {}.", (Object)checkpointId, (Object)e);
                    }
                }
            }
        };
        this.checkpointsInZooKeeper.remove((String)stateHandleAndPath.f1, callback);
    }

    protected static String checkpointIdToPath(long checkpointId) {
        return String.format("/%s", checkpointId);
    }

    protected static long pathToCheckpointId(String path) {
        try {
            String numberString = '/' == path.charAt(0) ? path.substring(1) : path;
            return Long.parseLong(numberString);
        }
        catch (NumberFormatException e) {
            LOG.warn("Could not parse checkpoint id from {}. This indicates that the checkpoint id to path conversion has changed.", (Object)path);
            return -1L;
        }
    }

    private static CompletedCheckpoint retrieveCompletedCheckpoint(Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandlePath) throws Exception {
        long checkpointId = ZooKeeperCompletedCheckpointStore.pathToCheckpointId((String)stateHandlePath.f1);
        LOG.info("Trying to retrieve checkpoint {}.", (Object)checkpointId);
        try {
            return (CompletedCheckpoint)((RetrievableStateHandle)stateHandlePath.f0).retrieveState();
        }
        catch (Exception e) {
            throw new Exception("Could not retrieve checkpoint " + checkpointId + ". The state handle seems to be broken.", e);
        }
    }
}

