/*
 * Copyright (c) 2021, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.graalvm.nativebridge;

import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.LongBinaryOperator;
import java.util.function.LongUnaryOperator;
import java.util.function.Supplier;

import org.graalvm.collections.Pair;

/**
 * A configuration used by the {@link NativeIsolate} and classes generated by the native bridge
 * processor.
 *
 * @see GenerateHotSpotToNativeBridge
 * @see GenerateHotSpotToNativeBridge
 */
public final class JNIConfig {

    private final Map<Type, BinaryMarshaller<?>> binaryMarshallers;
    private final Map<Class<? extends Annotation>, List<Pair<Class<?>, BinaryMarshaller<?>>>> annotationBinaryMarshallers;
    private final LongUnaryOperator attachThreadAction;
    private final LongUnaryOperator detachThreadAction;
    private final LongBinaryOperator shutDownIsolateAction;
    private final LongBinaryOperator releaseNativeObjectAction;
    private final Function<Supplier<NativeIsolateThread>, ThreadLocal<NativeIsolateThread>> threadLocalFactory;

    JNIConfig(Map<Type, BinaryMarshaller<?>> binaryMarshallers,
                    Map<Class<? extends Annotation>, List<Pair<Class<?>, BinaryMarshaller<?>>>> annotationBinaryMarshallers,
                    LongUnaryOperator attachThreadAction, LongUnaryOperator detachThreadAction,
                    LongBinaryOperator shutDownIsolateAction, LongBinaryOperator releaseNativeObjectAction,
                    Function<Supplier<NativeIsolateThread>, ThreadLocal<NativeIsolateThread>> threadLocalFactory) {
        this.binaryMarshallers = binaryMarshallers;
        this.annotationBinaryMarshallers = annotationBinaryMarshallers;
        this.attachThreadAction = attachThreadAction;
        this.detachThreadAction = detachThreadAction;
        this.shutDownIsolateAction = shutDownIsolateAction;
        this.releaseNativeObjectAction = releaseNativeObjectAction;
        this.threadLocalFactory = threadLocalFactory;
    }

    /**
     * Looks up {@link BinaryMarshaller} for the {@code type} and {@code annotationTypes}. The
     * method first tries to find a marshaller registered for the {@code type} and some annotation
     * from {@code annotationTypes}. If no such marshaller exists, it tries to find a marshaller
     * registered just for the {@code type}. If there is no such a marshaller it throws the
     * {@link UnsupportedOperationException}.
     *
     * @param type the parameter or return type.
     * @param annotationTypes parameter or method annotation types.
     * @throws UnsupportedOperationException if there is no registered marshaller for the
     *             {@code type}.
     */
    @SuppressWarnings("unchecked")
    @SafeVarargs
    public final <T> BinaryMarshaller<T> lookupMarshaller(Class<T> type, Class<? extends Annotation>... annotationTypes) {
        BinaryMarshaller<?> res = lookupBinaryMarshallerImpl(type, annotationTypes);
        if (res != null) {
            return (BinaryMarshaller<T>) res;
        } else {
            throw unsupported(type);
        }
    }

    /**
     * Looks up {@link BinaryMarshaller} for the {@code parameterizedType} and
     * {@code annotationTypes}. The method first tries to find a marshaller registered for the
     * {@code parameterizedType} and some annotation from {@code annotationTypes}. If no such
     * marshaller exists, it tries to find a marshaller registered just for the
     * {@code parameterizedType}. If there is no such a marshaller it throws the
     * {@link UnsupportedOperationException}.
     *
     * @param parameterizedType the parameter or return type.
     * @param annotationTypes parameter or method annotation types.
     * @throws UnsupportedOperationException if there is no registered marshaller for the
     *             {@code parameterizedType}.
     */
    @SuppressWarnings("unchecked")
    @SafeVarargs
    public final <T> BinaryMarshaller<T> lookupMarshaller(TypeLiteral<T> parameterizedType, Class<? extends Annotation>... annotationTypes) {
        BinaryMarshaller<?> res = lookupBinaryMarshallerImpl(parameterizedType.getType(), annotationTypes);
        if (res != null) {
            return (BinaryMarshaller<T>) res;
        } else {
            throw unsupported(parameterizedType.getType());
        }
    }

    Function<Supplier<NativeIsolateThread>, ThreadLocal<NativeIsolateThread>> getThreadLocalFactory() {
        return threadLocalFactory;
    }

    long attachThread(long isolate) {
        return attachThreadAction.applyAsLong(isolate);
    }

    boolean detachThread(long isolateThread) {
        return detachThreadAction.applyAsLong(isolateThread) == 0;
    }

    boolean releaseNativeObject(long isolateThread, long handle) {
        return releaseNativeObjectAction.applyAsLong(isolateThread, handle) == 0;
    }

    boolean shutDownIsolate(long isolate, long isolateThread) {
        return shutDownIsolateAction.applyAsLong(isolate, isolateThread) == 0;
    }

    private static RuntimeException unsupported(Type type) {
        throw new UnsupportedOperationException(String.format("Marshalling of %s is not supported", type));
    }

    @SafeVarargs
    private final BinaryMarshaller<?> lookupBinaryMarshallerImpl(Type type, Class<? extends Annotation>... annotationTypes) {
        for (Class<? extends Annotation> annotationType : annotationTypes) {
            verifyAnnotation(annotationType);
            BinaryMarshaller<?> res = lookup(annotationBinaryMarshallers, type, annotationType);
            if (res != null) {
                return res;
            }
        }
        return binaryMarshallers.get(type);
    }

    private static <T> T lookup(Map<Class<? extends Annotation>, List<Pair<Class<?>, T>>> marshallers, Type type, Class<? extends Annotation> annotationType) {
        List<Pair<Class<?>, T>> marshallersForAnnotation = marshallers.get(annotationType);
        if (marshallersForAnnotation != null) {
            Class<?> rawType = erasure(type);
            for (Pair<Class<?>, T> marshaller : marshallersForAnnotation) {
                if (marshaller.getLeft().isAssignableFrom(rawType)) {
                    return marshaller.getRight();
                }
            }
        }
        return null;
    }

    private static Class<?> erasure(Type type) {
        if (type instanceof Class) {
            return (Class<?>) type;
        } else if (type instanceof ParameterizedType) {
            return (Class<?>) ((ParameterizedType) type).getRawType();
        } else if (type instanceof GenericArrayType) {
            return arrayTypeFromComponentType(erasure(((GenericArrayType) type).getGenericComponentType()));
        } else {
            throw new IllegalArgumentException("Unsupported type: " + type);
        }
    }

    private static Class<?> arrayTypeFromComponentType(Class<?> componentType) {
        return Array.newInstance(componentType, 0).getClass();
    }

    private static void verifyAnnotation(Class<? extends Annotation> annotationType) {
        if (annotationType.getAnnotation(MarshallerAnnotation.class) == null) {
            throw new IllegalArgumentException(String.format("The %s in not a valid marshaller annotation. The marshaller annotation must be annotated by the %s meta-annotation.",
                            annotationType.getSimpleName(), MarshallerAnnotation.class.getSimpleName()));
        }
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    /**
     * A builder class to construct {@link JNIConfig} instances.
     */
    public static final class Builder {

        private static final LongUnaryOperator ATTACH_UNSUPPORTED = (isolate) -> {
            throw new UnsupportedOperationException("Attach is not supported.");
        };
        private static final LongUnaryOperator DETACH_UNSUPPORTED = (isolateThread) -> {
            throw new UnsupportedOperationException("Detach is not supported.");
        };
        private static final LongBinaryOperator SHUTDOWN_UNSUPPORTED = (isolate, isolateThread) -> {
            throw new UnsupportedOperationException("Isolate shutdown is not supported.");
        };
        private static final LongBinaryOperator RELEASE_UNSUPPORTED = (isolateThread, handle) -> {
            throw new UnsupportedOperationException("Native object clean up is not supported.");
        };

        private final Map<Type, BinaryMarshaller<?>> binaryMarshallers;
        private final Map<Class<? extends Annotation>, List<Pair<Class<?>, BinaryMarshaller<?>>>> annotationBinaryMarshallers;
        private LongUnaryOperator attachThreadAction = ATTACH_UNSUPPORTED;
        private LongUnaryOperator detachThreadAction = DETACH_UNSUPPORTED;
        private LongBinaryOperator shutDownIsolateAction = SHUTDOWN_UNSUPPORTED;
        private LongBinaryOperator releaseNativeObjectAction = RELEASE_UNSUPPORTED;
        private Function<Supplier<NativeIsolateThread>, ThreadLocal<NativeIsolateThread>> threadLocalFactory = ThreadLocal::withInitial;

        Builder() {
            this.binaryMarshallers = new HashMap<>();
            this.annotationBinaryMarshallers = new HashMap<>();
            // Register default marshallers
            this.binaryMarshallers.put(String.class, BinaryMarshaller.nullable(new StringMarshaller()));
            this.binaryMarshallers.put(Throwable.class, new DefaultThrowableMarshaller());
            this.binaryMarshallers.put(StackTraceElement.class, StackTraceElementMarshaller.INSTANCE);
        }

        /**
         * Registers a {@link BinaryMarshaller} for the {@code type}.
         *
         * @param type the type to register {@link BinaryMarshaller} for.
         * @param marshaller the marshaller to register.
         */
        public <T> Builder registerMarshaller(Class<T> type, BinaryMarshaller<T> marshaller) {
            Objects.requireNonNull(type, "Type must be non null.");
            Objects.requireNonNull(marshaller, "Marshaller must be non null.");
            this.binaryMarshallers.put(type, marshaller);
            return this;
        }

        /**
         * Registers a {@link BinaryMarshaller} for the {@code parameterizedType}.
         *
         * @param parameterizedType the type to register {@link BinaryMarshaller} for.
         * @param marshaller the marshaller to register.
         */
        public <T> Builder registerMarshaller(TypeLiteral<T> parameterizedType, BinaryMarshaller<T> marshaller) {
            Objects.requireNonNull(parameterizedType, "ParameterizedType must be non null.");
            Objects.requireNonNull(marshaller, "Marshaller must be non null.");
            this.binaryMarshallers.put(parameterizedType.getType(), marshaller);
            return this;
        }

        /**
         * Registers a {@link BinaryMarshaller} for the {@code type} and {@code annotationType}.
         *
         * @param type the type to register {@link BinaryMarshaller} for.
         * @param annotationType a required annotation to look up the marshaller.
         * @param marshaller the marshaller to register.
         *
         */
        public <T> Builder registerMarshaller(Class<T> type, Class<? extends Annotation> annotationType, BinaryMarshaller<T> marshaller) {
            Objects.requireNonNull(type, "Type must be non null.");
            Objects.requireNonNull(annotationType, "AnnotationType must be non null.");
            Objects.requireNonNull(marshaller, "Marshaller must be non null.");
            insert(annotationBinaryMarshallers, type, annotationType, marshaller);
            return this;
        }

        /**
         * Registers a {@link BinaryMarshaller} for the {@code parameterizedType} and
         * {@code annotationType}.
         *
         * @param parameterizedType the type to register {@link BinaryMarshaller} for.
         * @param annotationType a required annotation to look up the marshaller.
         * @param marshaller the marshaller to register.
         *
         */
        public <T> Builder registerMarshaller(TypeLiteral<T> parameterizedType, Class<? extends Annotation> annotationType, BinaryMarshaller<T> marshaller) {
            Objects.requireNonNull(parameterizedType, "ParameterizedType must be non null.");
            Objects.requireNonNull(annotationType, "AnnotationType must be non null.");
            Objects.requireNonNull(marshaller, "Marshaller must be non null.");
            insert(annotationBinaryMarshallers, parameterizedType.getRawType(), annotationType, marshaller);
            return this;
        }

        private static <T> void insert(Map<Class<? extends Annotation>, List<Pair<Class<?>, T>>> into, Class<?> type, Class<? extends Annotation> annotationType, T marshaller) {
            verifyAnnotation(annotationType);
            List<Pair<Class<?>, T>> types = into.computeIfAbsent(annotationType, (k) -> new LinkedList<>());
            Pair<Class<?>, T> toInsert = Pair.create(type, marshaller);
            boolean inserted = false;
            for (ListIterator<Pair<Class<?>, T>> it = types.listIterator(); it.hasNext();) {
                Pair<Class<?>, T> current = it.next();
                if (current.getLeft().isAssignableFrom(type)) {
                    it.set(toInsert);
                    it.add(current);
                    inserted = true;
                    break;
                }
            }
            if (!inserted) {
                types.add(toInsert);
            }
        }

        /**
         * Registers a callback used by the {@link NativeIsolate} to attach the current thread to an
         * isolate.
         *
         * @param action a {@link LongUnaryOperator} that takes an isolate address as a parameter
         *            and returns the isolate thread address.
         */
        public Builder setAttachThreadAction(LongUnaryOperator action) {
            Objects.requireNonNull(action, "Action must be non null.");
            this.attachThreadAction = action;
            return this;
        }

        /**
         * Registers a callback used by the {@link NativeIsolate} to detach the current thread from
         * an isolate.
         *
         * @param action a {@link LongUnaryOperator} that takes an isolate thread address as a
         *            parameter and returns {@code 0} on success or non-zero in case of an error.
         */
        public Builder setDetachThreadAction(LongUnaryOperator action) {
            Objects.requireNonNull(action, "Action must be non null.");
            this.detachThreadAction = action;
            return this;
        }

        /**
         * Registers a callback used by the {@link NativeIsolate} to tear down the isolate.
         *
         * @param action a {@link LongBinaryOperator} that takes an isolate address and an isolate
         *            thread address as parameters and returns {@code 0} on success or non-zero in
         *            case of an error.
         */
        public Builder setShutDownIsolateAction(LongBinaryOperator action) {
            Objects.requireNonNull(action, "Action must be non null.");
            this.shutDownIsolateAction = action;
            return this;
        }

        /**
         * Registers a callback used by the {@link NativeObject} to free an object in a native image
         * heap referenced by the garbage-collected handle. At some point after a
         * {@link NativeObject} is garbage collected, a call to the {@code action} is made to
         * release the corresponding object in the native image heap.
         *
         * @param action a {@link LongBinaryOperator} that takes an isolate thread address and
         *            object handle as parameters and returns {@code 0} on success or non-zero in
         *            case of an error.
         *
         * @see NativeObject
         */
        public Builder setReleaseNativeObjectAction(LongBinaryOperator action) {
            Objects.requireNonNull(action, "Action must be non null.");
            this.releaseNativeObjectAction = action;
            return this;
        }

        /**
         * Registers a thread local factory whenever the default thread local handling should be
         * overriden. This can be useful to install a terminating thread local using JVMCI services
         * when needed.
         *
         * @see NativeObject
         */
        public Builder setNativeThreadLocalFactory(Function<Supplier<NativeIsolateThread>, ThreadLocal<NativeIsolateThread>> factory) {
            Objects.requireNonNull(factory, "Action must be non null.");
            this.threadLocalFactory = factory;
            return this;
        }

        /**
         * Builds the {@link JNIConfig}.
         */
        public JNIConfig build() {
            return new JNIConfig(binaryMarshallers, annotationBinaryMarshallers,
                            attachThreadAction, detachThreadAction, shutDownIsolateAction,
                            releaseNativeObjectAction, threadLocalFactory);
        }

        /**
         * Returns a {@link BinaryMarshaller} for stack trace marshalling.
         */
        public static BinaryMarshaller<StackTraceElement[]> defaultStackTraceMarshaller() {
            return DefaultStackTraceMarshaller.INSTANCE;
        }
    }

}
