package org.pitest.mutationtest.build.intercept.equivalent;

import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.AbstractInsnNode;
import org.objectweb.asm.tree.LdcInsnNode;
import org.objectweb.asm.tree.MethodInsnNode;
import org.pitest.classinfo.ClassName;
import org.pitest.mutationtest.build.CompoundMutationInterceptor;
import org.pitest.mutationtest.build.InterceptorParameters;
import org.pitest.mutationtest.build.InterceptorType;
import org.pitest.mutationtest.build.MutationInterceptor;
import org.pitest.mutationtest.build.MutationInterceptorFactory;
import org.pitest.plugin.Feature;
import org.pitest.sequence.Match;
import org.pitest.sequence.QueryStart;
import org.pitest.sequence.SequenceQuery;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import static org.pitest.bytecode.analysis.InstructionMatchers.getStatic;
import static org.pitest.bytecode.analysis.InstructionMatchers.methodCallNamed;
import static org.pitest.bytecode.analysis.InstructionMatchers.methodCallTo;
import static org.pitest.bytecode.analysis.OpcodeMatchers.ACONST_NULL;
import static org.pitest.bytecode.analysis.OpcodeMatchers.ARETURN;
import static org.pitest.bytecode.analysis.OpcodeMatchers.DRETURN;
import static org.pitest.bytecode.analysis.OpcodeMatchers.FRETURN;
import static org.pitest.bytecode.analysis.OpcodeMatchers.ICONST_1;
import static org.pitest.bytecode.analysis.OpcodeMatchers.IRETURN;
import static org.pitest.bytecode.analysis.OpcodeMatchers.LDC;
import static org.pitest.bytecode.analysis.OpcodeMatchers.LRETURN;
import static org.pitest.mutationtest.engine.gregor.mutators.returns.BooleanFalseReturnValsMutator.FALSE_RETURNS;
import static org.pitest.mutationtest.engine.gregor.mutators.returns.BooleanTrueReturnValsMutator.TRUE_RETURNS;
import static org.pitest.mutationtest.engine.gregor.mutators.returns.EmptyObjectReturnValsMutator.EMPTY_RETURNS;
import static org.pitest.mutationtest.engine.gregor.mutators.returns.NullReturnValsMutator.NULL_RETURNS;
import static org.pitest.mutationtest.engine.gregor.mutators.returns.PrimitiveReturnsMutator.PRIMITIVE_RETURNS;
import static org.pitest.sequence.Result.result;

/**
 * Tightly coupled to the PrimitiveReturnsMutator and EmptyObjectReturnValsMutator
 *   - removes trivially equivalent mutants generated by these.
 * operator
 *
 */
public class EquivalentReturnMutationFilter implements MutationInterceptorFactory {
    
  @Override
  public String description() {
    return "Trivial return vals equivalence filter";
  }

  @Override
  public Feature provides() {
    return Feature.named("FRETEQUIV")
        .withOnByDefault(true)
        .withDescription("Filters return vals mutants with bytecode equivalent to the unmutated class");

  }

  @Override
  public MutationInterceptor createInterceptor(InterceptorParameters params) {
    return new CompoundMutationInterceptor(Arrays.asList(
            new EmptyReturnsFilter(primitiveZeroConstants(), primitiveReturns(), PRIMITIVE_RETURNS, FALSE_RETURNS),
            new EmptyReturnsFilter(QueryStart.match(ACONST_NULL), ARETURN, NULL_RETURNS),
            new EmptyReturnsFilter(emptyReturns(), ARETURN, EMPTY_RETURNS, FALSE_RETURNS),
            new EmptyReturnsFilter(QueryStart.match(ICONST_1), IRETURN, TRUE_RETURNS),
            new EmptyReturnsFilter(hardCodedTrue(), ARETURN, TRUE_RETURNS)
    )) {

      @Override
      public InterceptorType type() {
        return InterceptorType.FILTER;
      }
    };
  }

  private static Match<AbstractInsnNode> ldcConstant(String s) {
    return (c, n) -> result(s.equals(((LdcInsnNode) n).cst), c);
  }

  private static Match<AbstractInsnNode> isZeroConstant() {
    Set<Integer> zeroConstants = new HashSet<>();

    zeroConstants.add(Opcodes.ICONST_0);
    zeroConstants.add(Opcodes.LCONST_0);
    zeroConstants.add(Opcodes.FCONST_0);
    zeroConstants.add(Opcodes.DCONST_0);

    return (context, node) -> result(zeroConstants.contains(node.getOpcode()), context);
  }

  private SequenceQuery<AbstractInsnNode> emptyReturns() {
    SequenceQuery<AbstractInsnNode> constantZero = QueryStart
            .match(isZeroConstant())
            .then(methodCallNamed("valueOf"));

    SequenceQuery<AbstractInsnNode> constantFalse = QueryStart
            .match(getStatic("java/lang/Boolean", "FALSE"));

    SequenceQuery<AbstractInsnNode> emptyString = QueryStart
            .match(LDC.and(ldcConstant("")));

    return constantZero.or(constantFalse).or(emptyString).or(QueryStart.match(loadsEmptyReturnOntoStack()));
  }

  private static Match<AbstractInsnNode> loadsEmptyReturnOntoStack() {
    return noArgsCall("java/util/Optional", "empty")
            .or(noArgsCall("java/util/stream/Stream", "empty"))
            .or(noArgsCall("java/util/Collections", "emptyList"))
            .or(noArgsCall("java/util/Collections", "emptyMap"))
            .or(noArgsCall("java/util/Collections", "emptySet"))
            .or(noArgsCall("java/util/Map", "of"))
            .or(noArgsCall("java/util/List", "of"))
            .or(noArgsCall("java/util/Set", "of"));
  }

  private static Match<AbstractInsnNode> noArgsCall(String owner, String name) {
    return methodCallTo(ClassName.fromString(owner), name).and(takesNoArgs());
  }

  private static Match<AbstractInsnNode> takesNoArgs() {
    return (c, node) -> {
      if (node instanceof MethodInsnNode) {
        final MethodInsnNode call = (MethodInsnNode) node;
        return result(Type.getArgumentTypes(call.desc).length == 0, c);
      }
      return result(false, c);
    };
  }

  private SequenceQuery<AbstractInsnNode> hardCodedTrue() {
    SequenceQuery<AbstractInsnNode> boxedTrue = QueryStart
            .match(ICONST_1)
            .then(methodCallNamed("valueOf"));

    SequenceQuery<AbstractInsnNode> constantTrue = QueryStart
            .match(getStatic("java/lang/Boolean", "TRUE"));

    return boxedTrue.or(constantTrue);
  }

  private Match<AbstractInsnNode> primitiveReturns() {
    return IRETURN.or(FRETURN).or(DRETURN).or(LRETURN);
  }

  private SequenceQuery<AbstractInsnNode> primitiveZeroConstants() {
    Set<Integer> zeroConstants = new HashSet<>();
    zeroConstants.add(Opcodes.ICONST_0);
    zeroConstants.add(Opcodes.LCONST_0);
    zeroConstants.add(Opcodes.FCONST_0);
    zeroConstants.add(Opcodes.DCONST_0);
    return QueryStart.match((c, n) -> result(zeroConstants.contains(n.getOpcode()), c));
  }



}


