/*
 * Expectation.java February 2007
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

/**
 * The <code>Expectation</code> object is used to send the expect
 * continue status if required. This is delivered to the client to
 * tell the client that the server is willing to accept the 
 * request body. Once this is sent the transport will likely wait
 * until there is a read ready event.
 * 
 * @author Niall Gallagher
 */
class Expectation {

   /**
    * This is the status code that is sent to prompt the client. 
    */
   private static final byte[] STATUS = { 'H', 'T','T', 'P', '/','1','.', '1',' ', '1','0','0',' '};
   
   /**
    * This is the optional description for the expect status code.
    */
   private static final byte[] MESSAGE = {'C','o','n','t','i','n','u','e', '\r','\n','\r','\n'};
   
   /**
    * This is the sender that is used to deliver the continue.
    */
   private final Sender sender;
   
   /**
    * Constructor for the <code>Expectation</code> object. This will
    * create an object that will deliver the continue status code.
    * Because the transport performs an asynchronous write this will
    * not block the execution of this method and delay execution.
    * 
    * @param channel this is the channel used to deliver the prompt
    */
   public Expectation(Channel channel) {
      this.sender = channel.getSender();
   }
   
   /**
    * This will execute the continue if the header contains the 
    * expectation header. If there is no expectation then this will
    * return without sending anything back to the connected client.
    * 
    * @param header this is the header read from the channel
    */
   public void execute(Header header) throws IOException {
      if(header.isExpectContinue()) {
         sender.send(STATUS);
         sender.send(MESSAGE);
         sender.flush();
      }
   }
}
