/*
 * SecurePolicy.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.util.Random;

import org.simpleframework.http.Cookie;

/**
 * The <code>SecurePolicy</code> object is used to create a policy
 * that can create cookies that are secure. This ensures that the
 * cookies can not be predicted and thus hijacked. Because there is
 * no standard for session cookies, this uses the Java Servlet API
 * naming convention, which names cookie JSESSIONID.
 * 
 * @author Niall Gallagher
 */
class SecurePolicy implements Policy {

   /**
    * This provides the name of the cookies that are created.
    */
   private static final String NAME = "JSESSIONID";
   
   /**
    * This is the random number generator used to create values.
    */
   private Random random;
   
   /**
    * This is the header to acquire the cookie value from.
    */
   private Header header;
   
   /**
    * This is the cookie created for the session identifier.
    */
   private Cookie cookie;

   /**
    * Constructor for the <code>SecurePolicy</code> object. This is
    * used to create a policy that will generate cookies which have
    * secure value, which ensures session hijacking is not possible.
    * 
    * @param header this is the header to search for the cookie
    */
   public SecurePolicy(Header header) {
      this.random = new Random();
      this.header = header;
   } 

   /**
    * This is used to acquire the session cookie for the request. The
    * session cookie is either sent with the HTTP request header or
    * it can be created if required. This ensures that if no cookie
    * has been sent one can be created on demand. 
    * 
    * @param create if true the session cookie will be created
    * 
    * @return the cookie associated with the session or null
    */   
   public Cookie getSession(boolean create) {
      if(cookie != null) {
         return cookie;
      }
      cookie = header.getCookie(NAME);
      
      if(cookie == null) {
         if(create) {
            cookie = getCookie(NAME);
         }
      }      
      return cookie;
   }
   
   /**
    * This is used to create the cookie value to be used. The value 
    * used for the cookie is created using a random number generator.
    * This ensures that the cookie names created are secure to a
    * point that they can not be hijacked by another used.
    * 
    * @param name this is the name of the cookie to be created
    * 
    * @return a session cookie with a secure cookie value
    */
   private Cookie getCookie(String name) {
      long identity = random.nextLong();
      String value = Long.toHexString(identity);
      
      return new Cookie(name, value, true);
   }
}
