/*
 * FileAllocator.java February 2008
 *
 * Copyright (C) 2008, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.buffer;

import java.io.File;
import java.io.IOException;

/**
 * The <code>FileAllocator</code> object is used to create buffers
 * that can be written to the file system. This creates buffers as
 * files if they are larger than the specified limit. This ensures
 * that buffers of arbitrary large size can be created. All buffer
 * sizes under the limit are created using byte arrays allocated
 * on the executing VM heap. This ensures that optimal performance
 * is maintained for buffers of reasonable size.
 * 
 * @author Niall Gallagher
 */
public class FileAllocator implements Allocator {
     
   /**
    * This is the default prefix used when none has been specified.
    */
   private static final String PREFIX = "temp";
   
   /**
    * This is the file system used to create the buffer files.
    */
   private FileSystem system;
   
   /**
    * This is the limit up to which buffers are allocated in memory.
    */
   private int limit;
   
   /**
    * Constructor for the <code>FileAllocator</code> object. This is
    * used to create buffers in memory up to a threshold size. If a
    * buffer is required over the threshold size then the data is
    * written to a file, where it can be retrieved at a later point.
    */
   public FileAllocator() throws IOException {
      this(1048576);
   }

   /**
    * Constructor for the <code>FileAllocator</code> object. This is
    * used to create buffers in memory up to a threshold size. If a
    * buffer is required over the threshold size then the data is
    * written to a file, where it can be retrieved at a later point.
    * 
    * @param limit this is the maximum size for a heap buffer
    */
   public FileAllocator(int limit) throws IOException {
      this(PREFIX, limit);
   }
   
   /**
    * Constructor for the <code>FileAllocator</code> object. This is
    * used to create buffers in memory up to a threshold size. If a
    * buffer is required over the threshold size then the data is
    * written to a file, where it can be retrieved at a later point.
    * 
    * @param prefix this is the file prefix for the file buffers
    */
   public FileAllocator(String prefix) throws IOException {
      this(prefix, 1048576);
   }

   /**
    * Constructor for the <code>FileAllocator</code> object. This is
    * used to create buffers in memory up to a threshold size. If a
    * buffer is required over the threshold size then the data is
    * written to a file, where it can be retrieved at a later point.
    * 
    * @param prefix this is the file prefix for the file buffers
    * @param limit this is the maximum size for a heap buffer
    */
   public FileAllocator(String prefix, int limit) throws IOException {
      this.system = new FileSystem(prefix);
      this.limit = limit;
   }
   
   /**
    * This will allocate a file buffer which will write data for the
    * buffer to a file. Buffers allocated by this method can be of 
    * arbitrary size as data is appended directly to a temporary
    * file. This ensures there is no upper limit for appended data.
    * 
    * @return a buffer which will write to a temporary file
    */
   public Buffer allocate() throws IOException {
      File file = system.create();
      
      if(!file.exists()) {
         throw new BufferException("Could not create file %s", file);
      }
      return new FileBuffer(file);
   }

   /**
    * This will allocate a file buffer which will write data for the
    * buffer to a file. Buffers allocated by this method can be of 
    * arbitrary size as data is appended directly to a temporary
    * file. This ensures there is no upper limit for appended data.
    * If the size required is less than the limit then the buffer
    * is an in memory array which provides optimal performance.
    * 
    * @param size this is the size of the buffer to be created
    * 
    * @return a buffer which will write to a created temporary file
    */
   public Buffer allocate(int size) throws IOException {
      if(size <= limit) {
         return new ArrayBuffer(size); 
      }
      return allocate();
   }
   
   /**
    * This method is used to close the allocator so that resources
    * that are occupied by the allocator can be freed. This will
    * allow the allocator to be created and closed repeatedly in
    * a single process without holding on to resources such as
    * mapped file buffers or threads.
    */
   public void close() throws IOException {
      system.close();
   }
}
