/*
 * PoolQueue.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.thread;

import static java.util.concurrent.TimeUnit.SECONDS;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * The <code>PoolQueue</code> object is used to execute tasks in
 * a thread pool. This creates a thread pool with an unbounded list
 * of outstanding tasks, which ensures that any system requesting
 * a task to be executed will not block when handing it over.
 * 
 * @author Niall Gallagher
 */
class PoolQueue extends ThreadPoolExecutor {
   
   /**
    * Constructor for the <code>PoolQueue</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    * 
    * @param type this is the type of runnable that this accepts
    * @param rest this is the number of threads to use in the pool    
    * @param active this is the maximum size the pool can grow to 
    */    
   public PoolQueue(Class type, int rest, int active) {
      this(type, rest, active, 120, TimeUnit.SECONDS);
   }
  
   /**
    * Constructor for the <code>PoolQueue</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    *
    * @param type this is the type of runnable that this accepts
    * @param rest this is the number of threads to use in the pool    
    * @param active this is the maximum size the pool can grow to
    * @param duration the duration active threads remain idle for
    * @param unit this is the time unit used for the duration 
    */    
   public PoolQueue(Class type, int rest, int active, long duration, TimeUnit unit) {
      super(rest, active, duration, unit, new Queue(), new PoolFactory(type));
   }
   
   /**
    * This is used to wait until such time as the pool has terminated.
    * Using a join such as this allows the user to be sure that there
    * are no further tasks enqueued for execution and there are no
    * tasks currently executing. This helps provide graceful shutdown.
    */   
   public void join() {
      boolean dead = isTerminated();
   
      while(!dead) {
         try {
            dead = awaitTermination(10, SECONDS);
         } catch(InterruptedException e) {
            break;
         }
      }
   }
   
   /**
    * This is used to stop the executor by interrupting all running
    * tasks and shutting down the threads within the pool. This will
    * return once it has been stopped, and no further tasks will be 
    * accepted by this pool for execution.
    */   
   public void stop() {
      shutdown();
      join();
   }
   
   /**
    * This is the internal queue used by this implementation. This
    * provides an unlimited number of positions for new tasks to
    * be queued. Having an unlimited queue prevents deadlocks.
    * 
    * @author Niall Gallagher
    */
   private static class Queue extends LinkedBlockingQueue<Runnable> {
      
      /**
       * Constructor for the <code>Queue</code> object. This will
       * create a linked blocking queue with an unlimited capacity.
       */
      public Queue() {
         super();
      }
   }
}
