/*
 * Decompiled with CFR 0.152.
 */
package org.whitesource.agent.hash;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumMap;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BOMInputStream;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whitesource.agent.api.model.ChecksumType;
import org.whitesource.agent.api.model.DependencyType;
import org.whitesource.agent.hash.HashAlgorithm;
import org.whitesource.agent.hash.HashCalculationResult;
import org.whitesource.agent.hash.WssHashException;
import org.whitesource.agent.parser.JavaScriptParser;
import org.whitesource.agent.parser.ParseResult;

public class HashCalculator {
    private static final Logger logger = LoggerFactory.getLogger(HashCalculator.class);
    private static final int BUFFER_SIZE = 32768;
    private static final int FILE_MIN_SIZE_THRESHOLD = 512;
    private static final int FILE_PARTIAL_HASH_MIN_SIZE = 2048;
    private static final int FILE_SMALL_SIZE = 3072;
    private static final int FILE_MAX_SIZE_THRESHOLD = Integer.MAX_VALUE;
    private static final double FILE_SMALL_BUCKET_SIZE = 1280.0;
    private static final char ZERO = '0';
    private static final byte CARRIAGE_RETURN = 13;
    private static final byte NEW_LINE = 10;
    private static final byte HORIZONTAL_TAB = 9;
    private static final byte SPACE = 32;
    private static final Collection<Byte> WHITESPACES = Arrays.asList((byte)13, (byte)10, (byte)9, (byte)32);
    private static final String UTF_8 = "utf-8";
    private static final String UNDERSCORE = "_";

    public HashCalculationResult calculateSuperHash(File file) throws IOException {
        long fileSize = file.length();
        if (fileSize <= 512L) {
            logger.debug("Ignored file " + file.getName() + " (" + FileUtils.byteCountToDisplaySize((long)fileSize) + "): minimum file size is 512B");
            return null;
        }
        if (fileSize >= Integer.MAX_VALUE) {
            logger.debug("Ignore file {}, ({}): maximum file size is 2GB", (Object)file.getName(), (Object)FileUtils.byteCountToDisplaySize((long)fileSize));
            return null;
        }
        HashCalculationResult result = null;
        try {
            result = this.calculateSuperHash(FileUtils.readFileToByteArray((File)file));
        }
        catch (OutOfMemoryError e) {
            logger.debug(MessageFormat.format("Failed calculating SHA-1 for file {0}: size too big {1}", file.getAbsolutePath(), FileUtils.byteCountToDisplaySize((long)fileSize)));
        }
        return result;
    }

    public HashCalculationResult calculateSuperHash(byte[] bytes) throws IOException {
        HashCalculationResult result = null;
        byte[] bytesWithoutSpaces = this.stripWhiteSpaces(bytes);
        long fileSize = bytesWithoutSpaces.length;
        if (fileSize < 512L) {
            logger.debug("Ignoring file with size " + FileUtils.byteCountToDisplaySize((long)fileSize) + ": minimum file size is 512B");
        } else if (fileSize <= 2048L) {
            String fullFileHash = this.calculateByteArrayHash(bytesWithoutSpaces, HashAlgorithm.SHA1);
            result = new HashCalculationResult(fullFileHash);
        } else if (fileSize <= 3072L) {
            result = this.hashBuckets(bytesWithoutSpaces, 1280.0);
        } else {
            int baseLowNumber = 1;
            int digits = (int)Math.log10(fileSize);
            for (int i = 0; i < digits; ++i) {
                baseLowNumber *= 10;
            }
            double highNumber = Math.ceil((float)(fileSize + 1L) / (float)baseLowNumber) * (double)baseLowNumber;
            double lowNumber = highNumber - (double)baseLowNumber;
            double bucketSize = (highNumber + lowNumber) / 4.0;
            result = this.hashBuckets(bytesWithoutSpaces, bucketSize);
        }
        return result;
    }

    public String calculateSHA1(File resourceFile) throws IOException {
        return this.calculateHash(resourceFile, HashAlgorithm.SHA1);
    }

    public String calculateHash(File resourceFile, HashAlgorithm algorithm) throws IOException {
        MessageDigest messageDigest;
        try {
            messageDigest = MessageDigest.getInstance(algorithm.getAlgorithm());
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
        try (FileInputStream inputStream = new FileInputStream(resourceFile);
             BOMInputStream fis = new BOMInputStream((InputStream)inputStream);){
            byte[] buffer = new byte[32768];
            int len = fis.read(buffer, 0, 32768);
            while (len >= 0) {
                messageDigest.update(buffer, 0, len);
                len = fis.read(buffer, 0, 32768);
            }
        }
        return this.toHex(messageDigest.digest());
    }

    public String calculateByteArraySHA1(byte[] byteArray) throws IOException {
        return this.calculateByteArrayHash(byteArray, HashAlgorithm.SHA1);
    }

    public String calculateByteArrayHash(byte[] byteArray, HashAlgorithm algorithm) throws IOException {
        MessageDigest messageDigest;
        try {
            messageDigest = MessageDigest.getInstance(algorithm.getAlgorithm());
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
        messageDigest.update(byteArray, 0, byteArray.length);
        return this.toHex(messageDigest.digest());
    }

    public Map<ChecksumType, String> calculateJavaScriptHashes(File file) throws WssHashException {
        EnumMap<ChecksumType, String> checksums = new EnumMap(ChecksumType.class);
        try {
            long fileLength = file.length();
            if (fileLength >= Integer.MAX_VALUE) {
                logger.debug("Ignore file {}, ({}): maximum file size  is 2GB", (Object)file.getName(), (Object)FileUtils.byteCountToDisplaySize((long)fileLength));
                return checksums;
            }
            checksums = this.calculateJavaScriptHashes(FileUtils.readFileToByteArray((File)file));
        }
        catch (Exception e) {
            throw new WssHashException("Error calculating JavaScript hash: " + e.getMessage());
        }
        return checksums;
    }

    public Map<ChecksumType, String> calculateJavaScriptHashes(byte[] byteArray) throws WssHashException {
        EnumMap<ChecksumType, String> checksums = new EnumMap<ChecksumType, String>(ChecksumType.class);
        try {
            String fileContent = IOUtils.toString((byte[])byteArray, (String)UTF_8);
            ParseResult parseResult = new JavaScriptParser().parse(fileContent);
            if (parseResult != null) {
                String headerlessContent;
                HashCalculationResult noCommentsSha1;
                String contentWithoutComments = parseResult.getContentWithoutComments();
                if (StringUtils.isNotBlank((String)contentWithoutComments) && (noCommentsSha1 = this.calculateSuperHash(contentWithoutComments.getBytes())) != null) {
                    checksums.put(ChecksumType.SHA1_NO_COMMENTS_SUPER_HASH, noCommentsSha1.getFullHash());
                }
                if (StringUtils.isNotBlank((String)(headerlessContent = parseResult.getContentWithoutHeaderComments()))) {
                    String headerlessChecksum = this.calculateByteArrayHash(headerlessContent.getBytes(), HashAlgorithm.SHA1);
                    checksums.put(ChecksumType.SHA1_NO_HEADER, headerlessChecksum);
                }
            }
        }
        catch (Exception e) {
            throw new WssHashException("Error calculating JavaScript hash: " + e.getMessage());
        }
        return checksums;
    }

    public String calculateSha1ByNameVersionAndType(String name, String version, DependencyType dependencyType) throws IOException {
        String sha1ToCalc = name + UNDERSCORE + version + UNDERSCORE + dependencyType.toString();
        return this.calculateByteArraySHA1(sha1ToCalc.getBytes(StandardCharsets.UTF_8));
    }

    private HashCalculationResult hashBuckets(byte[] fileWithoutSpaces, double bucketSize) throws IOException {
        int bucketIntSize = (int)bucketSize;
        byte[] mostSigBytes = Arrays.copyOfRange(fileWithoutSpaces, 0, bucketIntSize);
        int length = fileWithoutSpaces.length;
        byte[] leastSigBytes = Arrays.copyOfRange(fileWithoutSpaces, length - bucketIntSize, length);
        String fullFileHash = this.calculateByteArraySHA1(fileWithoutSpaces);
        String mostSigBitsHash = this.calculateByteArraySHA1(mostSigBytes);
        String leastSigBitsHash = this.calculateByteArraySHA1(leastSigBytes);
        return new HashCalculationResult(fullFileHash, mostSigBitsHash, leastSigBitsHash);
    }

    private String toHex(byte[] bytes) {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        for (byte aByte : bytes) {
            int b = aByte & 0xFF;
            if (b < 16) {
                sb.append('0');
            }
            sb.append(Integer.toHexString(b));
        }
        return sb.toString();
    }

    private byte[] stripWhiteSpaces(byte[] data) {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        for (byte b : data) {
            if (WHITESPACES.contains(b)) continue;
            bos.write(b);
        }
        return bos.toByteArray();
    }
}

