/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that determines whether any URL query strings in viewer requests (and if so, which query strings) are
 * included in the cache key and automatically included in requests that CloudFront sends to the origin.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CachePolicyQueryStringsConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<CachePolicyQueryStringsConfig.Builder, CachePolicyQueryStringsConfig> {
    private static final SdkField<String> QUERY_STRING_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("QueryStringBehavior")
            .getter(getter(CachePolicyQueryStringsConfig::queryStringBehaviorAsString))
            .setter(setter(Builder::queryStringBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryStringBehavior")
                    .unmarshallLocationName("QueryStringBehavior").build()).build();

    private static final SdkField<QueryStringNames> QUERY_STRINGS_FIELD = SdkField
            .<QueryStringNames> builder(MarshallingType.SDK_POJO)
            .memberName("QueryStrings")
            .getter(getter(CachePolicyQueryStringsConfig::queryStrings))
            .setter(setter(Builder::queryStrings))
            .constructor(QueryStringNames::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryStrings")
                    .unmarshallLocationName("QueryStrings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUERY_STRING_BEHAVIOR_FIELD,
            QUERY_STRINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String queryStringBehavior;

    private final QueryStringNames queryStrings;

    private CachePolicyQueryStringsConfig(BuilderImpl builder) {
        this.queryStringBehavior = builder.queryStringBehavior;
        this.queryStrings = builder.queryStrings;
    }

    /**
     * <p>
     * Determines whether any URL query strings in viewer requests are included in the cache key and automatically
     * included in requests that CloudFront sends to the origin. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>none</code> – Query strings in viewer requests are not included in the cache key and are not automatically
     * included in requests that CloudFront sends to the origin. Even when this field is set to <code>none</code>, any
     * query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i> included in origin requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>whitelist</code> – The query strings in viewer requests that are listed in the
     * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
     * CloudFront sends to the origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in the
     * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
     * CloudFront sends to the origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>all</code> – All query strings in viewer requests are included in the cache key and are automatically
     * included in requests that CloudFront sends to the origin.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #queryStringBehavior} will return {@link CachePolicyQueryStringBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #queryStringBehaviorAsString}.
     * </p>
     * 
     * @return Determines whether any URL query strings in viewer requests are included in the cache key and
     *         automatically included in requests that CloudFront sends to the origin. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>none</code> – Query strings in viewer requests are not included in the cache key and are not
     *         automatically included in requests that CloudFront sends to the origin. Even when this field is set to
     *         <code>none</code>, any query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i>
     *         included in origin requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>whitelist</code> – The query strings in viewer requests that are listed in the
     *         <code>QueryStringNames</code> type are included in the cache key and automatically included in requests
     *         that CloudFront sends to the origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in the
     *         <code>QueryStringNames</code> type are included in the cache key and automatically included in requests
     *         that CloudFront sends to the origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>all</code> – All query strings in viewer requests are included in the cache key and are
     *         automatically included in requests that CloudFront sends to the origin.
     *         </p>
     *         </li>
     * @see CachePolicyQueryStringBehavior
     */
    public CachePolicyQueryStringBehavior queryStringBehavior() {
        return CachePolicyQueryStringBehavior.fromValue(queryStringBehavior);
    }

    /**
     * <p>
     * Determines whether any URL query strings in viewer requests are included in the cache key and automatically
     * included in requests that CloudFront sends to the origin. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>none</code> – Query strings in viewer requests are not included in the cache key and are not automatically
     * included in requests that CloudFront sends to the origin. Even when this field is set to <code>none</code>, any
     * query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i> included in origin requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>whitelist</code> – The query strings in viewer requests that are listed in the
     * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
     * CloudFront sends to the origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in the
     * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
     * CloudFront sends to the origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>all</code> – All query strings in viewer requests are included in the cache key and are automatically
     * included in requests that CloudFront sends to the origin.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #queryStringBehavior} will return {@link CachePolicyQueryStringBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #queryStringBehaviorAsString}.
     * </p>
     * 
     * @return Determines whether any URL query strings in viewer requests are included in the cache key and
     *         automatically included in requests that CloudFront sends to the origin. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>none</code> – Query strings in viewer requests are not included in the cache key and are not
     *         automatically included in requests that CloudFront sends to the origin. Even when this field is set to
     *         <code>none</code>, any query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i>
     *         included in origin requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>whitelist</code> – The query strings in viewer requests that are listed in the
     *         <code>QueryStringNames</code> type are included in the cache key and automatically included in requests
     *         that CloudFront sends to the origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in the
     *         <code>QueryStringNames</code> type are included in the cache key and automatically included in requests
     *         that CloudFront sends to the origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>all</code> – All query strings in viewer requests are included in the cache key and are
     *         automatically included in requests that CloudFront sends to the origin.
     *         </p>
     *         </li>
     * @see CachePolicyQueryStringBehavior
     */
    public String queryStringBehaviorAsString() {
        return queryStringBehavior;
    }

    /**
     * <p>
     * Contains the specific query strings in viewer requests that either <i> <b>are</b> </i> or <i> <b>are not</b> </i>
     * included in the cache key and automatically included in requests that CloudFront sends to the origin. The
     * behavior depends on whether the <code>QueryStringBehavior</code> field in the
     * <code>CachePolicyQueryStringsConfig</code> type is set to <code>whitelist</code> (the listed query strings <i>
     * <b>are</b> </i> included) or <code>allExcept</code> (the listed query strings <i> <b>are not</b> </i> included,
     * but all other query strings are).
     * </p>
     * 
     * @return Contains the specific query strings in viewer requests that either <i> <b>are</b> </i> or <i> <b>are
     *         not</b> </i> included in the cache key and automatically included in requests that CloudFront sends to
     *         the origin. The behavior depends on whether the <code>QueryStringBehavior</code> field in the
     *         <code>CachePolicyQueryStringsConfig</code> type is set to <code>whitelist</code> (the listed query
     *         strings <i> <b>are</b> </i> included) or <code>allExcept</code> (the listed query strings <i> <b>are
     *         not</b> </i> included, but all other query strings are).
     */
    public QueryStringNames queryStrings() {
        return queryStrings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(queryStringBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(queryStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CachePolicyQueryStringsConfig)) {
            return false;
        }
        CachePolicyQueryStringsConfig other = (CachePolicyQueryStringsConfig) obj;
        return Objects.equals(queryStringBehaviorAsString(), other.queryStringBehaviorAsString())
                && Objects.equals(queryStrings(), other.queryStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CachePolicyQueryStringsConfig").add("QueryStringBehavior", queryStringBehaviorAsString())
                .add("QueryStrings", queryStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueryStringBehavior":
            return Optional.ofNullable(clazz.cast(queryStringBehaviorAsString()));
        case "QueryStrings":
            return Optional.ofNullable(clazz.cast(queryStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CachePolicyQueryStringsConfig, T> g) {
        return obj -> g.apply((CachePolicyQueryStringsConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CachePolicyQueryStringsConfig> {
        /**
         * <p>
         * Determines whether any URL query strings in viewer requests are included in the cache key and automatically
         * included in requests that CloudFront sends to the origin. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>none</code> – Query strings in viewer requests are not included in the cache key and are not
         * automatically included in requests that CloudFront sends to the origin. Even when this field is set to
         * <code>none</code>, any query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i>
         * included in origin requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>whitelist</code> – The query strings in viewer requests that are listed in the
         * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
         * CloudFront sends to the origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in the
         * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
         * CloudFront sends to the origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>all</code> – All query strings in viewer requests are included in the cache key and are automatically
         * included in requests that CloudFront sends to the origin.
         * </p>
         * </li>
         * </ul>
         * 
         * @param queryStringBehavior
         *        Determines whether any URL query strings in viewer requests are included in the cache key and
         *        automatically included in requests that CloudFront sends to the origin. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>none</code> – Query strings in viewer requests are not included in the cache key and are not
         *        automatically included in requests that CloudFront sends to the origin. Even when this field is set to
         *        <code>none</code>, any query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i>
         *        included in origin requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>whitelist</code> – The query strings in viewer requests that are listed in the
         *        <code>QueryStringNames</code> type are included in the cache key and automatically included in
         *        requests that CloudFront sends to the origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in
         *        the <code>QueryStringNames</code> type are included in the cache key and automatically included in
         *        requests that CloudFront sends to the origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>all</code> – All query strings in viewer requests are included in the cache key and are
         *        automatically included in requests that CloudFront sends to the origin.
         *        </p>
         *        </li>
         * @see CachePolicyQueryStringBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CachePolicyQueryStringBehavior
         */
        Builder queryStringBehavior(String queryStringBehavior);

        /**
         * <p>
         * Determines whether any URL query strings in viewer requests are included in the cache key and automatically
         * included in requests that CloudFront sends to the origin. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>none</code> – Query strings in viewer requests are not included in the cache key and are not
         * automatically included in requests that CloudFront sends to the origin. Even when this field is set to
         * <code>none</code>, any query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i>
         * included in origin requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>whitelist</code> – The query strings in viewer requests that are listed in the
         * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
         * CloudFront sends to the origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in the
         * <code>QueryStringNames</code> type are included in the cache key and automatically included in requests that
         * CloudFront sends to the origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>all</code> – All query strings in viewer requests are included in the cache key and are automatically
         * included in requests that CloudFront sends to the origin.
         * </p>
         * </li>
         * </ul>
         * 
         * @param queryStringBehavior
         *        Determines whether any URL query strings in viewer requests are included in the cache key and
         *        automatically included in requests that CloudFront sends to the origin. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>none</code> – Query strings in viewer requests are not included in the cache key and are not
         *        automatically included in requests that CloudFront sends to the origin. Even when this field is set to
         *        <code>none</code>, any query strings that are listed in an <code>OriginRequestPolicy</code> <i>are</i>
         *        included in origin requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>whitelist</code> – The query strings in viewer requests that are listed in the
         *        <code>QueryStringNames</code> type are included in the cache key and automatically included in
         *        requests that CloudFront sends to the origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>allExcept</code> – All query strings in viewer requests that are <i> <b>not</b> </i> listed in
         *        the <code>QueryStringNames</code> type are included in the cache key and automatically included in
         *        requests that CloudFront sends to the origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>all</code> – All query strings in viewer requests are included in the cache key and are
         *        automatically included in requests that CloudFront sends to the origin.
         *        </p>
         *        </li>
         * @see CachePolicyQueryStringBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CachePolicyQueryStringBehavior
         */
        Builder queryStringBehavior(CachePolicyQueryStringBehavior queryStringBehavior);

        /**
         * <p>
         * Contains the specific query strings in viewer requests that either <i> <b>are</b> </i> or <i> <b>are not</b>
         * </i> included in the cache key and automatically included in requests that CloudFront sends to the origin.
         * The behavior depends on whether the <code>QueryStringBehavior</code> field in the
         * <code>CachePolicyQueryStringsConfig</code> type is set to <code>whitelist</code> (the listed query strings
         * <i> <b>are</b> </i> included) or <code>allExcept</code> (the listed query strings <i> <b>are not</b> </i>
         * included, but all other query strings are).
         * </p>
         * 
         * @param queryStrings
         *        Contains the specific query strings in viewer requests that either <i> <b>are</b> </i> or <i> <b>are
         *        not</b> </i> included in the cache key and automatically included in requests that CloudFront sends to
         *        the origin. The behavior depends on whether the <code>QueryStringBehavior</code> field in the
         *        <code>CachePolicyQueryStringsConfig</code> type is set to <code>whitelist</code> (the listed query
         *        strings <i> <b>are</b> </i> included) or <code>allExcept</code> (the listed query strings <i> <b>are
         *        not</b> </i> included, but all other query strings are).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryStrings(QueryStringNames queryStrings);

        /**
         * <p>
         * Contains the specific query strings in viewer requests that either <i> <b>are</b> </i> or <i> <b>are not</b>
         * </i> included in the cache key and automatically included in requests that CloudFront sends to the origin.
         * The behavior depends on whether the <code>QueryStringBehavior</code> field in the
         * <code>CachePolicyQueryStringsConfig</code> type is set to <code>whitelist</code> (the listed query strings
         * <i> <b>are</b> </i> included) or <code>allExcept</code> (the listed query strings <i> <b>are not</b> </i>
         * included, but all other query strings are).
         * </p>
         * This is a convenience that creates an instance of the {@link QueryStringNames.Builder} avoiding the need to
         * create one manually via {@link QueryStringNames#builder()}.
         *
         * When the {@link Consumer} completes, {@link QueryStringNames.Builder#build()} is called immediately and its
         * result is passed to {@link #queryStrings(QueryStringNames)}.
         * 
         * @param queryStrings
         *        a consumer that will call methods on {@link QueryStringNames.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryStrings(QueryStringNames)
         */
        default Builder queryStrings(Consumer<QueryStringNames.Builder> queryStrings) {
            return queryStrings(QueryStringNames.builder().applyMutation(queryStrings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String queryStringBehavior;

        private QueryStringNames queryStrings;

        private BuilderImpl() {
        }

        private BuilderImpl(CachePolicyQueryStringsConfig model) {
            queryStringBehavior(model.queryStringBehavior);
            queryStrings(model.queryStrings);
        }

        public final String getQueryStringBehavior() {
            return queryStringBehavior;
        }

        @Override
        public final Builder queryStringBehavior(String queryStringBehavior) {
            this.queryStringBehavior = queryStringBehavior;
            return this;
        }

        @Override
        public final Builder queryStringBehavior(CachePolicyQueryStringBehavior queryStringBehavior) {
            this.queryStringBehavior(queryStringBehavior == null ? null : queryStringBehavior.toString());
            return this;
        }

        public final void setQueryStringBehavior(String queryStringBehavior) {
            this.queryStringBehavior = queryStringBehavior;
        }

        public final QueryStringNames.Builder getQueryStrings() {
            return queryStrings != null ? queryStrings.toBuilder() : null;
        }

        @Override
        public final Builder queryStrings(QueryStringNames queryStrings) {
            this.queryStrings = queryStrings;
            return this;
        }

        public final void setQueryStrings(QueryStringNames.BuilderImpl queryStrings) {
            this.queryStrings = queryStrings != null ? queryStrings.build() : null;
        }

        @Override
        public CachePolicyQueryStringsConfig build() {
            return new CachePolicyQueryStringsConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
