/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An origin.
 * </p>
 * <p>
 * An origin is the location where content is stored, and from which CloudFront gets content to serve to viewers. To
 * specify an origin:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Use the <code>S3OriginConfig</code> type to specify an Amazon S3 bucket that is <i> <b>not</b> </i> configured with
 * static website hosting.
 * </p>
 * </li>
 * <li>
 * <p>
 * Use the <code>CustomOriginConfig</code> type to specify various other kinds of content containers or HTTP servers,
 * including:
 * </p>
 * <ul>
 * <li>
 * <p>
 * An Amazon S3 bucket that is configured with static website hosting
 * </p>
 * </li>
 * <li>
 * <p>
 * An Elastic Load Balancing load balancer
 * </p>
 * </li>
 * <li>
 * <p>
 * An AWS Elemental MediaPackage origin
 * </p>
 * </li>
 * <li>
 * <p>
 * An AWS Elemental MediaStore container
 * </p>
 * </li>
 * <li>
 * <p>
 * Any other HTTP server, running on an Amazon EC2 instance or any other kind of host
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * For the current maximum number of origins that you can specify per distribution, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cloudfront-limits.html#limits-web-distributions"
 * >General Quotas on Web Distributions</a> in the <i>Amazon CloudFront Developer Guide</i> (quotas were formerly
 * referred to as limits).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Origin implements SdkPojo, Serializable, ToCopyableBuilder<Origin.Builder, Origin> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Id")
            .getter(getter(Origin::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DomainName")
            .getter(getter(Origin::domainName))
            .setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName")
                    .unmarshallLocationName("DomainName").build()).build();

    private static final SdkField<String> ORIGIN_PATH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OriginPath")
            .getter(getter(Origin::originPath))
            .setter(setter(Builder::originPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginPath")
                    .unmarshallLocationName("OriginPath").build()).build();

    private static final SdkField<CustomHeaders> CUSTOM_HEADERS_FIELD = SdkField
            .<CustomHeaders> builder(MarshallingType.SDK_POJO)
            .memberName("CustomHeaders")
            .getter(getter(Origin::customHeaders))
            .setter(setter(Builder::customHeaders))
            .constructor(CustomHeaders::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomHeaders")
                    .unmarshallLocationName("CustomHeaders").build()).build();

    private static final SdkField<S3OriginConfig> S3_ORIGIN_CONFIG_FIELD = SdkField
            .<S3OriginConfig> builder(MarshallingType.SDK_POJO)
            .memberName("S3OriginConfig")
            .getter(getter(Origin::s3OriginConfig))
            .setter(setter(Builder::s3OriginConfig))
            .constructor(S3OriginConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3OriginConfig")
                    .unmarshallLocationName("S3OriginConfig").build()).build();

    private static final SdkField<CustomOriginConfig> CUSTOM_ORIGIN_CONFIG_FIELD = SdkField
            .<CustomOriginConfig> builder(MarshallingType.SDK_POJO)
            .memberName("CustomOriginConfig")
            .getter(getter(Origin::customOriginConfig))
            .setter(setter(Builder::customOriginConfig))
            .constructor(CustomOriginConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomOriginConfig")
                    .unmarshallLocationName("CustomOriginConfig").build()).build();

    private static final SdkField<Integer> CONNECTION_ATTEMPTS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ConnectionAttempts")
            .getter(getter(Origin::connectionAttempts))
            .setter(setter(Builder::connectionAttempts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionAttempts")
                    .unmarshallLocationName("ConnectionAttempts").build()).build();

    private static final SdkField<Integer> CONNECTION_TIMEOUT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ConnectionTimeout")
            .getter(getter(Origin::connectionTimeout))
            .setter(setter(Builder::connectionTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionTimeout")
                    .unmarshallLocationName("ConnectionTimeout").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DOMAIN_NAME_FIELD,
            ORIGIN_PATH_FIELD, CUSTOM_HEADERS_FIELD, S3_ORIGIN_CONFIG_FIELD, CUSTOM_ORIGIN_CONFIG_FIELD,
            CONNECTION_ATTEMPTS_FIELD, CONNECTION_TIMEOUT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String domainName;

    private final String originPath;

    private final CustomHeaders customHeaders;

    private final S3OriginConfig s3OriginConfig;

    private final CustomOriginConfig customOriginConfig;

    private final Integer connectionAttempts;

    private final Integer connectionTimeout;

    private Origin(BuilderImpl builder) {
        this.id = builder.id;
        this.domainName = builder.domainName;
        this.originPath = builder.originPath;
        this.customHeaders = builder.customHeaders;
        this.s3OriginConfig = builder.s3OriginConfig;
        this.customOriginConfig = builder.customOriginConfig;
        this.connectionAttempts = builder.connectionAttempts;
        this.connectionTimeout = builder.connectionTimeout;
    }

    /**
     * <p>
     * A unique identifier for the origin. This value must be unique within the distribution.
     * </p>
     * <p>
     * Use this value to specify the <code>TargetOriginId</code> in a <code>CacheBehavior</code> or
     * <code>DefaultCacheBehavior</code>.
     * </p>
     * 
     * @return A unique identifier for the origin. This value must be unique within the distribution.</p>
     *         <p>
     *         Use this value to specify the <code>TargetOriginId</code> in a <code>CacheBehavior</code> or
     *         <code>DefaultCacheBehavior</code>.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The domain name for the origin.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
     * >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The domain name for the origin.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
     *         >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * An optional path that CloudFront appends to the origin domain name when CloudFront requests content from the
     * origin.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginPath"
     * >Origin Path</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return An optional path that CloudFront appends to the origin domain name when CloudFront requests content from
     *         the origin.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginPath"
     *         >Origin Path</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public String originPath() {
        return originPath;
    }

    /**
     * <p>
     * A list of HTTP header names and values that CloudFront adds to requests it sends to the origin.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html">Adding
     * Custom Headers to Origin Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return A list of HTTP header names and values that CloudFront adds to requests it sends to the origin.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html"
     *         >Adding Custom Headers to Origin Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public CustomHeaders customHeaders() {
        return customHeaders;
    }

    /**
     * <p>
     * Use this type to specify an origin that is an Amazon S3 bucket that is <i> <b>not</b> </i> configured with static
     * website hosting. To specify any other type of origin, including an Amazon S3 bucket that is configured with
     * static website hosting, use the <code>CustomOriginConfig</code> type instead.
     * </p>
     * 
     * @return Use this type to specify an origin that is an Amazon S3 bucket that is <i> <b>not</b> </i> configured
     *         with static website hosting. To specify any other type of origin, including an Amazon S3 bucket that is
     *         configured with static website hosting, use the <code>CustomOriginConfig</code> type instead.
     */
    public S3OriginConfig s3OriginConfig() {
        return s3OriginConfig;
    }

    /**
     * <p>
     * Use this type to specify an origin that is a content container or HTTP server, including an Amazon S3 bucket that
     * is configured with static website hosting. To specify an Amazon S3 bucket that is <i> <b>not</b> </i> configured
     * with static website hosting, use the <code>S3OriginConfig</code> type instead.
     * </p>
     * 
     * @return Use this type to specify an origin that is a content container or HTTP server, including an Amazon S3
     *         bucket that is configured with static website hosting. To specify an Amazon S3 bucket that is <i>
     *         <b>not</b> </i> configured with static website hosting, use the <code>S3OriginConfig</code> type instead.
     */
    public CustomOriginConfig customOriginConfig() {
        return customOriginConfig;
    }

    /**
     * <p>
     * The number of times that CloudFront attempts to connect to the origin. The minimum number is 1, the maximum is 3,
     * and the default (if you don’t specify otherwise) is 3.
     * </p>
     * <p>
     * For a custom origin (including an Amazon S3 bucket that’s configured with static website hosting), this value
     * also specifies the number of times that CloudFront attempts to get a response from the origin, in the case of an
     * <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout"
     * >Origin Response Timeout</a>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts"
     * >Origin Connection Attempts</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The number of times that CloudFront attempts to connect to the origin. The minimum number is 1, the
     *         maximum is 3, and the default (if you don’t specify otherwise) is 3.</p>
     *         <p>
     *         For a custom origin (including an Amazon S3 bucket that’s configured with static website hosting), this
     *         value also specifies the number of times that CloudFront attempts to get a response from the origin, in
     *         the case of an <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout"
     *         >Origin Response Timeout</a>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts"
     *         >Origin Connection Attempts</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public Integer connectionAttempts() {
        return connectionAttempts;
    }

    /**
     * <p>
     * The number of seconds that CloudFront waits when trying to establish a connection to the origin. The minimum
     * timeout is 1 second, the maximum is 10 seconds, and the default (if you don’t specify otherwise) is 10 seconds.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout"
     * >Origin Connection Timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The number of seconds that CloudFront waits when trying to establish a connection to the origin. The
     *         minimum timeout is 1 second, the maximum is 10 seconds, and the default (if you don’t specify otherwise)
     *         is 10 seconds.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout"
     *         >Origin Connection Timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public Integer connectionTimeout() {
        return connectionTimeout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(originPath());
        hashCode = 31 * hashCode + Objects.hashCode(customHeaders());
        hashCode = 31 * hashCode + Objects.hashCode(s3OriginConfig());
        hashCode = 31 * hashCode + Objects.hashCode(customOriginConfig());
        hashCode = 31 * hashCode + Objects.hashCode(connectionAttempts());
        hashCode = 31 * hashCode + Objects.hashCode(connectionTimeout());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Origin)) {
            return false;
        }
        Origin other = (Origin) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(domainName(), other.domainName())
                && Objects.equals(originPath(), other.originPath()) && Objects.equals(customHeaders(), other.customHeaders())
                && Objects.equals(s3OriginConfig(), other.s3OriginConfig())
                && Objects.equals(customOriginConfig(), other.customOriginConfig())
                && Objects.equals(connectionAttempts(), other.connectionAttempts())
                && Objects.equals(connectionTimeout(), other.connectionTimeout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Origin").add("Id", id()).add("DomainName", domainName()).add("OriginPath", originPath())
                .add("CustomHeaders", customHeaders()).add("S3OriginConfig", s3OriginConfig())
                .add("CustomOriginConfig", customOriginConfig()).add("ConnectionAttempts", connectionAttempts())
                .add("ConnectionTimeout", connectionTimeout()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "OriginPath":
            return Optional.ofNullable(clazz.cast(originPath()));
        case "CustomHeaders":
            return Optional.ofNullable(clazz.cast(customHeaders()));
        case "S3OriginConfig":
            return Optional.ofNullable(clazz.cast(s3OriginConfig()));
        case "CustomOriginConfig":
            return Optional.ofNullable(clazz.cast(customOriginConfig()));
        case "ConnectionAttempts":
            return Optional.ofNullable(clazz.cast(connectionAttempts()));
        case "ConnectionTimeout":
            return Optional.ofNullable(clazz.cast(connectionTimeout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Origin, T> g) {
        return obj -> g.apply((Origin) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Origin> {
        /**
         * <p>
         * A unique identifier for the origin. This value must be unique within the distribution.
         * </p>
         * <p>
         * Use this value to specify the <code>TargetOriginId</code> in a <code>CacheBehavior</code> or
         * <code>DefaultCacheBehavior</code>.
         * </p>
         * 
         * @param id
         *        A unique identifier for the origin. This value must be unique within the distribution.</p>
         *        <p>
         *        Use this value to specify the <code>TargetOriginId</code> in a <code>CacheBehavior</code> or
         *        <code>DefaultCacheBehavior</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The domain name for the origin.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
         * >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param domainName
         *        The domain name for the origin.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName"
         *        >Origin Domain Name</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * An optional path that CloudFront appends to the origin domain name when CloudFront requests content from the
         * origin.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginPath"
         * >Origin Path</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param originPath
         *        An optional path that CloudFront appends to the origin domain name when CloudFront requests content
         *        from the origin.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginPath"
         *        >Origin Path</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originPath(String originPath);

        /**
         * <p>
         * A list of HTTP header names and values that CloudFront adds to requests it sends to the origin.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html"
         * >Adding Custom Headers to Origin Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param customHeaders
         *        A list of HTTP header names and values that CloudFront adds to requests it sends to the origin.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html"
         *        >Adding Custom Headers to Origin Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customHeaders(CustomHeaders customHeaders);

        /**
         * <p>
         * A list of HTTP header names and values that CloudFront adds to requests it sends to the origin.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html"
         * >Adding Custom Headers to Origin Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link CustomHeaders.Builder} avoiding the need to
         * create one manually via {@link CustomHeaders#builder()}.
         *
         * When the {@link Consumer} completes, {@link CustomHeaders.Builder#build()} is called immediately and its
         * result is passed to {@link #customHeaders(CustomHeaders)}.
         * 
         * @param customHeaders
         *        a consumer that will call methods on {@link CustomHeaders.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customHeaders(CustomHeaders)
         */
        default Builder customHeaders(Consumer<CustomHeaders.Builder> customHeaders) {
            return customHeaders(CustomHeaders.builder().applyMutation(customHeaders).build());
        }

        /**
         * <p>
         * Use this type to specify an origin that is an Amazon S3 bucket that is <i> <b>not</b> </i> configured with
         * static website hosting. To specify any other type of origin, including an Amazon S3 bucket that is configured
         * with static website hosting, use the <code>CustomOriginConfig</code> type instead.
         * </p>
         * 
         * @param s3OriginConfig
         *        Use this type to specify an origin that is an Amazon S3 bucket that is <i> <b>not</b> </i> configured
         *        with static website hosting. To specify any other type of origin, including an Amazon S3 bucket that
         *        is configured with static website hosting, use the <code>CustomOriginConfig</code> type instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3OriginConfig(S3OriginConfig s3OriginConfig);

        /**
         * <p>
         * Use this type to specify an origin that is an Amazon S3 bucket that is <i> <b>not</b> </i> configured with
         * static website hosting. To specify any other type of origin, including an Amazon S3 bucket that is configured
         * with static website hosting, use the <code>CustomOriginConfig</code> type instead.
         * </p>
         * This is a convenience that creates an instance of the {@link S3OriginConfig.Builder} avoiding the need to
         * create one manually via {@link S3OriginConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3OriginConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #s3OriginConfig(S3OriginConfig)}.
         * 
         * @param s3OriginConfig
         *        a consumer that will call methods on {@link S3OriginConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3OriginConfig(S3OriginConfig)
         */
        default Builder s3OriginConfig(Consumer<S3OriginConfig.Builder> s3OriginConfig) {
            return s3OriginConfig(S3OriginConfig.builder().applyMutation(s3OriginConfig).build());
        }

        /**
         * <p>
         * Use this type to specify an origin that is a content container or HTTP server, including an Amazon S3 bucket
         * that is configured with static website hosting. To specify an Amazon S3 bucket that is <i> <b>not</b> </i>
         * configured with static website hosting, use the <code>S3OriginConfig</code> type instead.
         * </p>
         * 
         * @param customOriginConfig
         *        Use this type to specify an origin that is a content container or HTTP server, including an Amazon S3
         *        bucket that is configured with static website hosting. To specify an Amazon S3 bucket that is <i>
         *        <b>not</b> </i> configured with static website hosting, use the <code>S3OriginConfig</code> type
         *        instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customOriginConfig(CustomOriginConfig customOriginConfig);

        /**
         * <p>
         * Use this type to specify an origin that is a content container or HTTP server, including an Amazon S3 bucket
         * that is configured with static website hosting. To specify an Amazon S3 bucket that is <i> <b>not</b> </i>
         * configured with static website hosting, use the <code>S3OriginConfig</code> type instead.
         * </p>
         * This is a convenience that creates an instance of the {@link CustomOriginConfig.Builder} avoiding the need to
         * create one manually via {@link CustomOriginConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CustomOriginConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #customOriginConfig(CustomOriginConfig)}.
         * 
         * @param customOriginConfig
         *        a consumer that will call methods on {@link CustomOriginConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customOriginConfig(CustomOriginConfig)
         */
        default Builder customOriginConfig(Consumer<CustomOriginConfig.Builder> customOriginConfig) {
            return customOriginConfig(CustomOriginConfig.builder().applyMutation(customOriginConfig).build());
        }

        /**
         * <p>
         * The number of times that CloudFront attempts to connect to the origin. The minimum number is 1, the maximum
         * is 3, and the default (if you don’t specify otherwise) is 3.
         * </p>
         * <p>
         * For a custom origin (including an Amazon S3 bucket that’s configured with static website hosting), this value
         * also specifies the number of times that CloudFront attempts to get a response from the origin, in the case of
         * an <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout"
         * >Origin Response Timeout</a>.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts"
         * >Origin Connection Attempts</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param connectionAttempts
         *        The number of times that CloudFront attempts to connect to the origin. The minimum number is 1, the
         *        maximum is 3, and the default (if you don’t specify otherwise) is 3.</p>
         *        <p>
         *        For a custom origin (including an Amazon S3 bucket that’s configured with static website hosting),
         *        this value also specifies the number of times that CloudFront attempts to get a response from the
         *        origin, in the case of an <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout"
         *        >Origin Response Timeout</a>.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts"
         *        >Origin Connection Attempts</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionAttempts(Integer connectionAttempts);

        /**
         * <p>
         * The number of seconds that CloudFront waits when trying to establish a connection to the origin. The minimum
         * timeout is 1 second, the maximum is 10 seconds, and the default (if you don’t specify otherwise) is 10
         * seconds.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout"
         * >Origin Connection Timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param connectionTimeout
         *        The number of seconds that CloudFront waits when trying to establish a connection to the origin. The
         *        minimum timeout is 1 second, the maximum is 10 seconds, and the default (if you don’t specify
         *        otherwise) is 10 seconds.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout"
         *        >Origin Connection Timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionTimeout(Integer connectionTimeout);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String domainName;

        private String originPath;

        private CustomHeaders customHeaders;

        private S3OriginConfig s3OriginConfig;

        private CustomOriginConfig customOriginConfig;

        private Integer connectionAttempts;

        private Integer connectionTimeout;

        private BuilderImpl() {
        }

        private BuilderImpl(Origin model) {
            id(model.id);
            domainName(model.domainName);
            originPath(model.originPath);
            customHeaders(model.customHeaders);
            s3OriginConfig(model.s3OriginConfig);
            customOriginConfig(model.customOriginConfig);
            connectionAttempts(model.connectionAttempts);
            connectionTimeout(model.connectionTimeout);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getOriginPath() {
            return originPath;
        }

        @Override
        public final Builder originPath(String originPath) {
            this.originPath = originPath;
            return this;
        }

        public final void setOriginPath(String originPath) {
            this.originPath = originPath;
        }

        public final CustomHeaders.Builder getCustomHeaders() {
            return customHeaders != null ? customHeaders.toBuilder() : null;
        }

        @Override
        public final Builder customHeaders(CustomHeaders customHeaders) {
            this.customHeaders = customHeaders;
            return this;
        }

        public final void setCustomHeaders(CustomHeaders.BuilderImpl customHeaders) {
            this.customHeaders = customHeaders != null ? customHeaders.build() : null;
        }

        public final S3OriginConfig.Builder getS3OriginConfig() {
            return s3OriginConfig != null ? s3OriginConfig.toBuilder() : null;
        }

        @Override
        public final Builder s3OriginConfig(S3OriginConfig s3OriginConfig) {
            this.s3OriginConfig = s3OriginConfig;
            return this;
        }

        public final void setS3OriginConfig(S3OriginConfig.BuilderImpl s3OriginConfig) {
            this.s3OriginConfig = s3OriginConfig != null ? s3OriginConfig.build() : null;
        }

        public final CustomOriginConfig.Builder getCustomOriginConfig() {
            return customOriginConfig != null ? customOriginConfig.toBuilder() : null;
        }

        @Override
        public final Builder customOriginConfig(CustomOriginConfig customOriginConfig) {
            this.customOriginConfig = customOriginConfig;
            return this;
        }

        public final void setCustomOriginConfig(CustomOriginConfig.BuilderImpl customOriginConfig) {
            this.customOriginConfig = customOriginConfig != null ? customOriginConfig.build() : null;
        }

        public final Integer getConnectionAttempts() {
            return connectionAttempts;
        }

        @Override
        public final Builder connectionAttempts(Integer connectionAttempts) {
            this.connectionAttempts = connectionAttempts;
            return this;
        }

        public final void setConnectionAttempts(Integer connectionAttempts) {
            this.connectionAttempts = connectionAttempts;
        }

        public final Integer getConnectionTimeout() {
            return connectionTimeout;
        }

        @Override
        public final Builder connectionTimeout(Integer connectionTimeout) {
            this.connectionTimeout = connectionTimeout;
            return this;
        }

        public final void setConnectionTimeout(Integer connectionTimeout) {
            this.connectionTimeout = connectionTimeout;
        }

        @Override
        public Origin build() {
            return new Origin(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
