/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that specifies the AWS accounts, if any, that you want to allow to create signed URLs for private
 * content.
 * </p>
 * <p>
 * If you want to require signed URLs in requests for objects in the target origin that match the
 * <code>PathPattern</code> for this cache behavior, specify <code>true</code> for <code>Enabled</code>, and specify the
 * applicable values for <code>Quantity</code> and <code>Items</code>. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">Serving Private Content
 * through CloudFront</a> in the <i> Amazon CloudFront Developer Guide</i>.
 * </p>
 * <p>
 * If you don't want to require signed URLs in requests for objects that match <code>PathPattern</code>, specify
 * <code>false</code> for <code>Enabled</code> and <code>0</code> for <code>Quantity</code>. Omit <code>Items</code>.
 * </p>
 * <p>
 * To add, change, or remove one or more trusted signers, change <code>Enabled</code> to <code>true</code> (if it's
 * currently <code>false</code>), change <code>Quantity</code> as applicable, and specify all of the trusted signers
 * that you want to include in the updated distribution.
 * </p>
 * <p>
 * For more information about updating the distribution configuration, see <a
 * href="https://docs.aws.amazon.com/cloudfront/latest/APIReference/DistributionConfig.html">DistributionConfig</a> in
 * the <i>Amazon CloudFront API Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TrustedSigners implements SdkPojo, Serializable, ToCopyableBuilder<TrustedSigners.Builder, TrustedSigners> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled")
            .getter(getter(TrustedSigners::enabled))
            .setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled")
                    .unmarshallLocationName("Enabled").build()).build();

    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Quantity")
            .getter(getter(TrustedSigners::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Items")
            .getter(getter(TrustedSigners::items))
            .setter(setter(Builder::items))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AwsAccountNumber")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AwsAccountNumber").unmarshallLocationName("AwsAccountNumber")
                                                    .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD, QUANTITY_FIELD,
            ITEMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final Integer quantity;

    private final List<String> items;

    private TrustedSigners(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * Specifies whether you want to require viewers to use signed URLs to access the files specified by
     * <code>PathPattern</code> and <code>TargetOriginId</code>.
     * </p>
     * 
     * @return Specifies whether you want to require viewers to use signed URLs to access the files specified by
     *         <code>PathPattern</code> and <code>TargetOriginId</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The number of trusted signers for this cache behavior.
     * </p>
     * 
     * @return The number of trusted signers for this cache behavior.
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * Returns true if the Items property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasItems() {
        return items != null && !(items instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If <code>Quantity</code>
     * is <code>0</code>, you can omit <code>Items</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasItems()} to see if a value was sent in this field.
     * </p>
     * 
     * @return <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
     *         <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
     */
    public List<String> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(items());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrustedSigners)) {
            return false;
        }
        TrustedSigners other = (TrustedSigners) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(quantity(), other.quantity())
                && Objects.equals(items(), other.items());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TrustedSigners").add("Enabled", enabled()).add("Quantity", quantity()).add("Items", items())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(items()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TrustedSigners, T> g) {
        return obj -> g.apply((TrustedSigners) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TrustedSigners> {
        /**
         * <p>
         * Specifies whether you want to require viewers to use signed URLs to access the files specified by
         * <code>PathPattern</code> and <code>TargetOriginId</code>.
         * </p>
         * 
         * @param enabled
         *        Specifies whether you want to require viewers to use signed URLs to access the files specified by
         *        <code>PathPattern</code> and <code>TargetOriginId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The number of trusted signers for this cache behavior.
         * </p>
         * 
         * @param quantity
         *        The number of trusted signers for this cache behavior.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         * <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         *        <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         * <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        <b>Optional</b>: A complex type that contains trusted signers for this cache behavior. If
         *        <code>Quantity</code> is <code>0</code>, you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TrustedSigners model) {
            enabled(model.enabled);
            quantity(model.quantity);
            items(model.items);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = AwsAccountNumberListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = AwsAccountNumberListCopier.copy(items);
        }

        @Override
        public TrustedSigners build() {
            return new TrustedSigners(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
