/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eventbridge.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateEventBusRequest extends EventBridgeRequest implements
        ToCopyableBuilder<CreateEventBusRequest.Builder, CreateEventBusRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateEventBusRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> EVENT_SOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventSourceName").getter(getter(CreateEventBusRequest::eventSourceName))
            .setter(setter(Builder::eventSourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventSourceName").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateEventBusRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            EVENT_SOURCE_NAME_FIELD, TAGS_FIELD));

    private final String name;

    private final String eventSourceName;

    private final List<Tag> tags;

    private CreateEventBusRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.eventSourceName = builder.eventSourceName;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the new event bus.
     * </p>
     * <p>
     * Event bus names cannot contain the / character. You can't use the name <code>default</code> for a custom event
     * bus, as this name is already used for your account's default event bus.
     * </p>
     * <p>
     * If this is a partner event bus, the name must exactly match the name of the partner event source that this event
     * bus is matched to.
     * </p>
     * 
     * @return The name of the new event bus. </p>
     *         <p>
     *         Event bus names cannot contain the / character. You can't use the name <code>default</code> for a custom
     *         event bus, as this name is already used for your account's default event bus.
     *         </p>
     *         <p>
     *         If this is a partner event bus, the name must exactly match the name of the partner event source that
     *         this event bus is matched to.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * If you are creating a partner event bus, this specifies the partner event source that the new event bus will be
     * matched with.
     * </p>
     * 
     * @return If you are creating a partner event bus, this specifies the partner event source that the new event bus
     *         will be matched with.
     */
    public String eventSourceName() {
        return eventSourceName;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags to associate with the event bus.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Tags to associate with the event bus.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(eventSourceName());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateEventBusRequest)) {
            return false;
        }
        CreateEventBusRequest other = (CreateEventBusRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(eventSourceName(), other.eventSourceName())
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateEventBusRequest").add("Name", name()).add("EventSourceName", eventSourceName())
                .add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "EventSourceName":
            return Optional.ofNullable(clazz.cast(eventSourceName()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateEventBusRequest, T> g) {
        return obj -> g.apply((CreateEventBusRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EventBridgeRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateEventBusRequest> {
        /**
         * <p>
         * The name of the new event bus.
         * </p>
         * <p>
         * Event bus names cannot contain the / character. You can't use the name <code>default</code> for a custom
         * event bus, as this name is already used for your account's default event bus.
         * </p>
         * <p>
         * If this is a partner event bus, the name must exactly match the name of the partner event source that this
         * event bus is matched to.
         * </p>
         * 
         * @param name
         *        The name of the new event bus. </p>
         *        <p>
         *        Event bus names cannot contain the / character. You can't use the name <code>default</code> for a
         *        custom event bus, as this name is already used for your account's default event bus.
         *        </p>
         *        <p>
         *        If this is a partner event bus, the name must exactly match the name of the partner event source that
         *        this event bus is matched to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If you are creating a partner event bus, this specifies the partner event source that the new event bus will
         * be matched with.
         * </p>
         * 
         * @param eventSourceName
         *        If you are creating a partner event bus, this specifies the partner event source that the new event
         *        bus will be matched with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSourceName(String eventSourceName);

        /**
         * <p>
         * Tags to associate with the event bus.
         * </p>
         * 
         * @param tags
         *        Tags to associate with the event bus.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Tags to associate with the event bus.
         * </p>
         * 
         * @param tags
         *        Tags to associate with the event bus.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Tags to associate with the event bus.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EventBridgeRequest.BuilderImpl implements Builder {
        private String name;

        private String eventSourceName;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateEventBusRequest model) {
            super(model);
            name(model.name);
            eventSourceName(model.eventSourceName);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getEventSourceName() {
            return eventSourceName;
        }

        @Override
        public final Builder eventSourceName(String eventSourceName) {
            this.eventSourceName = eventSourceName;
            return this;
        }

        public final void setEventSourceName(String eventSourceName) {
            this.eventSourceName = eventSourceName;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateEventBusRequest build() {
            return new CreateEventBusRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
