/*
 * Decompiled with CFR 0.152.
 */
package com.blazebit.storage.nfs.executor;

import com.google.common.primitives.Longs;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.DosFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.UserDefinedFileAttributeView;
import java.nio.file.attribute.UserPrincipal;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.security.Principal;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.security.auth.Subject;
import org.dcache.auth.GidPrincipal;
import org.dcache.auth.UidPrincipal;
import org.dcache.nfs.status.ExistException;
import org.dcache.nfs.status.NoEntException;
import org.dcache.nfs.status.NotEmptyException;
import org.dcache.nfs.status.NotSuppException;
import org.dcache.nfs.status.PermException;
import org.dcache.nfs.status.ServerFaultException;
import org.dcache.nfs.v4.NfsIdMapping;
import org.dcache.nfs.v4.SimpleIdMap;
import org.dcache.nfs.v4.xdr.nfsace4;
import org.dcache.nfs.vfs.AclCheckable;
import org.dcache.nfs.vfs.DirectoryEntry;
import org.dcache.nfs.vfs.FsStat;
import org.dcache.nfs.vfs.Inode;
import org.dcache.nfs.vfs.Stat;
import org.dcache.nfs.vfs.VirtualFileSystem;

public class NioVirtualFileSystem
implements VirtualFileSystem {
    private static final Logger LOG = Logger.getLogger(NioVirtualFileSystem.class.getName());
    public static final int S_IRUSR = 256;
    public static final int S_IWUSR = 128;
    public static final int S_IXUSR = 64;
    public static final int S_IRGRP = 32;
    public static final int S_IWGRP = 16;
    public static final int S_IXGRP = 8;
    public static final int S_IROTH = 4;
    public static final int S_IWOTH = 2;
    public static final int S_IXOTH = 1;
    public static final int S_IFSOCK = 49152;
    public static final int S_IFLNK = 40960;
    public static final int S_IFREG = 32768;
    public static final int S_IFBLK = 24576;
    public static final int S_IFDIR = 16384;
    private final Path _root;
    private final ConcurrentMap<Long, Path> inodeToPath = new ConcurrentHashMap<Long, Path>();
    private final ConcurrentMap<Path, Long> pathToInode = new ConcurrentHashMap<Path, Long>();
    private final AtomicLong fileId = new AtomicLong(1L);
    private final NfsIdMapping _idMapper = new SimpleIdMap();
    private final UserPrincipalLookupService _lookupService;
    private final FileSystem fs;
    private final Class<? extends BasicFileAttributeView> attributeClass;

    public NioVirtualFileSystem(Path root) throws IOException {
        this.fs = root.getFileSystem();
        this.attributeClass = this.fs.supportedFileAttributeViews().contains("posix") ? PosixFileAttributeView.class : DosFileAttributeView.class;
        this._root = root;
        this.map(this.fileId.getAndIncrement(), this._root);
        this._lookupService = this.fs.getUserPrincipalLookupService();
    }

    private Inode toFileHandle(long inodeNumber) {
        return Inode.forFile((byte[])Longs.toByteArray((long)inodeNumber));
    }

    private long toInodeNumber(Inode inode) {
        return Longs.fromByteArray((byte[])inode.getFileId());
    }

    private Path resolveInode(long inodeNumber) throws NoEntException {
        Path path = (Path)this.inodeToPath.get(inodeNumber);
        if (path == null) {
            throw new NoEntException("inode #" + inodeNumber);
        }
        return path;
    }

    private long resolvePath(Path path) throws NoEntException {
        Long inodeNumber = (Long)this.pathToInode.get(path);
        if (inodeNumber == null) {
            throw new NoEntException("path " + path);
        }
        return inodeNumber;
    }

    private long resolveOrMapPath(Path path) throws NoEntException {
        return this.pathToInode.computeIfAbsent(path, p -> {
            long inodeNumber = this.fileId.getAndIncrement();
            this.inodeToPath.putIfAbsent(inodeNumber, (Path)p);
            return inodeNumber;
        });
    }

    private void map(long inodeNumber, Path path) {
        if (this.inodeToPath.putIfAbsent(inodeNumber, path) != null) {
            throw new IllegalStateException();
        }
        Long otherInodeNumber = this.pathToInode.putIfAbsent(path, inodeNumber);
        if (otherInodeNumber != null) {
            if (this.inodeToPath.remove(inodeNumber) != path) {
                throw new IllegalStateException("cant map, rollback failed");
            }
            throw new IllegalStateException("path ");
        }
    }

    private void unmap(long inodeNumber, Path path) {
        Path removedPath = (Path)this.inodeToPath.remove(inodeNumber);
        if (!path.equals(removedPath)) {
            throw new IllegalStateException();
        }
        if ((Long)this.pathToInode.remove(path) != inodeNumber) {
            throw new IllegalStateException();
        }
    }

    private void remap(long inodeNumber, Path oldPath, Path newPath) {
        this.unmap(inodeNumber, oldPath);
        this.map(inodeNumber, newPath);
    }

    public Inode create(Inode parent, Stat.Type type, String path, Subject subject, int mode) throws IOException {
        long parentInodeNumber = this.toInodeNumber(parent);
        Path parentPath = this.resolveInode(parentInodeNumber);
        Path newPath = parentPath.resolve(path);
        try {
            Files.createFile(newPath, new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            throw new ExistException("path " + newPath);
        }
        long newInodeNumber = this.fileId.getAndIncrement();
        this.map(newInodeNumber, newPath);
        this.setOwnershipAndMode(newPath, subject, mode);
        return this.toFileHandle(newInodeNumber);
    }

    public FsStat getFsStat() throws IOException {
        FileStore store = Files.getFileStore(this._root);
        long total = store.getTotalSpace();
        long free = store.getUsableSpace();
        return new FsStat(total, Long.MAX_VALUE, total - free, (long)this.pathToInode.size());
    }

    public Inode getRootInode() throws IOException {
        return this.toFileHandle(1L);
    }

    public Inode lookup(Inode parent, String path) throws IOException {
        long parentInodeNumber = this.toInodeNumber(parent);
        Path parentPath = this.resolveInode(parentInodeNumber);
        Path child = path.equals(".") ? parentPath : (path.equals("..") ? parentPath.getParent() : parentPath.resolve(path));
        long childInodeNumber = this.resolveOrMapPath(child);
        return this.toFileHandle(childInodeNumber);
    }

    public Inode link(Inode parent, Inode existing, String target, Subject subject) throws IOException {
        long parentInodeNumber = this.toInodeNumber(parent);
        Path parentPath = this.resolveInode(parentInodeNumber);
        long existingInodeNumber = this.toInodeNumber(existing);
        Path existingPath = this.resolveInode(existingInodeNumber);
        Path targetPath = parentPath.resolve(target);
        try {
            Files.createLink(targetPath, existingPath);
        }
        catch (UnsupportedOperationException e) {
            throw new NotSuppException("Not supported", (Throwable)e);
        }
        catch (FileAlreadyExistsException e) {
            throw new ExistException("Path exists " + target, (Throwable)e);
        }
        catch (SecurityException e) {
            throw new PermException("Permission denied: " + e.getMessage(), (Throwable)e);
        }
        catch (IOException e) {
            throw new ServerFaultException("Failed to create: " + e.getMessage(), (Throwable)e);
        }
        long newInodeNumber = this.fileId.getAndIncrement();
        this.map(newInodeNumber, targetPath);
        return this.toFileHandle(newInodeNumber);
    }

    public org.dcache.nfs.vfs.DirectoryStream list(Inode inode, byte[] bytes, long l) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        ArrayList<DirectoryEntry> list = new ArrayList<DirectoryEntry>();
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(path);){
            int cookie = 2;
            for (Path p : ds) {
                if ((long)(++cookie) <= l) continue;
                long ino = this.resolveOrMapPath(p);
                list.add(new DirectoryEntry(p.getFileName().toString(), this.toFileHandle(ino), this.statPath(p, ino), (long)cookie));
            }
        }
        return new org.dcache.nfs.vfs.DirectoryStream(list);
    }

    public byte[] directoryVerifier(Inode inode) throws IOException {
        return org.dcache.nfs.vfs.DirectoryStream.ZERO_VERIFIER;
    }

    public Inode mkdir(Inode parent, String path, Subject subject, int mode) throws IOException {
        long parentInodeNumber = this.toInodeNumber(parent);
        Path parentPath = this.resolveInode(parentInodeNumber);
        Path newPath = parentPath.resolve(path);
        try {
            Files.createDirectory(newPath, new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            throw new ExistException("path " + newPath);
        }
        long newInodeNumber = this.fileId.getAndIncrement();
        this.map(newInodeNumber, newPath);
        this.setOwnershipAndMode(newPath, subject, mode);
        return this.toFileHandle(newInodeNumber);
    }

    private void setOwnershipAndMode(Path target, Subject subject, int mode) {
        Set<PosixFilePermission> permission = NioVirtualFileSystem.modeToPermissions(mode);
        int uid = -1;
        int gid = -1;
        for (Principal principal : subject.getPrincipals()) {
            if (principal instanceof UidPrincipal) {
                uid = (int)((UidPrincipal)principal).getUid();
            }
            if (!(principal instanceof GidPrincipal)) continue;
            gid = (int)((GidPrincipal)principal).getGid();
        }
        if (uid != -1) {
            try {
                Files.setAttribute(target, "owner:owner", NioVirtualFileSystem.asUserPrincipal(uid), LinkOption.NOFOLLOW_LINKS);
            }
            catch (IOException e) {
                LOG.log(Level.WARNING, e, () -> "Unable to chown file " + target + ": " + e.getMessage());
            }
        } else {
            LOG.log(Level.WARNING, "File created without uid: {}", target);
        }
        if (gid != -1) {
            try {
                Files.setAttribute(target, "posix:group", NioVirtualFileSystem.asGroupPrincipal(gid), LinkOption.NOFOLLOW_LINKS);
            }
            catch (IOException e) {
                LOG.log(Level.WARNING, e, () -> "Unable to chown file " + target + ": " + e.getMessage());
            }
        } else {
            LOG.log(Level.WARNING, "File created without gid: {}", target);
        }
        try {
            Files.setAttribute(target, "posix:permissions", permission, LinkOption.NOFOLLOW_LINKS);
        }
        catch (IOException e) {
            LOG.log(Level.WARNING, e, () -> "Unable to set mode of file " + target + ": " + e.getMessage());
        }
    }

    public boolean move(Inode src, String oldName, Inode dest, String newName) throws IOException {
        long currentParentInodeNumber = this.toInodeNumber(src);
        Path currentParentPath = this.resolveInode(currentParentInodeNumber);
        long destParentInodeNumber = this.toInodeNumber(dest);
        Path destPath = this.resolveInode(destParentInodeNumber);
        Path currentPath = currentParentPath.resolve(oldName);
        long targetInodeNumber = this.resolvePath(currentPath);
        Path newPath = destPath.resolve(newName);
        try {
            Files.move(currentPath, newPath, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (FileAlreadyExistsException e) {
            throw new ExistException("path " + newPath);
        }
        this.remap(targetInodeNumber, currentPath, newPath);
        return true;
    }

    public Inode parentOf(Inode inode) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        if (inodeNumber == 1L) {
            throw new NoEntException("no parent");
        }
        Path path = this.resolveInode(inodeNumber);
        Path parentPath = path.getParent();
        long parentInodeNumber = this.resolvePath(parentPath);
        return this.toFileHandle(parentInodeNumber);
    }

    public int read(Inode inode, byte[] data, long offset, int count) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        ByteBuffer destBuffer = ByteBuffer.wrap(data, 0, count);
        try (FileChannel channel = FileChannel.open(path, StandardOpenOption.READ);){
            int n = channel.read(destBuffer, offset);
            return n;
        }
    }

    public String readlink(Inode inode) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        return Files.readSymbolicLink(path).toString();
    }

    public void remove(Inode parent, String path) throws IOException {
        long parentInodeNumber = this.toInodeNumber(parent);
        Path parentPath = this.resolveInode(parentInodeNumber);
        Path targetPath = parentPath.resolve(path);
        long targetInodeNumber = this.resolvePath(targetPath);
        try {
            Files.delete(targetPath);
        }
        catch (DirectoryNotEmptyException e) {
            throw new NotEmptyException("dir " + targetPath + " is note empty", (Throwable)e);
        }
        this.unmap(targetInodeNumber, targetPath);
    }

    public Inode symlink(Inode parent, String linkName, String targetName, Subject subject, int mode) throws IOException {
        long parentInodeNumber = this.toInodeNumber(parent);
        Path parentPath = this.resolveInode(parentInodeNumber);
        Path link = parentPath.resolve(linkName);
        Path target = parentPath.resolve(targetName);
        if (!targetName.startsWith("/")) {
            target = parentPath.relativize(target);
        }
        try {
            Files.createSymbolicLink(link, target, new FileAttribute[0]);
        }
        catch (UnsupportedOperationException e) {
            throw new NotSuppException("Not supported", (Throwable)e);
        }
        catch (FileAlreadyExistsException e) {
            throw new ExistException("Path exists " + linkName, (Throwable)e);
        }
        catch (SecurityException e) {
            throw new PermException("Permission denied: " + e.getMessage(), (Throwable)e);
        }
        catch (IOException e) {
            throw new ServerFaultException("Failed to create: " + e.getMessage(), (Throwable)e);
        }
        this.setOwnershipAndMode(link, subject, mode);
        long newInodeNumber = this.fileId.getAndIncrement();
        this.map(newInodeNumber, link);
        return this.toFileHandle(newInodeNumber);
    }

    public VirtualFileSystem.WriteResult write(Inode inode, byte[] data, long offset, int count, VirtualFileSystem.StabilityLevel stabilityLevel) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        ByteBuffer srcBuffer = ByteBuffer.wrap(data, 0, count);
        try (FileChannel channel = FileChannel.open(path, StandardOpenOption.WRITE);){
            int bytesWritten = channel.write(srcBuffer, offset);
            VirtualFileSystem.WriteResult writeResult = new VirtualFileSystem.WriteResult(VirtualFileSystem.StabilityLevel.FILE_SYNC, bytesWritten);
            return writeResult;
        }
    }

    public void commit(Inode inode, long l, int i) throws IOException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    private Stat statPath(Path p, long inodeNumber) throws IOException {
        BasicFileAttributes attrs = Files.getFileAttributeView(p, this.attributeClass, LinkOption.NOFOLLOW_LINKS).readAttributes();
        Stat stat = new Stat();
        stat.setATime(attrs.lastAccessTime().toMillis());
        stat.setCTime(attrs.creationTime().toMillis());
        stat.setMTime(attrs.lastModifiedTime().toMillis());
        if (this.attributeClass == PosixFileAttributeView.class) {
            stat.setGid(Integer.parseInt(((Principal)Files.getAttribute(p, "posix:group", LinkOption.NOFOLLOW_LINKS)).getName()));
            stat.setUid(Integer.parseInt(((Principal)Files.getAttribute(p, "owner:owner", LinkOption.NOFOLLOW_LINKS)).getName()));
            Set permissions = (Set)Files.getAttribute(p, "posix:permissions", LinkOption.NOFOLLOW_LINKS);
            stat.setMode(NioVirtualFileSystem.permissionsToMode(permissions, attrs));
        } else {
            stat.setUid(0);
            stat.setGid(0);
            stat.setMode(NioVirtualFileSystem.permissionsToMode(EnumSet.allOf(PosixFilePermission.class), attrs));
        }
        stat.setNlink(1);
        stat.setDev(17);
        stat.setIno((int)inodeNumber);
        stat.setRdev(17);
        stat.setSize(attrs.size());
        stat.setFileid((long)((int)inodeNumber));
        stat.setGeneration(attrs.lastModifiedTime().toMillis());
        return stat;
    }

    public int access(Inode inode, int mode) throws IOException {
        return mode;
    }

    public Stat getattr(Inode inode) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        return this.statPath(path, inodeNumber);
    }

    public void setattr(Inode inode, Stat stat) throws IOException {
        FileTime time;
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        PosixFileAttributeView attributeView = Files.getFileAttributeView(path, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        if (stat.isDefined(Stat.StatAttribute.OWNER)) {
            try {
                String uid = String.valueOf(stat.getUid());
                UserPrincipal user = this._lookupService.lookupPrincipalByName(uid);
                attributeView.setOwner(user);
            }
            catch (IOException e) {
                throw new UnsupportedOperationException("set uid failed: " + e.getMessage(), e);
            }
        }
        if (stat.isDefined(Stat.StatAttribute.GROUP)) {
            try {
                String gid = String.valueOf(stat.getGid());
                GroupPrincipal group = this._lookupService.lookupPrincipalByGroupName(gid);
                attributeView.setGroup(group);
            }
            catch (IOException e) {
                throw new UnsupportedOperationException("set gid failed: " + e.getMessage(), e);
            }
        }
        if (stat.isDefined(Stat.StatAttribute.MODE)) {
            try {
                Files.setAttribute(path, "posix:permissions", NioVirtualFileSystem.modeToPermissions(stat.getMode()), LinkOption.NOFOLLOW_LINKS);
            }
            catch (IOException e) {
                throw new UnsupportedOperationException("set mode unsupported: " + e.getMessage(), e);
            }
        }
        if (stat.isDefined(Stat.StatAttribute.SIZE)) {
            try (RandomAccessFile raf = new RandomAccessFile(path.toFile(), "rw");){
                raf.setLength(stat.getSize());
            }
        }
        if (stat.isDefined(Stat.StatAttribute.ATIME)) {
            try {
                time = FileTime.fromMillis(stat.getCTime());
                Files.setAttribute(path, "unix:lastAccessTime", time, LinkOption.NOFOLLOW_LINKS);
            }
            catch (IOException e) {
                throw new UnsupportedOperationException("set atime failed: " + e.getMessage(), e);
            }
        }
        if (stat.isDefined(Stat.StatAttribute.MTIME)) {
            try {
                time = FileTime.fromMillis(stat.getMTime());
                Files.setAttribute(path, "unix:lastModifiedTime", time, LinkOption.NOFOLLOW_LINKS);
            }
            catch (IOException e) {
                throw new UnsupportedOperationException("set mtime failed: " + e.getMessage(), e);
            }
        }
        if (stat.isDefined(Stat.StatAttribute.CTIME)) {
            try {
                time = FileTime.fromMillis(stat.getCTime());
                Files.setAttribute(path, "unix:ctime", time, LinkOption.NOFOLLOW_LINKS);
            }
            catch (IOException e) {
                throw new UnsupportedOperationException("set ctime failed: " + e.getMessage(), e);
            }
        }
    }

    public nfsace4[] getAcl(Inode inode) throws IOException {
        return new nfsace4[0];
    }

    public void setAcl(Inode inode, nfsace4[] acl) throws IOException {
        throw new UnsupportedOperationException("No ACL support");
    }

    public boolean hasIOLayout(Inode inode) throws IOException {
        return false;
    }

    public AclCheckable getAclCheckable() {
        return AclCheckable.UNDEFINED_ALL;
    }

    public NfsIdMapping getIdMapper() {
        return this._idMapper;
    }

    public byte[] getXattr(Inode inode, String attr) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
        ByteBuffer buf = ByteBuffer.allocate(view.size(attr));
        view.read(attr, buf);
        return buf.array();
    }

    public void setXattr(Inode inode, String attr, byte[] value, VirtualFileSystem.SetXattrMode mode) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
        ByteBuffer buf = ByteBuffer.wrap(value);
        view.write(attr, buf);
    }

    public String[] listXattrs(Inode inode) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
        return view.list().toArray(new String[0]);
    }

    public void removeXattr(Inode inode, String attr) throws IOException {
        long inodeNumber = this.toInodeNumber(inode);
        Path path = this.resolveInode(inodeNumber);
        UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
        view.delete(attr);
    }

    private static Principal asUserPrincipal(final int uid) {
        return new UserPrincipal(){

            @Override
            public String getName() {
                return Integer.toString(uid);
            }
        };
    }

    private static Principal asGroupPrincipal(final int gid) {
        return new GroupPrincipal(){

            @Override
            public String getName() {
                return Integer.toString(gid);
            }
        };
    }

    private static Set<PosixFilePermission> modeToPermissions(int mode) {
        EnumSet<PosixFilePermission> perms = EnumSet.noneOf(PosixFilePermission.class);
        if ((mode & 0x100) == 256) {
            perms.add(PosixFilePermission.OWNER_READ);
        }
        if ((mode & 0x80) == 128) {
            perms.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((mode & 0x40) == 64) {
            perms.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((mode & 0x20) == 32) {
            perms.add(PosixFilePermission.GROUP_READ);
        }
        if ((mode & 0x10) == 16) {
            perms.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((mode & 8) == 8) {
            perms.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((mode & 4) == 4) {
            perms.add(PosixFilePermission.OTHERS_READ);
        }
        if ((mode & 2) == 2) {
            perms.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((mode & 1) == 1) {
            perms.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return perms;
    }

    private static int permissionsToMode(Set<PosixFilePermission> permissions, BasicFileAttributes attrs) {
        int mode = 0;
        for (PosixFilePermission p : permissions) {
            switch (p) {
                case OWNER_READ: {
                    mode |= 0x100;
                    break;
                }
                case OWNER_WRITE: {
                    mode |= 0x80;
                    break;
                }
                case OWNER_EXECUTE: {
                    mode |= 0x40;
                    break;
                }
                case GROUP_READ: {
                    mode |= 0x20;
                    break;
                }
                case GROUP_WRITE: {
                    mode |= 0x10;
                    break;
                }
                case GROUP_EXECUTE: {
                    mode |= 8;
                    break;
                }
                case OTHERS_READ: {
                    mode |= 4;
                    break;
                }
                case OTHERS_WRITE: {
                    mode |= 2;
                    break;
                }
                case OTHERS_EXECUTE: {
                    mode |= 1;
                }
            }
        }
        mode = attrs.isDirectory() ? (mode |= 0x4000) : (attrs.isRegularFile() ? (mode |= 0x8000) : (attrs.isSymbolicLink() ? (mode |= 0xA000) : (mode |= 0xC000)));
        return mode;
    }
}

