/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.couchbase.client.core.deps.io.opentelemetry.instrumentation.grpc.v1_6;

import com.couchbase.client.core.deps.io.grpc.ClientInterceptor;
import com.couchbase.client.core.deps.io.grpc.ServerInterceptor;
import com.couchbase.client.core.deps.io.grpc.Status;
import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.context.propagation.ContextPropagators;
import io.opentelemetry.instrumentation.api.instrumenter.Instrumenter;

/** Entrypoint for instrumenting gRPC servers or clients. */
public final class GrpcTelemetry {

  /** Returns a new {@link GrpcTelemetry} configured with the given {@link OpenTelemetry}. */
  public static GrpcTelemetry create(OpenTelemetry openTelemetry) {
    return builder(openTelemetry).build();
  }

  /** Returns a new {@link GrpcTelemetryBuilder} configured with the given {@link OpenTelemetry}. */
  public static GrpcTelemetryBuilder builder(OpenTelemetry openTelemetry) {
    return new GrpcTelemetryBuilder(openTelemetry);
  }

  private final Instrumenter<GrpcRequest, Status> serverInstrumenter;
  private final Instrumenter<GrpcRequest, Status> clientInstrumenter;
  private final ContextPropagators propagators;
  private final boolean captureExperimentalSpanAttributes;

  GrpcTelemetry(
      Instrumenter<GrpcRequest, Status> serverInstrumenter,
      Instrumenter<GrpcRequest, Status> clientInstrumenter,
      ContextPropagators propagators,
      boolean captureExperimentalSpanAttributes) {
    this.serverInstrumenter = serverInstrumenter;
    this.clientInstrumenter = clientInstrumenter;
    this.propagators = propagators;
    this.captureExperimentalSpanAttributes = captureExperimentalSpanAttributes;
  }

  /**
   * Returns a new {@link ClientInterceptor} for use with methods like {@link
   * com.couchbase.client.core.deps.io.grpc.ManagedChannelBuilder#intercept(ClientInterceptor...)}.
   */
  public ClientInterceptor newClientInterceptor() {
    return new TracingClientInterceptor(clientInstrumenter, propagators);
  }

  /**
   * Returns a new {@link ServerInterceptor} for use with methods like {@link
   * com.couchbase.client.core.deps.io.grpc.ServerBuilder#intercept(ServerInterceptor)}.
   */
  public ServerInterceptor newServerInterceptor() {
    return new TracingServerInterceptor(serverInstrumenter, captureExperimentalSpanAttributes);
  }
}
