/*
 * Decompiled with CFR 0.152.
 */
package com.github.marschall.memoryfilesystem;

import com.github.marschall.memoryfilesystem.AutoRelease;
import com.github.marschall.memoryfilesystem.AutoReleaseLock;
import com.github.marschall.memoryfilesystem.LockSet;
import com.github.marschall.memoryfilesystem.MemoryFileLock;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

final class MemoryInode {
    private static final int ARRAY_HEADER = 20;
    static final int BLOCK_SIZE = 4076;
    static final int NUMBER_OF_BLOCKS = 4076;
    private LockSet lockSet;
    private byte[] directBlock;
    private byte[][] indirectBlocks;
    private long size;
    private int indirectBlocksAllocated;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();

    MemoryInode(int initialBlocks) {
        this.directBlock = initialBlocks == 0 ? new byte[4076] : new byte[4076];
        if (initialBlocks > 1) {
            this.indirectBlocks = new byte[4076][];
            for (int i = 0; i < initialBlocks - 1; ++i) {
                this.indirectBlocks[i] = new byte[4076];
            }
            this.indirectBlocksAllocated = initialBlocks - 1;
        }
        this.size = 0L;
    }

    MemoryInode(MemoryInode other) {
        if (other.directBlock != null) {
            this.directBlock = (byte[])other.directBlock.clone();
        }
        if (other.indirectBlocks != null) {
            this.indirectBlocks = (byte[][])other.indirectBlocks.clone();
            for (int i = 0; i < other.indirectBlocksAllocated; ++i) {
                this.indirectBlocks[i] = (byte[])other.indirectBlocks[i].clone();
            }
        }
        this.indirectBlocksAllocated = other.indirectBlocksAllocated;
        this.size = other.size;
    }

    AutoRelease readLock() {
        return AutoReleaseLock.autoRelease(this.lock.readLock());
    }

    AutoRelease writeLock() {
        return AutoReleaseLock.autoRelease(this.lock.writeLock());
    }

    long size() {
        try (AutoRelease lock = this.readLock();){
            long l = this.size;
            return l;
        }
    }

    long read(ByteBuffer dst, long position, long maximum) throws IOException {
        try (AutoRelease lock = this.readLock();){
            if (position >= this.size) {
                long l = -1L;
                return l;
            }
            long remaining = dst.remaining();
            long toRead = Math.min(Math.min(this.size - position, remaining), maximum);
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            long read = 0L;
            while (read < toRead) {
                int lengthInBlock = (int)Math.min((long)(4076 - startIndexInBlock), toRead - read);
                byte[] block = this.getBlock(currentBlock);
                dst.put(block, startIndexInBlock, lengthInBlock);
                read += (long)lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            long l = read;
            return l;
        }
    }

    int readShort(ByteBuffer dst, long position) throws IOException {
        return (int)this.read(dst, position, Integer.MAX_VALUE);
    }

    int read(byte[] dst, long position, int off, int len) throws IOException {
        try (AutoRelease lock = this.readLock();){
            if (position >= this.size) {
                int n = -1;
                return n;
            }
            int toRead = (int)Math.min(Math.min(this.size - position, (long)len), Integer.MAX_VALUE);
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            int read = 0;
            while (read < toRead) {
                int lengthInBlock = Math.min(4076 - startIndexInBlock, toRead - read);
                byte[] block = this.getBlock(currentBlock);
                System.arraycopy(block, startIndexInBlock, dst, off + read, lengthInBlock);
                read += lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            int n = read;
            return n;
        }
    }

    long transferFrom(ReadableByteChannel src, long position, long count) throws IOException {
        try (AutoRelease lock = this.writeLock();){
            this.ensureCapacity(position + count);
            long transferred = 0L;
            long toTransfer = count;
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            while (transferred < toTransfer) {
                int lengthInBlock = (int)Math.min((long)(4076 - startIndexInBlock), toTransfer - transferred);
                byte[] block = this.getBlock(currentBlock);
                ByteBuffer buffer = ByteBuffer.wrap(block, startIndexInBlock, lengthInBlock);
                MemoryInode.readFully(buffer, src, lengthInBlock);
                transferred += (long)lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            this.size = Math.max(this.size, position + transferred);
            long l = transferred;
            return l;
        }
    }

    long transferTo(WritableByteChannel target, long position, long count) throws IOException {
        try (AutoRelease lock = this.readLock();){
            long transferred = 0L;
            long toTransfer = Math.min(count, this.size - position);
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            while (transferred < toTransfer) {
                int lengthInBlock = (int)Math.min((long)(4076 - startIndexInBlock), toTransfer - transferred);
                byte[] block = this.getBlock(currentBlock);
                ByteBuffer buffer = ByteBuffer.wrap(block, startIndexInBlock, lengthInBlock);
                MemoryInode.writeFully(buffer, target, lengthInBlock);
                transferred += (long)lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            long l = transferred;
            return l;
        }
    }

    long transferTo(OutputStream target, long position) throws IOException {
        try (AutoRelease lock = this.readLock();){
            long transferred = 0L;
            long toTransfer = this.size - position;
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            while (transferred < toTransfer) {
                int lengthInBlock = (int)Math.min((long)(4076 - startIndexInBlock), toTransfer - transferred);
                byte[] block = this.getBlock(currentBlock);
                target.write(block, startIndexInBlock, lengthInBlock);
                transferred += (long)lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            long l = transferred;
            return l;
        }
    }

    long write(ByteBuffer src, long position, long maximum) {
        try (AutoRelease lock = this.writeLock();){
            long remaining = src.remaining();
            this.ensureCapacity(position + remaining);
            long toWrite = Math.min(remaining, maximum);
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            long written = 0L;
            while (written < toWrite) {
                int lengthInBlock = (int)Math.min(4076L - (long)startIndexInBlock, toWrite - written);
                byte[] block = this.getBlock(currentBlock);
                src.get(block, startIndexInBlock, lengthInBlock);
                written += (long)lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            this.size = Math.max(this.size, position + written);
            long l = written;
            return l;
        }
    }

    int writeShort(ByteBuffer src, long position) {
        return (int)this.write(src, position, Integer.MAX_VALUE);
    }

    int write(byte[] src, long position, int off, int len) {
        try (AutoRelease lock = this.writeLock();){
            this.ensureCapacity(position + (long)len);
            int toWrite = Math.min(len, Integer.MAX_VALUE);
            int currentBlock = (int)(position / 4076L);
            int startIndexInBlock = (int)(position - (long)currentBlock * 4076L);
            int written = 0;
            while (written < toWrite) {
                int lengthInBlock = Math.min(4076 - startIndexInBlock, toWrite - written);
                byte[] block = this.getBlock(currentBlock);
                System.arraycopy(src, off + written, block, startIndexInBlock, lengthInBlock);
                written += lengthInBlock;
                startIndexInBlock = 0;
                ++currentBlock;
            }
            this.size = Math.max(this.size, position + (long)written);
            int n = written;
            return n;
        }
    }

    long writeAtEnd(ByteBuffer src, long maximum) {
        try (AutoRelease lock = this.writeLock();){
            long l = this.write(src, this.size, maximum);
            return l;
        }
    }

    int writeAtEnd(ByteBuffer src) {
        try (AutoRelease lock = this.writeLock();){
            int n = this.writeShort(src, this.size);
            return n;
        }
    }

    int writeAtEnd(byte[] src, int off, int len) {
        try (AutoRelease lock = this.writeLock();){
            int n = this.write(src, this.size, off, len);
            return n;
        }
    }

    void truncate(long newSize) {
        try (AutoRelease lock = this.writeLock();){
            if (newSize < this.size) {
                this.size = newSize;
            }
        }
    }

    MemoryFileLock tryLock(MemoryFileLock lock) {
        try (AutoRelease autoRelease = this.writeLock();){
            MemoryFileLock memoryFileLock = this.lockSet().tryLock(lock);
            return memoryFileLock;
        }
    }

    MemoryFileLock lock(MemoryFileLock lock) throws IOException {
        try (AutoRelease autoRelease = this.writeLock();){
            MemoryFileLock memoryFileLock = this.lockSet().lock(lock);
            return memoryFileLock;
        }
    }

    void unlock(MemoryFileLock lock) {
        try (AutoRelease autoRelease = this.writeLock();){
            this.lockSet.remove(lock);
        }
    }

    private LockSet lockSet() {
        if (this.lockSet == null) {
            this.lockSet = new LockSet();
        }
        return this.lockSet;
    }

    private byte[] getBlock(int currentBlock) {
        if (currentBlock == 0) {
            return this.directBlock;
        }
        return this.indirectBlocks[currentBlock - 1];
    }

    private static int writeFully(ByteBuffer src, WritableByteChannel target, int toWrite) throws IOException {
        int written;
        for (written = 0; written < toWrite; written += target.write(src)) {
        }
        return written;
    }

    private static int readFully(ByteBuffer src, ReadableByteChannel target, int toRead) throws IOException {
        int read;
        for (read = 0; read < toRead; read += target.read(src)) {
        }
        return read;
    }

    private void ensureCapacity(long capacity) {
        int blocksRequired;
        if (capacity <= 4076L) {
            return;
        }
        if (this.indirectBlocks == null) {
            this.indirectBlocks = new byte[4076][];
        }
        if ((blocksRequired = (int)((capacity - 1L) / 4076L)) > 4076) {
            throw new AssertionError((Object)"files bigger than 16MB not yet supported");
        }
        if (blocksRequired > this.indirectBlocksAllocated) {
            for (int i = this.indirectBlocksAllocated; i < blocksRequired; ++i) {
                this.indirectBlocks[i] = new byte[4076];
                ++this.indirectBlocksAllocated;
            }
        }
    }
}

