/*
 * Copyright (c) 2016-2017, Joyent, Inc. All rights reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */
package com.joyent.manta.http;

import com.codahale.metrics.Meter;
import com.joyent.manta.config.ConfigContext;
import com.joyent.manta.config.MantaClientMetricConfiguration;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.ConnectException;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;

import static org.apache.commons.lang3.Validate.notNull;

/**
 * Implementation of {@link HttpRequestRetryHandler} customized for use with
 * Manta.
 *
 * @author <a href="https://github.com/dekobon">Elijah Zupancic</a>
 * @since 3.0.0
 */
public class MantaHttpRequestRetryHandler
        extends DefaultHttpRequestRetryHandler
        implements HttpContextRetryCancellation {

    /**
     * Logger instance.
     */
    private final Logger logger = LoggerFactory.getLogger(MantaHttpRequestRetryHandler.class);

    /**
     * List of all exception types that can't be retried.
     */
    protected static final List<Class<? extends IOException>> NON_RETRIABLE = Arrays.asList(
            InterruptedIOException.class,
            UnknownHostException.class,
            ConnectException.class,
            SSLException.class);

    /**
     * Key for HttpContext setting indicating the request should NOT be retried under any circumstances.
     * @deprecated 3.2.3
     * @see HttpContextRetryCancellation#CONTEXT_ATTRIBUTE_MANTA_RETRY_DISABLE
     */
    @Deprecated
    public static final String CONTEXT_ATTRIBUTE_MANTA_RETRY_DISABLE =
            HttpContextRetryCancellation.CONTEXT_ATTRIBUTE_MANTA_RETRY_DISABLE;

    /**
     * The name used to publish the retry metrics.
     */
    public static final String METRIC_NAME_RETRIES = "retries";

    /**
     * Nullable meter for keeping track of the count and rate of retries.
     */
    private final Meter retries;

    /**
     * Deprecated constructor.
     *
     * @param config configuration indicating retry count
     */
    @Deprecated
    public MantaHttpRequestRetryHandler(final ConfigContext config) {
        this(notNull(config).getRetries(), null);
    }

    /**
     * Creates a new instance with the passed configuration.
     *  @param retryCount     how many times to retry; 0 means no retries
     *
     */
    public MantaHttpRequestRetryHandler(final int retryCount) {
        this(retryCount, null);
    }

    /**
     * Creates a new instance with the passed configuration.
     *
     * @param retryCount   how many times to retry; 0 means no retries
     * @param metricConfig potentially-null configuration for tracking client metrics
     */
    public MantaHttpRequestRetryHandler(final int retryCount, final MantaClientMetricConfiguration metricConfig) {
        super(retryCount, true, NON_RETRIABLE);

        if (metricConfig != null && metricConfig.getRegistry() != null) {
            this.retries = metricConfig.getRegistry().meter(METRIC_NAME_RETRIES);
        } else {
            this.retries = null;
        }
    }

    @Override
    public boolean retryRequest(final IOException exception,
                                final int executionCount,
                                final HttpContext context) {
        notNull(context, "HTTP context cannot be null");

        if (neverRetry(context)) {
            return false;
        }

        final boolean toBeRetried = super.retryRequest(exception, executionCount, context);

        if (logger.isDebugEnabled()) {
            if (toBeRetried) {
                String msg = String.format("Request failed, %d/%d retry.",
                        executionCount, getRetryCount());
                logger.debug(msg, exception);
            } else {
                logger.debug("Request failed, unable to retry.", exception);
            }
        }

        if (toBeRetried && retries != null) {
            retries.mark();
        }

        return toBeRetried;
    }
}
