/*-
 *
 *  This file is part of Oracle Berkeley DB Java Edition
 *  Copyright (C) 2002, 2015 Oracle and/or its affiliates.  All rights reserved.
 *
 *  Oracle Berkeley DB Java Edition is free software: you can redistribute it
 *  and/or modify it under the terms of the GNU Affero General Public License
 *  as published by the Free Software Foundation, version 3.
 *
 *  Oracle Berkeley DB Java Edition is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License in
 *  the LICENSE file along with Oracle Berkeley DB Java Edition.  If not, see
 *  <http://www.gnu.org/licenses/>.
 *
 *  An active Oracle commercial licensing agreement for this product
 *  supercedes this license.
 *
 *  For more information please contact:
 *
 *  Vice President Legal, Development
 *  Oracle America, Inc.
 *  5OP-10
 *  500 Oracle Parkway
 *  Redwood Shores, CA 94065
 *
 *  or
 *
 *  berkeleydb-info_us@oracle.com
 *
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  [This line intentionally left blank.]
 *  EOF
 *
 */

package com.sleepycat.je.log.entry;

import java.nio.ByteBuffer;

import com.sleepycat.je.dbi.DatabaseId;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.LogEntryHeader;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.tree.IN;
import com.sleepycat.je.tree.BIN;
import com.sleepycat.je.utilint.DbLsn;

import static com.sleepycat.je.EnvironmentFailureException.unexpectedState;

/**
 * - INLogEntry is used to read/write full-version IN logrecs.
 *
 * - BINDeltaLogEntry subclasses INLogEntry and is used to read/write
 *   BIN-delta logrecs for log versions 9 or later.
 *
 * - OldBINDeltaLogEntry is used to read/write BIN-delta logrecs for
 *   log versions earlier than 9. OldBINDeltaLogEntry is not a subclass
 *   of INLogEntry.
 *
 * On disk, a full IN logrec contains:
 * 
 * <pre>
 * (3 <= version < 6)
 *        IN
 *        database id
 *        prevFullLsn  -- in version 2
 *
 * (6 <= version < 8)
 *        database id
 *        prevFullLsn
 *        IN
 *
 * (8 <= version)
 *        database id
 *        prevFullLsn
 *        prevDeltaLsn
 *        IN
 * </pre>
 *
 *  On disk, a BIN-delta logrec written via the BINDeltaLogEntry contains:
 *
 * <pre>
 * (version == 9)
 *        database id
 *        prevFullLsn  -- always NULL
 *        prevDeltaLsn
 *        BIN (dirty slots only)
 *        prevFullLsn
 *
 * (version >= 10)
 *        database id
 *        prevFullLsn
 *        prevDeltaLsn
 *        BIN (dirty slots only and including the new fullBinNEntries and
 *             fullBinMaxEntries fields) 
 * </pre>
 *
 */
public class INLogEntry<T extends IN> extends BaseEntry<T>
    implements LogEntry, INContainingEntry {

    /*
     * Persistent fields in an IN entry.
     */

    private DatabaseId dbId;

    /*
     * this.in may be a (a) UIN, (b) full BIN, or (c) BIN delta.
     * In case (a), "this" is a INLogEntry
     * In case (c), "this" is a BINDeltaLogEntry instance.
     * In case (b), "this" may be either a INLogEntry or a BINDeltaLogEntry
     * instance. It will be a BINDeltaLogEntry instance, if "this" is used
     * to log a full in-memory BIN as a BIN-delta.
     */
    private T in;

    /*
     * The lsn of the previous full-version logrec for the same IN.
     *
     * See comment above about the evolution of this field.
     */
    private long prevFullLsn;

    /*
     * If this is a BIN logrec and the previous logrec for the same BIN was
     * a BIN-delta, prevDeltaLsn is the lsn of that previous logrec. Otherwise,
     * prevDeltaLsn is NULL.
     *
     * See comment above about the evolution of this field.
     */
    private long prevDeltaLsn;

    /**
     * Construct a log entry for reading.
     */
    public static <T extends IN> INLogEntry<T> create(final Class<T> INClass) {
        return new INLogEntry<T>(INClass);
    }

    INLogEntry(Class<T> INClass) {
        super(INClass);
    }

    /**
     * Construct an INLogEntry for writing to the log.
     */
    public INLogEntry(T in) {
        this(in, false /*isBINDelta*/);
    }

    /*
     * Used by both INLogEntry and BINDeltaLogEntry for writing to the log.
     */
    INLogEntry(T in, boolean isBINDelta) {

        setLogType(isBINDelta ? LogEntryType.LOG_BIN_DELTA : in.getLogType());

        dbId = in.getDatabase().getId();

        this.in = in;

        prevFullLsn = in.getLastFullLsn();
        prevDeltaLsn = in.getLastDeltaLsn();
    }

    /*
     * Whether this LogEntry reads/writes a BIN-Delta logrec.
     * Overriden by the BINDeltaLogEntry subclass.
     */
    @Override
    public boolean isBINDelta() {
        return false;
    }

    @Override
    public DatabaseId getDbId() {
        return dbId;
    }

    @Override
    public long getPrevFullLsn() {
        return prevFullLsn;
    }

    @Override
    public long getPrevDeltaLsn() {
        return prevDeltaLsn;
    }

    @Override
    public T getMainItem() {
        return in;
    }

    @Override
    public IN getIN(DatabaseImpl dbImpl) {
        return in;
    }

    public long getNodeId() {
        return in.getNodeId();
    }

    /*
     * Read support
     */

    @Override
    public void readEntry(
        EnvironmentImpl envImpl,
        LogEntryHeader header,
        ByteBuffer entryBuffer) {

        int logVersion = header.getVersion();
        boolean version6OrLater = (logVersion >= 6);

        if (logVersion < 2) {
            throw unexpectedState(
                "Attempt to read from log file with version " +
                logVersion + ", which is not supported any more");
        }

        if (version6OrLater) {
            dbId = new DatabaseId();
            dbId.readFromLog(entryBuffer, logVersion);

            prevFullLsn = LogUtils.readLong(entryBuffer, false/*unpacked*/);
            if (logVersion >= 8) {
                prevDeltaLsn = LogUtils.readPackedLong(entryBuffer);
            } else {
                prevDeltaLsn = DbLsn.NULL_LSN;
            }
        }

        /* Read IN. */
        in = newInstanceOfType();
        readMainItem(in, entryBuffer, logVersion);

        if (!version6OrLater) {
            dbId = new DatabaseId();
            dbId.readFromLog(entryBuffer, logVersion);

            prevFullLsn = LogUtils.readLong(entryBuffer, true/*unpacked*/);
            prevDeltaLsn = DbLsn.NULL_LSN;
        }
    }

    private void readMainItem(T in, ByteBuffer entryBuffer, int logVersion) {

        if (isBINDelta()) {
            assert(logVersion >= 9);

            in.readFromLog(entryBuffer, logVersion, true/*deltasOnly*/);

            if (logVersion == 9) {
                in.setLastFullLsn(LogUtils.readPackedLong(entryBuffer));
            } else {
                in.setLastFullLsn(prevFullLsn);
            }
        } else {
            in.readFromLog(entryBuffer, logVersion);
        }
    }

    /*
     * Writing support
     */
    @Override
    public int getSize() {
        int inSize;

        if (isBINDelta()) {
            inSize = in.getLogSize(true/*deltasOnly*/);
        } else {
            inSize = in.getLogSize();
        }

        return (inSize +
                dbId.getLogSize() +
                LogUtils.getPackedLongLogSize(prevFullLsn) +
                LogUtils.getPackedLongLogSize(prevDeltaLsn));
    }

    @Override
    public void writeEntry(ByteBuffer destBuffer) {

        dbId.writeToLog(destBuffer);

        LogUtils.writePackedLong(destBuffer, prevFullLsn);
        LogUtils.writePackedLong(destBuffer, prevDeltaLsn);

        if (isBINDelta()) {
            in.writeToLog(destBuffer, true/*deltasOnly*/);
        } else {
            in.writeToLog(destBuffer);
        }
    }

    @Override
    public long getTransactionId() {
        return 0;
    }

    /**
     * INs from two different environments are never considered equal,
     * because they have lsns that are environment-specific.
     */
    @Override
    public boolean logicalEquals(@SuppressWarnings("unused") LogEntry other) {
        return false;
    }

    @Override
    public StringBuilder dumpEntry(StringBuilder sb, boolean verbose) {

        dbId.dumpLog(sb, verbose);

        in.dumpLog(sb, verbose);

        if (prevFullLsn != DbLsn.NULL_LSN) {
            sb.append("<prevFullLsn>");
            sb.append(DbLsn.getNoFormatString(prevFullLsn));
            sb.append("</prevFullLsn>");
        }
        if (prevDeltaLsn != DbLsn.NULL_LSN) {
            sb.append("<prevDeltaLsn>");
            sb.append(DbLsn.getNoFormatString(prevDeltaLsn));
            sb.append("</prevDeltaLsn>");
        }
        return sb;
    }

    /** Never replicated. */
    public void dumpRep(@SuppressWarnings("unused") StringBuilder sb) {
    }

}
