/*
 * Decompiled with CFR 0.152.
 */
package com.stormpath.sdk.servlet.filter.cors;

import com.stormpath.sdk.lang.Assert;
import com.stormpath.sdk.servlet.filter.HttpFilter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Locale;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CorsFilter
extends HttpFilter {
    private static final Logger log = LoggerFactory.getLogger(CorsFilter.class);
    public static final String RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_ORIGIN = "Access-Control-Allow-Origin";
    public static final String RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_CREDENTIALS = "Access-Control-Allow-Credentials";
    public static final String RESPONSE_HEADER_ACCESS_CONTROL_EXPOSE_HEADERS = "Access-Control-Expose-Headers";
    public static final String RESPONSE_HEADER_ACCESS_CONTROL_MAX_AGE = "Access-Control-Max-Age";
    public static final String RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_METHODS = "Access-Control-Allow-Methods";
    public static final String RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_HEADERS = "Access-Control-Allow-Headers";
    public static final String REQUEST_HEADER_ORIGIN = "Origin";
    public static final String REQUEST_HEADER_ACCESS_CONTROL_REQUEST_METHOD = "Access-Control-Request-Method";
    public static final String REQUEST_HEADER_ACCESS_CONTROL_REQUEST_HEADERS = "Access-Control-Request-Headers";
    public static final Collection<String> SIMPLE_HTTP_REQUEST_CONTENT_TYPE_VALUES = new HashSet<String>(Arrays.asList("application/x-www-form-urlencoded", "multipart/form-data", "text/plain"));
    private boolean anyOriginAllowed = false;
    private boolean supportsCredentials = true;
    private long preflightMaxAge = 1800L;
    private Collection<String> allowedOrigins = new HashSet<String>();
    private Collection<String> allowedHttpMethods = new HashSet<String>();
    private Collection<String> allowedHttpHeaders = new HashSet<String>();
    private Collection<String> exposedHeaders = new HashSet<String>();

    public void setAllowedOrigins(Collection<String> allowedOrigins) {
        this.allowedOrigins = allowedOrigins;
    }

    public void setAllowedHttpMethods(Collection<String> allowedHttpMethods) {
        this.allowedHttpMethods = allowedHttpMethods;
    }

    public void setAllowedHttpHeaders(Collection<String> allowedHttpHeaders) {
        this.allowedHttpHeaders = allowedHttpHeaders;
    }

    public void setAnyOriginAllowed(boolean anyOriginAllowed) {
        this.anyOriginAllowed = anyOriginAllowed;
    }

    public void setSupportsCredentials(boolean supportsCredentials) {
        this.supportsCredentials = supportsCredentials;
    }

    public void setPreflightMaxAge(long preflightMaxAge) {
        this.preflightMaxAge = preflightMaxAge;
    }

    public void setExposedHeaders(Collection<String> exposedHeaders) {
        this.exposedHeaders = exposedHeaders;
    }

    @Override
    protected void onInit() throws Exception {
        super.onInit();
        Assert.notNull(this.allowedOrigins, (String)"allowedOrigins cannot be null.");
        Assert.notEmpty(this.allowedHttpMethods, (String)"allowedHttpMethods cannot be empty.");
        Assert.notEmpty(this.allowedHttpHeaders, (String)"allowedHttpHeaders cannot be empty.");
    }

    @Override
    protected boolean isEnabled(HttpServletRequest request, HttpServletResponse response) throws Exception {
        return request.getHeader(REQUEST_HEADER_ORIGIN) != null && this.isEnabled();
    }

    @Override
    protected void filter(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws Exception {
        CORSRequestType requestType = this.checkRequestType(request);
        switch (requestType) {
            case SIMPLE: {
                this.handleSimpleCORS(request, response, filterChain);
                break;
            }
            case ACTUAL: {
                this.handleSimpleCORS(request, response, filterChain);
                break;
            }
            case PRE_FLIGHT: {
                this.handlePreflightCORS(request, response, filterChain);
                break;
            }
            case NOT_CORS: {
                this.handleNonCORS(request, response, filterChain);
                break;
            }
            default: {
                this.handleInvalidCORS(request, response, filterChain);
            }
        }
    }

    protected void handleSimpleCORS(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws IOException, ServletException {
        String origin = request.getHeader(REQUEST_HEADER_ORIGIN);
        String method = request.getMethod();
        if (!this.isOriginAllowed(origin)) {
            this.handleInvalidCORS(request, response, filterChain);
            return;
        }
        if (!this.allowedHttpMethods.contains(method)) {
            this.handleInvalidCORS(request, response, filterChain);
            return;
        }
        if (this.anyOriginAllowed && !this.supportsCredentials) {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_ORIGIN, "*");
        } else {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_ORIGIN, origin);
        }
        if (this.supportsCredentials) {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_CREDENTIALS, "true");
        }
        if (this.exposedHeaders != null && this.exposedHeaders.size() > 0) {
            String exposedHeadersString = CorsFilter.join(this.exposedHeaders, ",");
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_EXPOSE_HEADERS, exposedHeadersString);
        }
        filterChain.doFilter((ServletRequest)request, (ServletResponse)response);
    }

    protected void handlePreflightCORS(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws IOException, ServletException {
        String origin = request.getHeader(REQUEST_HEADER_ORIGIN);
        if (!this.isOriginAllowed(origin)) {
            this.handleInvalidCORS(request, response, filterChain);
            return;
        }
        String accessControlRequestMethod = request.getHeader(REQUEST_HEADER_ACCESS_CONTROL_REQUEST_METHOD);
        if (accessControlRequestMethod == null) {
            this.handleInvalidCORS(request, response, filterChain);
            return;
        }
        accessControlRequestMethod = accessControlRequestMethod.trim();
        String accessControlRequestHeadersHeader = request.getHeader(REQUEST_HEADER_ACCESS_CONTROL_REQUEST_HEADERS);
        LinkedList<String> accessControlRequestHeaders = new LinkedList<String>();
        if (accessControlRequestHeadersHeader != null && !accessControlRequestHeadersHeader.trim().isEmpty()) {
            String[] headers;
            for (String header : headers = accessControlRequestHeadersHeader.trim().split(",")) {
                accessControlRequestHeaders.add(header.trim().toLowerCase(Locale.ENGLISH));
            }
        }
        if (!this.allowedHttpMethods.contains(accessControlRequestMethod)) {
            this.handleInvalidCORS(request, response, filterChain);
            return;
        }
        if (!accessControlRequestHeaders.isEmpty()) {
            for (String header : accessControlRequestHeaders) {
                if (this.allowedHttpHeaders.contains(header)) continue;
                this.handleInvalidCORS(request, response, filterChain);
                return;
            }
        }
        if (this.supportsCredentials) {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_ORIGIN, origin);
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_CREDENTIALS, "true");
        } else if (this.anyOriginAllowed) {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_ORIGIN, "*");
        } else {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_ORIGIN, origin);
        }
        if (this.preflightMaxAge > 0L) {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_MAX_AGE, String.valueOf(this.preflightMaxAge));
        }
        response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_METHODS, accessControlRequestMethod);
        if (this.allowedHttpHeaders != null && !this.allowedHttpHeaders.isEmpty()) {
            response.addHeader(RESPONSE_HEADER_ACCESS_CONTROL_ALLOW_HEADERS, CorsFilter.join(this.allowedHttpHeaders, ","));
        }
    }

    private void handleNonCORS(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws IOException, ServletException {
        filterChain.doFilter((ServletRequest)request, (ServletResponse)response);
    }

    private void handleInvalidCORS(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) {
        String origin = request.getHeader(REQUEST_HEADER_ORIGIN);
        String method = request.getMethod();
        String accessControlRequestHeaders = request.getHeader(REQUEST_HEADER_ACCESS_CONTROL_REQUEST_HEADERS);
        response.setContentType("text/plain");
        response.setStatus(403);
        response.resetBuffer();
        if (log.isDebugEnabled()) {
            StringBuilder message = new StringBuilder("Invalid CORS request; Origin=");
            message.append(origin);
            message.append(";Method=");
            message.append(method);
            if (accessControlRequestHeaders != null) {
                message.append(";Access-Control-Request-Headers=");
                message.append(accessControlRequestHeaders);
            }
            log.debug(message.toString());
        }
    }

    protected CORSRequestType checkRequestType(HttpServletRequest request) {
        CORSRequestType requestType = CORSRequestType.INVALID_CORS;
        String originHeader = request.getHeader(REQUEST_HEADER_ORIGIN);
        if (originHeader != null) {
            if (originHeader.isEmpty()) {
                requestType = CORSRequestType.INVALID_CORS;
            } else if (!CorsFilter.isValidOrigin(originHeader)) {
                requestType = CORSRequestType.INVALID_CORS;
            } else {
                if (this.isLocalOrigin(request, originHeader)) {
                    return CORSRequestType.NOT_CORS;
                }
                String method = request.getMethod();
                if (method != null) {
                    if ("OPTIONS".equals(method)) {
                        String accessControlRequestMethodHeader = request.getHeader(REQUEST_HEADER_ACCESS_CONTROL_REQUEST_METHOD);
                        requestType = accessControlRequestMethodHeader != null && !accessControlRequestMethodHeader.isEmpty() ? CORSRequestType.PRE_FLIGHT : (accessControlRequestMethodHeader != null && accessControlRequestMethodHeader.isEmpty() ? CORSRequestType.INVALID_CORS : CORSRequestType.ACTUAL);
                    } else if ("GET".equals(method) || "HEAD".equals(method)) {
                        requestType = CORSRequestType.SIMPLE;
                    } else if ("POST".equals(method)) {
                        String mediaType = this.getMediaType(request.getContentType());
                        if (mediaType != null) {
                            requestType = SIMPLE_HTTP_REQUEST_CONTENT_TYPE_VALUES.contains(mediaType) ? CORSRequestType.SIMPLE : CORSRequestType.ACTUAL;
                        }
                    } else {
                        requestType = CORSRequestType.ACTUAL;
                    }
                }
            }
        } else {
            requestType = CORSRequestType.NOT_CORS;
        }
        return requestType;
    }

    protected static boolean isValidOrigin(String origin) {
        URI originURI;
        if (origin.contains("%")) {
            return false;
        }
        if ("null".equals(origin)) {
            return true;
        }
        try {
            originURI = new URI(origin);
        }
        catch (URISyntaxException e) {
            return false;
        }
        return originURI.getScheme() != null;
    }

    private boolean isLocalOrigin(HttpServletRequest request, String origin) {
        StringBuilder target = new StringBuilder();
        String scheme = request.getScheme();
        if (scheme == null) {
            return false;
        }
        scheme = scheme.toLowerCase(Locale.ENGLISH);
        target.append(scheme);
        target.append("://");
        String host = request.getServerName();
        if (host == null) {
            return false;
        }
        target.append(host);
        int port = request.getServerPort();
        if ("http".equals(scheme) && port != 80 || "https".equals(scheme) && port != 443) {
            target.append(':');
            target.append(port);
        }
        return origin.equalsIgnoreCase(target.toString());
    }

    private String getMediaType(String contentType) {
        if (contentType == null) {
            return null;
        }
        String result = contentType.toLowerCase(Locale.ENGLISH);
        int firstSemiColonIndex = result.indexOf(59);
        if (firstSemiColonIndex > -1) {
            result = result.substring(0, firstSemiColonIndex);
        }
        result = result.trim();
        return result;
    }

    private boolean isOriginAllowed(String origin) {
        if (this.anyOriginAllowed) {
            return true;
        }
        return this.allowedOrigins.contains(origin);
    }

    protected static String join(Collection<String> elements, String joinSeparator) {
        String separator = ",";
        if (elements == null) {
            return null;
        }
        if (joinSeparator != null) {
            separator = joinSeparator;
        }
        StringBuilder buffer = new StringBuilder();
        boolean isFirst = true;
        for (String element : elements) {
            if (!isFirst) {
                buffer.append(separator);
            } else {
                isFirst = false;
            }
            if (element == null) continue;
            buffer.append(element);
        }
        return buffer.toString();
    }

    protected static enum CORSRequestType {
        SIMPLE,
        ACTUAL,
        PRE_FLIGHT,
        NOT_CORS,
        INVALID_CORS;

    }
}

