/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.impl.prefetch;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.LocalDirAllocator;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.impl.prefetch.BlockCache;
import org.apache.hadoop.fs.impl.prefetch.BufferData;
import org.apache.hadoop.fs.impl.prefetch.PrefetchConstants;
import org.apache.hadoop.fs.impl.prefetch.PrefetchingStatistics;
import org.apache.hadoop.fs.impl.prefetch.Validate;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableSet;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SingleFilePerBlockCache
implements BlockCache {
    private static final Logger LOG = LoggerFactory.getLogger(SingleFilePerBlockCache.class);
    private final Map<Integer, Entry> blocks;
    private final int maxBlocksCount;
    private final ReentrantReadWriteLock blocksLock;
    private Entry head;
    private Entry tail;
    private int entryListSize;
    private int numGets = 0;
    private final AtomicBoolean closed;
    private final PrefetchingStatistics prefetchingStatistics;
    private static final Set<PosixFilePermission> TEMP_FILE_ATTRS = ImmutableSet.of((Object)((Object)PosixFilePermission.OWNER_READ), (Object)((Object)PosixFilePermission.OWNER_WRITE));
    private static final Set<? extends OpenOption> CREATE_OPTIONS = EnumSet.of(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    private static final String CACHE_FILE_PREFIX = "fs-cache-";
    private static final String BINARY_FILE_SUFFIX = ".bin";

    public SingleFilePerBlockCache(PrefetchingStatistics prefetchingStatistics, int maxBlocksCount) {
        this.prefetchingStatistics = Objects.requireNonNull(prefetchingStatistics);
        this.closed = new AtomicBoolean(false);
        this.maxBlocksCount = maxBlocksCount;
        Preconditions.checkArgument(maxBlocksCount > 0, "maxBlocksCount should be more than 0");
        this.blocks = new ConcurrentHashMap<Integer, Entry>();
        this.blocksLock = new ReentrantReadWriteLock();
    }

    @Override
    public boolean containsBlock(int blockNumber) {
        return this.blocks.containsKey(blockNumber);
    }

    @Override
    public Iterable<Integer> blocks() {
        return Collections.unmodifiableList(new ArrayList<Integer>(this.blocks.keySet()));
    }

    @Override
    public int size() {
        return this.blocks.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void get(int blockNumber, ByteBuffer buffer) throws IOException {
        if (this.closed.get()) {
            return;
        }
        Validate.checkNotNull(buffer, "buffer");
        Entry entry = this.getEntry(blockNumber);
        entry.takeLock(Entry.LockType.READ);
        try {
            buffer.clear();
            this.readFile(entry.path, buffer);
            buffer.rewind();
            this.validateEntry(entry, buffer);
        }
        finally {
            entry.releaseLock(Entry.LockType.READ);
        }
    }

    protected int readFile(java.nio.file.Path path, ByteBuffer buffer) throws IOException {
        int numBytesRead = 0;
        try (FileChannel channel = FileChannel.open(path, StandardOpenOption.READ);){
            int numBytes;
            while ((numBytes = channel.read(buffer)) > 0) {
                numBytesRead += numBytes;
            }
            buffer.limit(buffer.position());
        }
        return numBytesRead;
    }

    private Entry getEntry(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        Entry entry = this.blocks.get(blockNumber);
        if (entry == null) {
            throw new IllegalStateException(String.format("block %d not found in cache", blockNumber));
        }
        ++this.numGets;
        this.addToLinkedListHead(entry);
        return entry;
    }

    private void addToLinkedListHead(Entry entry) {
        this.blocksLock.writeLock().lock();
        try {
            this.addToHeadOfLinkedList(entry);
        }
        finally {
            this.blocksLock.writeLock().unlock();
        }
    }

    private void addToHeadOfLinkedList(Entry entry) {
        if (this.head == null) {
            this.head = entry;
            this.tail = entry;
        }
        LOG.debug("Block num {} to be added to the head. Current head block num: {} and tail block num: {}", new Object[]{entry.blockNumber, this.head.blockNumber, this.tail.blockNumber});
        if (entry != this.head) {
            Entry prev = entry.getPrevious();
            Entry nxt = entry.getNext();
            if (!this.blocks.containsKey(entry.blockNumber)) {
                return;
            }
            if (prev != null) {
                prev.setNext(nxt);
            }
            if (nxt != null) {
                nxt.setPrevious(prev);
            }
            entry.setPrevious(null);
            entry.setNext(this.head);
            this.head.setPrevious(entry);
            this.head = entry;
            if (prev != null && prev.getNext() == null) {
                this.tail = prev;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(int blockNumber, ByteBuffer buffer, Configuration conf, LocalDirAllocator localDirAllocator) throws IOException {
        if (this.closed.get()) {
            return;
        }
        Validate.checkNotNull(buffer, "buffer");
        if (this.blocks.containsKey(blockNumber)) {
            Entry entry = this.blocks.get(blockNumber);
            entry.takeLock(Entry.LockType.READ);
            try {
                this.validateEntry(entry, buffer);
            }
            finally {
                entry.releaseLock(Entry.LockType.READ);
            }
            this.addToLinkedListHead(entry);
            return;
        }
        Validate.checkPositiveInteger(buffer.limit(), "buffer.limit()");
        java.nio.file.Path blockFilePath = this.getCacheFilePath(conf, localDirAllocator);
        long size = Files.size(blockFilePath);
        if (size != 0L) {
            String message = String.format("[%d] temp file already has data. %s (%d)", blockNumber, blockFilePath, size);
            throw new IllegalStateException(message);
        }
        this.writeFile(blockFilePath, buffer);
        long checksum = BufferData.getChecksum(buffer);
        Entry entry = new Entry(blockNumber, blockFilePath, buffer.limit(), checksum);
        this.blocks.put(blockNumber, entry);
        this.prefetchingStatistics.blockAddedToFileCache();
        this.addToLinkedListAndEvictIfRequired(entry);
    }

    private void addToLinkedListAndEvictIfRequired(Entry entry) {
        this.blocksLock.writeLock().lock();
        try {
            this.addToHeadOfLinkedList(entry);
            ++this.entryListSize;
            if (this.entryListSize > this.maxBlocksCount && !this.closed.get()) {
                Entry elementToPurge = this.tail;
                this.tail = this.tail.getPrevious();
                if (this.tail == null) {
                    this.tail = this.head;
                }
                this.tail.setNext(null);
                elementToPurge.setPrevious(null);
                this.deleteBlockFileAndEvictCache(elementToPurge);
            }
        }
        finally {
            this.blocksLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deleteBlockFileAndEvictCache(Entry elementToPurge) {
        boolean lockAcquired = elementToPurge.takeLock(Entry.LockType.WRITE, 5L, PrefetchConstants.PREFETCH_WRITE_LOCK_TIMEOUT_UNIT);
        if (!lockAcquired) {
            LOG.error("Cache file {} deletion would not be attempted as write lock could not be acquired within {} {}", new Object[]{elementToPurge.path, 5, PrefetchConstants.PREFETCH_WRITE_LOCK_TIMEOUT_UNIT});
        } else {
            try {
                if (Files.deleteIfExists(elementToPurge.path)) {
                    --this.entryListSize;
                    this.prefetchingStatistics.blockRemovedFromFileCache();
                    this.blocks.remove(elementToPurge.blockNumber);
                }
            }
            catch (IOException e) {
                LOG.warn("Failed to delete cache file {}", (Object)elementToPurge.path, (Object)e);
            }
            finally {
                elementToPurge.releaseLock(Entry.LockType.WRITE);
            }
        }
    }

    protected void writeFile(java.nio.file.Path path, ByteBuffer buffer) throws IOException {
        buffer.rewind();
        try (SeekableByteChannel writeChannel = Files.newByteChannel(path, CREATE_OPTIONS, new FileAttribute[0]);){
            while (buffer.hasRemaining()) {
                writeChannel.write(buffer);
            }
        }
    }

    protected java.nio.file.Path getCacheFilePath(Configuration conf, LocalDirAllocator localDirAllocator) throws IOException {
        return SingleFilePerBlockCache.getTempFilePath(conf, localDirAllocator);
    }

    @Override
    public void close() throws IOException {
        if (this.closed.compareAndSet(false, true)) {
            LOG.debug(this.getStats());
            this.deleteCacheFiles();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deleteCacheFiles() {
        int numFilesDeleted = 0;
        for (Entry entry : this.blocks.values()) {
            boolean lockAcquired = entry.takeLock(Entry.LockType.WRITE, 5L, PrefetchConstants.PREFETCH_WRITE_LOCK_TIMEOUT_UNIT);
            if (!lockAcquired) {
                LOG.error("Cache file {} deletion would not be attempted as write lock could not be acquired within {} {}", new Object[]{entry.path, 5, PrefetchConstants.PREFETCH_WRITE_LOCK_TIMEOUT_UNIT});
                continue;
            }
            try {
                if (!Files.deleteIfExists(entry.path)) continue;
                this.prefetchingStatistics.blockRemovedFromFileCache();
                ++numFilesDeleted;
            }
            catch (IOException e) {
                LOG.warn("Failed to delete cache file {}", (Object)entry.path, (Object)e);
            }
            finally {
                entry.releaseLock(Entry.LockType.WRITE);
            }
        }
        LOG.debug("Prefetch cache close: Deleted {} cache files", (Object)numFilesDeleted);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("stats: ");
        sb.append(this.getStats());
        sb.append(", blocks:[");
        sb.append(this.getIntList(this.blocks()));
        sb.append("]");
        return sb.toString();
    }

    private void validateEntry(Entry entry, ByteBuffer buffer) {
        if (entry.size != buffer.limit()) {
            String message = String.format("[%d] entry.size(%d) != buffer.limit(%d)", entry.blockNumber, entry.size, buffer.limit());
            throw new IllegalStateException(message);
        }
        long checksum = BufferData.getChecksum(buffer);
        if (entry.checksum != checksum) {
            String message = String.format("[%d] entry.checksum(%d) != buffer checksum(%d)", entry.blockNumber, entry.checksum, checksum);
            throw new IllegalStateException(message);
        }
    }

    private String getIntList(Iterable<Integer> nums) {
        ArrayList<String> numList = new ArrayList<String>();
        ArrayList<Integer> numbers = new ArrayList<Integer>();
        for (Integer n : nums) {
            numbers.add(n);
        }
        Collections.sort(numbers);
        int index = 0;
        while (index < numbers.size()) {
            int start;
            int prev = start = ((Integer)numbers.get(index)).intValue();
            int end = start;
            while (++index < numbers.size() && (end = ((Integer)numbers.get(index)).intValue()) == prev + 1) {
                prev = end;
            }
            if (start == prev) {
                numList.add(Integer.toString(start));
                continue;
            }
            numList.add(String.format("%d~%d", start, prev));
        }
        return String.join((CharSequence)", ", numList);
    }

    private String getStats() {
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("#entries = %d, #gets = %d", this.blocks.size(), this.numGets));
        return sb.toString();
    }

    public static boolean isCacheSpaceAvailable(long fileSize, Configuration conf, LocalDirAllocator localDirAllocator) {
        try {
            java.nio.file.Path cacheFilePath = SingleFilePerBlockCache.getTempFilePath(conf, localDirAllocator);
            long freeSpace = new File(cacheFilePath.toString()).getUsableSpace();
            LOG.info("fileSize = {}, freeSpace = {}", (Object)fileSize, (Object)freeSpace);
            Files.deleteIfExists(cacheFilePath);
            return fileSize < freeSpace;
        }
        catch (IOException e) {
            LOG.error("isCacheSpaceAvailable", (Throwable)e);
            return false;
        }
    }

    private static java.nio.file.Path getTempFilePath(Configuration conf, LocalDirAllocator localDirAllocator) throws IOException {
        Path path = localDirAllocator.getLocalPathForWrite(CACHE_FILE_PREFIX, conf);
        File dir = new File(path.getParent().toUri().getPath());
        String prefix = path.getName();
        File tmpFile = File.createTempFile(prefix, BINARY_FILE_SUFFIX, dir);
        java.nio.file.Path tmpFilePath = Paths.get(tmpFile.toURI());
        return Files.setPosixFilePermissions(tmpFilePath, TEMP_FILE_ATTRS);
    }

    private static final class Entry {
        private final int blockNumber;
        private final java.nio.file.Path path;
        private final int size;
        private final long checksum;
        private final ReentrantReadWriteLock lock;
        private Entry previous;
        private Entry next;

        Entry(int blockNumber, java.nio.file.Path path, int size, long checksum) {
            this.blockNumber = blockNumber;
            this.path = path;
            this.size = size;
            this.checksum = checksum;
            this.lock = new ReentrantReadWriteLock();
            this.previous = null;
            this.next = null;
        }

        public String toString() {
            return String.format("([%03d] %s: size = %d, checksum = %d)", this.blockNumber, this.path, this.size, this.checksum);
        }

        private void takeLock(LockType lockType) {
            if (LockType.READ == lockType) {
                this.lock.readLock().lock();
            } else if (LockType.WRITE == lockType) {
                this.lock.writeLock().lock();
            }
        }

        private void releaseLock(LockType lockType) {
            if (LockType.READ == lockType) {
                this.lock.readLock().unlock();
            } else if (LockType.WRITE == lockType) {
                this.lock.writeLock().unlock();
            }
        }

        private boolean takeLock(LockType lockType, long timeout, TimeUnit unit) {
            try {
                if (LockType.READ == lockType) {
                    return this.lock.readLock().tryLock(timeout, unit);
                }
                if (LockType.WRITE == lockType) {
                    return this.lock.writeLock().tryLock(timeout, unit);
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Thread interrupted while trying to acquire {} lock", (Object)lockType, (Object)e);
                Thread.currentThread().interrupt();
            }
            return false;
        }

        private Entry getPrevious() {
            return this.previous;
        }

        private void setPrevious(Entry previous) {
            this.previous = previous;
        }

        private Entry getNext() {
            return this.next;
        }

        private void setNext(Entry next) {
            this.next = next;
        }

        private static enum LockType {
            READ,
            WRITE;

        }
    }
}

