/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import javax.security.auth.kerberos.KerberosPrincipal;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.StringUtils;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class TestSecurityUtil {
    @BeforeClass
    public static void unsetKerberosRealm() {
        System.setProperty("java.security.krb5.kdc", "");
        System.setProperty("java.security.krb5.realm", "NONE");
    }

    @Test
    public void isOriginalTGTReturnsCorrectValues() {
        Assert.assertTrue((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/foo@foo")));
        Assert.assertTrue((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/foo.bar.bat@foo.bar.bat")));
        Assert.assertFalse((boolean)SecurityUtil.isTGSPrincipal(null));
        Assert.assertFalse((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("blah")));
        Assert.assertFalse((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/hello")));
        Assert.assertFalse((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/foo@FOO")));
    }

    private void verify(String original, String hostname, String expected) throws IOException {
        Assert.assertEquals((Object)expected, (Object)SecurityUtil.getServerPrincipal((String)original, (String)hostname));
        InetAddress addr = this.mockAddr(hostname);
        Assert.assertEquals((Object)expected, (Object)SecurityUtil.getServerPrincipal((String)original, (InetAddress)addr));
    }

    private InetAddress mockAddr(String reverseTo) {
        InetAddress mock = (InetAddress)Mockito.mock(InetAddress.class);
        ((InetAddress)Mockito.doReturn((Object)reverseTo).when((Object)mock)).getCanonicalHostName();
        return mock;
    }

    @Test
    public void testGetServerPrincipal() throws IOException {
        String service = "hdfs/";
        String realm = "@REALM";
        String hostname = "foohost";
        String userPrincipal = "foo@FOOREALM";
        String shouldReplace = service + "_HOST" + realm;
        String replaced = service + hostname + realm;
        this.verify(shouldReplace, hostname, replaced);
        String shouldNotReplace = service + "_HOST" + "NAME" + realm;
        this.verify(shouldNotReplace, hostname, shouldNotReplace);
        this.verify(userPrincipal, hostname, userPrincipal);
        InetAddress notUsed = (InetAddress)Mockito.mock(InetAddress.class);
        Assert.assertEquals((Object)shouldNotReplace, (Object)SecurityUtil.getServerPrincipal((String)shouldNotReplace, (InetAddress)notUsed));
        ((InetAddress)Mockito.verify((Object)notUsed, (VerificationMode)Mockito.never())).getCanonicalHostName();
    }

    @Test
    public void testPrincipalsWithLowerCaseHosts() throws IOException {
        String service = "xyz/";
        String realm = "@REALM";
        String principalInConf = service + "_HOST" + realm;
        String hostname = "FooHost";
        String principal = service + StringUtils.toLowerCase((String)hostname) + realm;
        this.verify(principalInConf, hostname, principal);
    }

    @Test
    public void testLocalHostNameForNullOrWild() throws Exception {
        String local = StringUtils.toLowerCase((String)SecurityUtil.getLocalHostName());
        Assert.assertEquals((Object)("hdfs/" + local + "@REALM"), (Object)SecurityUtil.getServerPrincipal((String)"hdfs/_HOST@REALM", (String)null));
        Assert.assertEquals((Object)("hdfs/" + local + "@REALM"), (Object)SecurityUtil.getServerPrincipal((String)"hdfs/_HOST@REALM", (String)"0.0.0.0"));
    }

    @Test
    public void testStartsWithIncorrectSettings() throws IOException {
        Configuration conf = new Configuration();
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)conf);
        String keyTabKey = "key";
        conf.set(keyTabKey, "");
        UserGroupInformation.setConfiguration((Configuration)conf);
        boolean gotException = false;
        try {
            SecurityUtil.login((Configuration)conf, (String)keyTabKey, (String)"", (String)"");
        }
        catch (IOException e) {
            gotException = true;
        }
        Assert.assertTrue((String)"Exception for empty keytabfile name was expected", (boolean)gotException);
    }

    @Test
    public void testGetHostFromPrincipal() {
        Assert.assertEquals((Object)"host", (Object)SecurityUtil.getHostFromPrincipal((String)"service/host@realm"));
        Assert.assertEquals(null, (Object)SecurityUtil.getHostFromPrincipal((String)"service@realm"));
    }

    @Test
    public void testBuildDTServiceName() {
        SecurityUtil.setTokenServiceUseIp((boolean)true);
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://LocalHost"), (int)123));
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://LocalHost:123"), (int)456));
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://127.0.0.1"), (int)123));
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://127.0.0.1:123"), (int)456));
    }

    @Test
    public void testBuildTokenServiceSockAddr() {
        SecurityUtil.setTokenServiceUseIp((boolean)true);
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildTokenService((InetSocketAddress)new InetSocketAddress("LocalHost", 123)).toString());
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildTokenService((InetSocketAddress)new InetSocketAddress("127.0.0.1", 123)).toString());
        Assert.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildTokenService((InetSocketAddress)NetUtils.createSocketAddr((String)"127.0.0.1", (int)123)).toString());
    }

    @Test
    public void testGoodHostsAndPorts() {
        InetSocketAddress compare = NetUtils.createSocketAddrForHost((String)"localhost", (int)123);
        this.runGoodCases(compare, "localhost", 123);
        this.runGoodCases(compare, "localhost:", 123);
        this.runGoodCases(compare, "localhost:123", 456);
    }

    void runGoodCases(InetSocketAddress addr, String host, int port) {
        Assert.assertEquals((Object)addr, (Object)NetUtils.createSocketAddr((String)host, (int)port));
        Assert.assertEquals((Object)addr, (Object)NetUtils.createSocketAddr((String)("hdfs://" + host), (int)port));
        Assert.assertEquals((Object)addr, (Object)NetUtils.createSocketAddr((String)("hdfs://" + host + "/path"), (int)port));
    }

    @Test
    public void testBadHostsAndPorts() {
        this.runBadCases("", true);
        this.runBadCases(":", false);
        this.runBadCases("hdfs/", false);
        this.runBadCases("hdfs:/", false);
        this.runBadCases("hdfs://", true);
    }

    void runBadCases(String prefix, boolean validIfPosPort) {
        this.runBadPortPermutes(prefix, false);
        this.runBadPortPermutes(prefix + "*", false);
        this.runBadPortPermutes(prefix + "localhost", validIfPosPort);
        this.runBadPortPermutes(prefix + "localhost:-1", false);
        this.runBadPortPermutes(prefix + "localhost:-123", false);
        this.runBadPortPermutes(prefix + "localhost:xyz", false);
        this.runBadPortPermutes(prefix + "localhost/xyz", validIfPosPort);
        this.runBadPortPermutes(prefix + "localhost/:123", validIfPosPort);
        this.runBadPortPermutes(prefix + ":123", false);
        this.runBadPortPermutes(prefix + ":xyz", false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void runBadPortPermutes(String arg, boolean validIfPosPort) {
        int[] ports = new int[]{-123, -1, 123};
        boolean bad = false;
        try {
            NetUtils.createSocketAddr((String)arg);
        }
        catch (IllegalArgumentException e) {
            bad = true;
        }
        finally {
            Assert.assertTrue((String)("should be bad: '" + arg + "'"), (boolean)bad);
        }
        for (int port : ports) {
            if (validIfPosPort && port > 0) continue;
            bad = false;
            try {
                NetUtils.createSocketAddr((String)arg, (int)port);
            }
            catch (IllegalArgumentException e) {
                bad = true;
            }
            finally {
                Assert.assertTrue((String)("should be bad: '" + arg + "' (default port:" + port + ")"), (boolean)bad);
            }
        }
    }

    private void verifyValues(InetSocketAddress addr, String host, String ip, int port) {
        Assert.assertTrue((!addr.isUnresolved() ? 1 : 0) != 0);
        if (!SecurityUtil.useIpForTokenService) {
            Assert.assertEquals((Object)host, (Object)addr.getHostName());
            Assert.assertEquals((Object)host, (Object)addr.getAddress().getHostName());
        }
        Assert.assertEquals((Object)ip, (Object)addr.getAddress().getHostAddress());
        Assert.assertEquals((long)port, (long)addr.getPort());
    }

    private void verifyTokenService(InetSocketAddress addr, String host, String ip, int port, boolean useIp) {
        SecurityUtil.setTokenServiceUseIp((boolean)useIp);
        String serviceHost = useIp ? ip : StringUtils.toLowerCase((String)host);
        Token token = new Token();
        Text service = new Text(serviceHost + ":" + port);
        Assert.assertEquals((Object)service, (Object)SecurityUtil.buildTokenService((InetSocketAddress)addr));
        SecurityUtil.setTokenService((Token)token, (InetSocketAddress)addr);
        Assert.assertEquals((Object)service, (Object)token.getService());
        InetSocketAddress serviceAddr = SecurityUtil.getTokenServiceAddr((Token)token);
        Assert.assertNotNull((Object)serviceAddr);
        this.verifyValues(serviceAddr, serviceHost, ip, port);
    }

    private void verifyAddress(InetSocketAddress addr, String host, String ip, int port) {
        this.verifyValues(addr, host, ip, port);
        this.verifyTokenService(addr, host, ip, port, true);
        this.verifyTokenService(addr, host, ip, port, false);
    }

    private void verifyServiceAddr(String host, String ip) {
        int port = 123;
        InetSocketAddress addr = NetUtils.createSocketAddrForHost((String)host, (int)port);
        this.verifyAddress(addr, host, ip, port);
        addr = NetUtils.createSocketAddr((String)(host + ":" + port));
        this.verifyAddress(addr, host, ip, port);
        addr = NetUtils.createSocketAddr((String)(host + ":" + port), (int)(port + 1));
        this.verifyAddress(addr, host, ip, port);
        addr = NetUtils.createSocketAddr((String)host, (int)port);
        this.verifyAddress(addr, host, ip, port);
    }

    @Test
    public void testSocketAddrWithName() {
        String staticHost = "my";
        NetUtils.addStaticResolution((String)staticHost, (String)"localhost");
        this.verifyServiceAddr("LocalHost", "127.0.0.1");
    }

    @Test
    public void testSocketAddrWithIP() {
        String staticHost = "127.0.0.1";
        NetUtils.addStaticResolution((String)staticHost, (String)"localhost");
        this.verifyServiceAddr(staticHost, "127.0.0.1");
    }

    @Test
    public void testSocketAddrWithNameToStaticName() {
        String staticHost = "host1";
        NetUtils.addStaticResolution((String)staticHost, (String)"localhost");
        this.verifyServiceAddr(staticHost, "127.0.0.1");
    }

    @Test
    public void testSocketAddrWithNameToStaticIP() {
        String staticHost = "host3";
        NetUtils.addStaticResolution((String)staticHost, (String)"255.255.255.255");
        this.verifyServiceAddr(staticHost, "255.255.255.255");
    }

    @Test
    public void testSocketAddrWithIPToStaticIP() {
        String staticHost = "1.2.3.4";
        NetUtils.addStaticResolution((String)staticHost, (String)"255.255.255.255");
        this.verifyServiceAddr(staticHost, "255.255.255.255");
    }

    @Test
    public void testGetAuthenticationMethod() {
        Configuration conf = new Configuration();
        conf.unset("hadoop.security.authentication");
        Assert.assertEquals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE, (Object)SecurityUtil.getAuthenticationMethod((Configuration)conf));
        conf.set("hadoop.security.authentication", "simple");
        Assert.assertEquals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE, (Object)SecurityUtil.getAuthenticationMethod((Configuration)conf));
        conf.set("hadoop.security.authentication", "kerberos");
        Assert.assertEquals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS, (Object)SecurityUtil.getAuthenticationMethod((Configuration)conf));
        conf.set("hadoop.security.authentication", "kaboom");
        String error = null;
        try {
            SecurityUtil.getAuthenticationMethod((Configuration)conf);
        }
        catch (Exception e) {
            error = e.toString();
        }
        Assert.assertEquals((Object)"java.lang.IllegalArgumentException: Invalid attribute value for hadoop.security.authentication of kaboom", (Object)error);
    }

    @Test
    public void testSetAuthenticationMethod() {
        Configuration conf = new Configuration();
        SecurityUtil.setAuthenticationMethod(null, (Configuration)conf);
        Assert.assertEquals((Object)"simple", (Object)conf.get("hadoop.security.authentication"));
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.SIMPLE, (Configuration)conf);
        Assert.assertEquals((Object)"simple", (Object)conf.get("hadoop.security.authentication"));
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)conf);
        Assert.assertEquals((Object)"kerberos", (Object)conf.get("hadoop.security.authentication"));
    }
}

