/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.metrics;

import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.util.EnumSet;
import java.util.List;
import java.util.Random;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.key.KeyProvider;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemTestHelper;
import org.apache.hadoop.fs.FileSystemTestWrapper;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSNNTopology;
import org.apache.hadoop.hdfs.client.CreateEncryptionZoneFlag;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManagerTestUtil;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataNodeTestUtils;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.server.namenode.ha.HATestUtil;
import org.apache.hadoop.metrics2.MetricsRecordBuilder;
import org.apache.hadoop.metrics2.MetricsSource;
import org.apache.hadoop.metrics2.lib.DefaultMetricsSystem;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.MetricsAsserts;
import org.apache.log4j.Level;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestNameNodeMetrics {
    private static final Configuration CONF = new HdfsConfiguration();
    private static final int DFS_REPLICATION_INTERVAL = 1;
    private static final Path TEST_ROOT_DIR_PATH = new Path("/testNameNodeMetrics");
    private static final String NN_METRICS = "NameNodeActivity";
    private static final String NS_METRICS = "FSNamesystem";
    public static final Log LOG = LogFactory.getLog(TestNameNodeMetrics.class);
    private static final int DATANODE_COUNT = 3;
    private static final int WAIT_GAUGE_VALUE_RETRIES = 20;
    private static final int PERCENTILES_INTERVAL = 1;
    private MiniDFSCluster cluster;
    private DistributedFileSystem fs;
    private final Random rand = new Random();
    private FSNamesystem namesystem;
    private BlockManager bm;

    private static Path getTestPath(String fileName) {
        return new Path(TEST_ROOT_DIR_PATH, fileName);
    }

    @Before
    public void setUp() throws Exception {
        this.cluster = new MiniDFSCluster.Builder(CONF).numDataNodes(3).build();
        this.cluster.waitActive();
        this.namesystem = this.cluster.getNamesystem();
        this.bm = this.namesystem.getBlockManager();
        this.fs = this.cluster.getFileSystem();
    }

    @After
    public void tearDown() throws Exception {
        MetricsSource source = DefaultMetricsSystem.instance().getSource("UgiMetrics");
        if (source != null) {
            MetricsRecordBuilder rb = MetricsAsserts.getMetrics((MetricsSource)source);
            MetricsAsserts.assertQuantileGauges((String)"GetGroups1s", (MetricsRecordBuilder)rb);
        }
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    private void createFile(Path file, long fileLen, short replicas) throws IOException {
        DFSTestUtil.createFile((FileSystem)this.fs, file, fileLen, replicas, this.rand.nextLong());
    }

    private void updateMetrics() throws Exception {
        Thread.sleep(1000L);
    }

    private void readFile(FileSystem fileSys, Path name) throws IOException {
        FSDataInputStream stm = fileSys.open(name);
        byte[] buffer = new byte[4];
        stm.read(buffer, 0, 4);
        stm.close();
    }

    @Test(timeout=1800L)
    public void testCapacityMetrics() throws Exception {
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NS_METRICS);
        long capacityTotal = MetricsAsserts.getLongGauge((String)"CapacityTotal", (MetricsRecordBuilder)rb);
        assert (capacityTotal != 0L);
        long capacityUsed = MetricsAsserts.getLongGauge((String)"CapacityUsed", (MetricsRecordBuilder)rb);
        long capacityRemaining = MetricsAsserts.getLongGauge((String)"CapacityRemaining", (MetricsRecordBuilder)rb);
        long capacityUsedNonDFS = MetricsAsserts.getLongGauge((String)"CapacityUsedNonDFS", (MetricsRecordBuilder)rb);
        assert (capacityUsed + capacityRemaining + capacityUsedNonDFS <= capacityTotal);
    }

    @Test
    public void testStaleNodes() throws Exception {
        DataNode dn;
        int i;
        for (i = 0; i < 2; ++i) {
            dn = this.cluster.getDataNodes().get(i);
            DataNodeTestUtils.setHeartbeatsDisabledForTests(dn, true);
            long staleInterval = CONF.getLong("dfs.namenode.stale.datanode.interval", 30000L);
            DatanodeDescriptor dnDes = this.cluster.getNameNode().getNamesystem().getBlockManager().getDatanodeManager().getDatanode(dn.getDatanodeId());
            DFSTestUtil.resetLastUpdatesWithOffset((DatanodeInfo)dnDes, -(staleInterval + 1L));
        }
        BlockManagerTestUtil.checkHeartbeat(this.cluster.getNameNode().getNamesystem().getBlockManager());
        MetricsAsserts.assertGauge((String)"StaleDataNodes", (int)2, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        for (i = 0; i < 2; ++i) {
            dn = this.cluster.getDataNodes().get(i);
            DataNodeTestUtils.setHeartbeatsDisabledForTests(dn, false);
            DatanodeDescriptor dnDes = this.cluster.getNameNode().getNamesystem().getBlockManager().getDatanodeManager().getDatanode(dn.getDatanodeId());
            DFSTestUtil.resetLastUpdatesWithOffset((DatanodeInfo)dnDes, 0L);
        }
        BlockManagerTestUtil.checkHeartbeat(this.cluster.getNameNode().getNamesystem().getBlockManager());
        MetricsAsserts.assertGauge((String)"StaleDataNodes", (int)0, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
    }

    @Test
    public void testFileAdd() throws Exception {
        Path file = TestNameNodeMetrics.getTestPath("testFileAdd");
        this.createFile(file, 3200L, (short)3);
        long blockCount = 32L;
        int blockCapacity = this.namesystem.getBlockCapacity();
        this.updateMetrics();
        MetricsAsserts.assertGauge((String)"BlockCapacity", (int)blockCapacity, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NN_METRICS);
        MetricsAsserts.assertCounter((String)"CreateFileOps", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertCounter((String)"FilesCreated", (long)file.depth(), (MetricsRecordBuilder)rb);
        this.updateMetrics();
        long filesTotal = file.depth() + 1;
        rb = MetricsAsserts.getMetrics((String)NS_METRICS);
        MetricsAsserts.assertGauge((String)"FilesTotal", (long)filesTotal, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"BlocksTotal", (long)32L, (MetricsRecordBuilder)rb);
        this.fs.delete(file, true);
        rb = this.waitForDnMetricValue(NS_METRICS, "FilesTotal", --filesTotal);
        MetricsAsserts.assertGauge((String)"BlocksTotal", (long)0L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"PendingDeletionBlocks", (long)0L, (MetricsRecordBuilder)rb);
        rb = MetricsAsserts.getMetrics((String)NN_METRICS);
        MetricsAsserts.assertCounter((String)"DeleteFileOps", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertCounter((String)"FilesDeleted", (long)1L, (MetricsRecordBuilder)rb);
    }

    @Test
    public void testCorruptBlock() throws Exception {
        Path file = TestNameNodeMetrics.getTestPath("testCorruptBlock");
        this.createFile(file, 100L, (short)2);
        LocatedBlock block = NameNodeAdapter.getBlockLocations(this.cluster.getNameNode(), file.toString(), 0L, 1L).get(0);
        this.cluster.getNamesystem().writeLock();
        try {
            this.bm.findAndMarkBlockAsCorrupt(block.getBlock(), block.getLocations()[0], "STORAGE_ID", "TEST");
        }
        finally {
            this.cluster.getNamesystem().writeUnlock();
        }
        this.updateMetrics();
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NS_METRICS);
        MetricsAsserts.assertGauge((String)"CorruptBlocks", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"PendingReplicationBlocks", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"ScheduledReplicationBlocks", (long)1L, (MetricsRecordBuilder)rb);
        this.fs.delete(file, true);
        rb = this.waitForDnMetricValue(NS_METRICS, "CorruptBlocks", 0L);
        MetricsAsserts.assertGauge((String)"PendingReplicationBlocks", (long)0L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"ScheduledReplicationBlocks", (long)0L, (MetricsRecordBuilder)rb);
    }

    @Test
    public void testExcessBlocks() throws Exception {
        Path file = TestNameNodeMetrics.getTestPath("testExcessBlocks");
        this.createFile(file, 100L, (short)2);
        NameNodeAdapter.setReplication(this.namesystem, file.toString(), (short)1);
        this.updateMetrics();
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NS_METRICS);
        MetricsAsserts.assertGauge((String)"ExcessBlocks", (long)1L, (MetricsRecordBuilder)rb);
        this.fs.delete(file, true);
        rb = MetricsAsserts.getMetrics((String)NS_METRICS);
        MetricsAsserts.assertGauge((String)"ExcessBlocks", (long)0L, (MetricsRecordBuilder)rb);
        Assert.assertTrue((boolean)this.bm.excessReplicateMap.isEmpty());
    }

    @Test
    public void testMissingBlock() throws Exception {
        Path file = TestNameNodeMetrics.getTestPath("testMissingBlocks");
        this.createFile(file, 100L, (short)1);
        LocatedBlock block = NameNodeAdapter.getBlockLocations(this.cluster.getNameNode(), file.toString(), 0L, 1L).get(0);
        this.cluster.getNamesystem().writeLock();
        try {
            this.bm.findAndMarkBlockAsCorrupt(block.getBlock(), block.getLocations()[0], "STORAGE_ID", "TEST");
        }
        finally {
            this.cluster.getNamesystem().writeUnlock();
        }
        this.updateMetrics();
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NS_METRICS);
        MetricsAsserts.assertGauge((String)"UnderReplicatedBlocks", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"MissingBlocks", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertGauge((String)"MissingReplOneBlocks", (long)1L, (MetricsRecordBuilder)rb);
        this.fs.delete(file, true);
        this.waitForDnMetricValue(NS_METRICS, "UnderReplicatedBlocks", 0L);
    }

    private void waitForDeletion() throws InterruptedException {
        Thread.sleep(4000L);
    }

    private MetricsRecordBuilder waitForDnMetricValue(String source, String name, long expected) throws Exception {
        this.waitForDeletion();
        int retries = 80;
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)source);
        long gauge = MetricsAsserts.getLongGauge((String)name, (MetricsRecordBuilder)rb);
        while (gauge != expected && --retries > 0) {
            Thread.sleep(500L);
            rb = MetricsAsserts.getMetrics((String)source);
            gauge = MetricsAsserts.getLongGauge((String)name, (MetricsRecordBuilder)rb);
        }
        MetricsAsserts.assertGauge((String)name, (long)expected, (MetricsRecordBuilder)rb);
        return rb;
    }

    @Test
    public void testRenameMetrics() throws Exception {
        Path src = TestNameNodeMetrics.getTestPath("src");
        this.createFile(src, 100L, (short)1);
        Path target = TestNameNodeMetrics.getTestPath("target");
        this.createFile(target, 100L, (short)1);
        this.fs.rename(src, target, new Options.Rename[]{Options.Rename.OVERWRITE});
        this.updateMetrics();
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NN_METRICS);
        MetricsAsserts.assertCounter((String)"FilesRenamed", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertCounter((String)"FilesDeleted", (long)1L, (MetricsRecordBuilder)rb);
    }

    @Test
    public void testGetBlockLocationMetric() throws Exception {
        Path file1_Path = new Path(TEST_ROOT_DIR_PATH, "file1.dat");
        MetricsAsserts.assertCounter((String)"GetBlockLocations", (long)0L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NN_METRICS));
        this.createFile(file1_Path, 100L, (short)2);
        this.updateMetrics();
        MetricsAsserts.assertCounter((String)"GetBlockLocations", (long)0L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NN_METRICS));
        this.readFile((FileSystem)this.fs, file1_Path);
        this.updateMetrics();
        MetricsAsserts.assertCounter((String)"GetBlockLocations", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NN_METRICS));
        this.readFile((FileSystem)this.fs, file1_Path);
        this.readFile((FileSystem)this.fs, file1_Path);
        this.updateMetrics();
        MetricsAsserts.assertCounter((String)"GetBlockLocations", (long)3L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NN_METRICS));
    }

    @Test(timeout=300000L)
    public void testTransactionSinceLastCheckpointMetrics() throws Exception {
        Random random = new Random();
        for (int retryCount = 0; retryCount < 5; ++retryCount) {
            try {
                int basePort = 10060 + random.nextInt(100) * 2;
                MiniDFSNNTopology topology = new MiniDFSNNTopology().addNameservice(new MiniDFSNNTopology.NSConf("ns1").addNN(new MiniDFSNNTopology.NNConf("nn1").setHttpPort(basePort)).addNN(new MiniDFSNNTopology.NNConf("nn2").setHttpPort(basePort + 1)));
                HdfsConfiguration conf2 = new HdfsConfiguration();
                conf2.setInt("dfs.namenode.checkpoint.txns", 100);
                conf2.setInt("dfs.namenode.checkpoint.check.period", 1);
                conf2.setInt("dfs.ha.tail-edits.period", 1);
                MiniDFSCluster cluster2 = new MiniDFSCluster.Builder((Configuration)conf2).nnTopology(topology).numDataNodes(1).build();
                cluster2.waitActive();
                DistributedFileSystem fs2 = cluster2.getFileSystem(0);
                NameNode nn0 = cluster2.getNameNode(0);
                NameNode nn1 = cluster2.getNameNode(1);
                cluster2.transitionToActive(0);
                fs2.mkdirs(new Path("/tmp-t1"));
                fs2.mkdirs(new Path("/tmp-t2"));
                HATestUtil.waitForStandbyToCatchUp(nn0, nn1);
                Assert.assertEquals((String)"SBN failed to track 2 transactions pre-checkpoint.", (long)4L, (long)cluster2.getNameNode(1).getNamesystem().getTransactionsSinceLastCheckpoint());
                for (int i = 1; i <= 94; ++i) {
                    fs2.mkdirs(new Path("/tmp-" + i));
                }
                HATestUtil.waitForStandbyToCatchUp(nn0, nn1);
                HATestUtil.waitForCheckpoint(cluster2, 1, (List<Integer>)ImmutableList.of((Object)100));
                Assert.assertEquals((String)"Should be zero right after the checkpoint.", (long)0L, (long)cluster2.getNameNode(1).getNamesystem().getTransactionsSinceLastCheckpoint());
                fs2.mkdirs(new Path("/tmp-t3"));
                fs2.mkdirs(new Path("/tmp-t4"));
                HATestUtil.waitForStandbyToCatchUp(nn0, nn1);
                Assert.assertEquals((String)"SBN failed to track 2 added txns after the ckpt.", (long)4L, (long)cluster2.getNameNode(1).getNamesystem().getTransactionsSinceLastCheckpoint());
                cluster2.shutdown();
                break;
            }
            catch (Exception e) {
                LOG.warn((Object)("Unable to set up HA cluster, exception thrown: " + e));
                continue;
            }
        }
    }

    @Test
    public void testTransactionAndCheckpointMetrics() throws Exception {
        long lastCkptTime = MetricsAsserts.getLongGauge((String)"LastCheckpointTime", (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"LastCheckpointTime", (long)lastCkptTime, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"LastWrittenTransactionId", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastCheckpoint", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastLogRoll", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        this.fs.mkdirs(new Path(TEST_ROOT_DIR_PATH, "/tmp"));
        this.updateMetrics();
        MetricsAsserts.assertGauge((String)"LastCheckpointTime", (long)lastCkptTime, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"LastWrittenTransactionId", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastCheckpoint", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastLogRoll", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        this.cluster.getNameNodeRpc().rollEditLog();
        this.updateMetrics();
        MetricsAsserts.assertGauge((String)"LastCheckpointTime", (long)lastCkptTime, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"LastWrittenTransactionId", (long)4L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastCheckpoint", (long)4L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastLogRoll", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        this.cluster.getNameNodeRpc().setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER, false);
        this.cluster.getNameNodeRpc().saveNamespace();
        this.cluster.getNameNodeRpc().setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE, false);
        this.updateMetrics();
        long newLastCkptTime = MetricsAsserts.getLongGauge((String)"LastCheckpointTime", (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        Assert.assertTrue((lastCkptTime < newLastCkptTime ? 1 : 0) != 0);
        MetricsAsserts.assertGauge((String)"LastWrittenTransactionId", (long)6L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastCheckpoint", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"TransactionsSinceLastLogRoll", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
    }

    @Test
    public void testSyncAndBlockReportMetric() throws Exception {
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NN_METRICS);
        MetricsAsserts.assertCounter((String)"SyncsNumOps", (long)1L, (MetricsRecordBuilder)rb);
        MetricsAsserts.assertCounter((String)"BlockReportNumOps", (long)(3L * (long)this.cluster.getStoragesPerDatanode()), (MetricsRecordBuilder)rb);
        Thread.sleep(2000L);
        MetricsAsserts.assertQuantileGauges((String)"Syncs1s", (MetricsRecordBuilder)rb);
        MetricsAsserts.assertQuantileGauges((String)"BlockReport1s", (MetricsRecordBuilder)rb);
    }

    @Test
    public void testReadWriteOps() throws Exception {
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NN_METRICS);
        long startWriteCounter = MetricsAsserts.getLongCounter((String)"TransactionsNumOps", (MetricsRecordBuilder)rb);
        Path file1_Path = new Path(TEST_ROOT_DIR_PATH, "ReadData.dat");
        this.createFile(file1_Path, 0x100000L, (short)2);
        this.readFile((FileSystem)this.fs, file1_Path);
        MetricsRecordBuilder rbNew = MetricsAsserts.getMetrics((String)NN_METRICS);
        Assert.assertTrue((MetricsAsserts.getLongCounter((String)"TransactionsNumOps", (MetricsRecordBuilder)rbNew) > startWriteCounter ? 1 : 0) != 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=60000L)
    public void testNumActiveClientsAndFilesUnderConstructionMetrics() throws Exception {
        Path file1 = TestNameNodeMetrics.getTestPath("testFileAdd1");
        this.createFile(file1, 100L, (short)3);
        MetricsAsserts.assertGauge((String)"NumActiveClients", (long)0L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)0L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        Path file2 = new Path("/testFileAdd2");
        FSDataOutputStream output2 = this.fs.create(file2);
        output2.writeBytes("Some test data");
        MetricsAsserts.assertGauge((String)"NumActiveClients", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        Path file3 = new Path("/testFileAdd3");
        FSDataOutputStream output3 = this.fs.create(file3);
        output3.writeBytes("Some test data");
        MetricsAsserts.assertGauge((String)"NumActiveClients", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        try (DistributedFileSystem fs1 = (DistributedFileSystem)this.cluster.getNewFileSystemInstance(0);){
            Path file4 = new Path("/testFileAdd4");
            FSDataOutputStream output4 = fs1.create(file4);
            output4.writeBytes("Some test data");
            MetricsAsserts.assertGauge((String)"NumActiveClients", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)3L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            Path file5 = new Path("/testFileAdd35");
            FSDataOutputStream output5 = fs1.create(file5);
            output5.writeBytes("Some test data");
            MetricsAsserts.assertGauge((String)"NumActiveClients", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)4L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            output2.close();
            output3.close();
            MetricsAsserts.assertGauge((String)"NumActiveClients", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)2L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            output4.close();
            output5.close();
            MetricsAsserts.assertGauge((String)"NumActiveClients", (long)0L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
            MetricsAsserts.assertGauge((String)"NumFilesUnderConstruction", (long)0L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)NS_METRICS));
        }
    }

    @Test
    public void testGenerateEDEKTime() throws IOException, NoSuchAlgorithmException {
        HdfsConfiguration conf = new HdfsConfiguration();
        FileSystemTestHelper fsHelper = new FileSystemTestHelper();
        String testRoot = fsHelper.getTestRootDir();
        File testRootDir = new File(testRoot).getAbsoluteFile();
        conf.set("hadoop.security.key.provider.path", "jceks://file" + new Path(testRootDir.toString(), "test.jks").toUri());
        conf.setBoolean("dfs.namenode.delegation.token.always-use", true);
        conf.setInt("dfs.namenode.list.encryption.zones.num.responses", 2);
        try (MiniDFSCluster clusterEDEK = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(1).build();){
            DistributedFileSystem fsEDEK = clusterEDEK.getFileSystem();
            FileSystemTestWrapper fsWrapper = new FileSystemTestWrapper((FileSystem)fsEDEK);
            HdfsAdmin dfsAdmin = new HdfsAdmin(clusterEDEK.getURI(), (Configuration)conf);
            fsEDEK.getClient().setKeyProvider((KeyProvider)clusterEDEK.getNameNode().getNamesystem().getProvider());
            String testKey = "test_key";
            DFSTestUtil.createKey(testKey, clusterEDEK, (Configuration)conf);
            Path zoneParent = new Path("/zones");
            Path zone1 = new Path(zoneParent, "zone1");
            fsWrapper.mkdir(zone1, FsPermission.getDirDefault(), true);
            dfsAdmin.createEncryptionZone(zone1, "test_key", EnumSet.of(CreateEncryptionZoneFlag.NO_TRASH));
            MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)NN_METRICS);
            for (int i = 0; i < 3; ++i) {
                Path filePath = new Path("/zones/zone1/testfile-" + i);
                DFSTestUtil.createFile((FileSystem)fsEDEK, filePath, 1024L, (short)3, 1L);
                MetricsAsserts.assertQuantileGauges((String)"GenerateEDEKTime1s", (MetricsRecordBuilder)rb);
            }
        }
    }

    static {
        CONF.setLong("dfs.blocksize", 100L);
        CONF.setInt("dfs.bytes-per-checksum", 1);
        CONF.setLong("dfs.heartbeat.interval", 1L);
        CONF.setInt("dfs.namenode.replication.interval", 1);
        CONF.set("dfs.metrics.percentiles.intervals", "1");
        CONF.setBoolean("dfs.namenode.avoid.read.stale.datanode", true);
        GenericTestUtils.setLogLevel((Log)LogFactory.getLog(MetricsAsserts.class), (Level)Level.DEBUG);
    }
}

