/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.OpenFileEntry;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Lists;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Logger LOG = LoggerFactory.getLogger((String)LeaseManager.class.getName());
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit;
    static final int INODE_FILTER_WORKER_COUNT_MAX = 4;
    static final int INODE_FILTER_WORKER_TASK_MIN = 512;
    private long lastHolderUpdateTime;
    private String internalLeaseHolder;
    private final HashMap<String, Lease> leases = new HashMap();
    private final TreeMap<Long, Lease> leasesById = new TreeMap();
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        Configuration conf = new Configuration();
        this.fsnamesystem = fsnamesystem;
        this.hardLimit = conf.getLong("dfs.namenode.lease-hard-limit-sec", 1200L) * 1000L;
        this.updateInternalLeaseHolder();
    }

    private void updateInternalLeaseHolder() {
        this.lastHolderUpdateTime = Time.monotonicNow();
        this.internalLeaseHolder = "HDFS_NameNode-" + Time.formatTime((long)Time.now());
    }

    String getInternalLeaseHolder() {
        long elapsed = Time.monotonicNow() - this.lastHolderUpdateTime;
        if (elapsed > this.hardLimit) {
            this.updateInternalLeaseHolder();
        }
        return this.internalLeaseHolder;
    }

    Lease getLease(String holder) {
        return this.leases.get(holder);
    }

    synchronized long getNumUnderConstructionBlocks() {
        assert (this.fsnamesystem.hasReadLock()) : "The FSNamesystem read lock wasn'tacquired before counting under construction blocks";
        long numUCBlocks = 0L;
        for (Long id : this.getINodeIdWithLeases()) {
            INode inode = this.fsnamesystem.getFSDirectory().getInode(id);
            if (inode == null) {
                LOG.warn("Failed to find inode {} in getNumUnderConstructionBlocks().", (Object)id);
                continue;
            }
            INodeFile cons = inode.asFile();
            if (!cons.isUnderConstruction()) {
                LOG.warn("The file {} is not under construction but has lease.", (Object)cons.getFullPathName());
                continue;
            }
            BlockInfo[] blocks = cons.getBlocks();
            if (blocks == null) continue;
            for (BlockInfo b : blocks) {
                if (b.isComplete()) continue;
                ++numUCBlocks;
            }
        }
        LOG.info("Number of blocks under construction: {}", (Object)numUCBlocks);
        return numUCBlocks;
    }

    Collection<Long> getINodeIdWithLeases() {
        return this.leasesById.keySet();
    }

    @VisibleForTesting
    Set<INodesInPath> getINodeWithLeases() throws IOException {
        return this.getINodeWithLeases(null);
    }

    private synchronized INode[] getINodesWithLease() {
        ArrayList<INode> inodes = new ArrayList<INode>(this.leasesById.size());
        for (long inodeId : this.leasesById.keySet()) {
            INode currentINode = this.fsnamesystem.getFSDirectory().getInode(inodeId);
            if (currentINode == null || !currentINode.isFile() || this.fsnamesystem.isFileDeleted(currentINode.asFile())) continue;
            inodes.add(currentINode);
        }
        return inodes.toArray(new INode[0]);
    }

    public Set<INodesInPath> getINodeWithLeases(final INodeDirectory ancestorDir) throws IOException {
        assert (this.fsnamesystem.hasReadLock());
        long startTimeMs = Time.monotonicNow();
        HashSet<INodesInPath> iipSet = new HashSet<INodesInPath>();
        final INode[] inodes = this.getINodesWithLease();
        final int inodeCount = inodes.length;
        if (inodeCount == 0) {
            return iipSet;
        }
        ArrayList futureList = Lists.newArrayList();
        final int workerCount = Math.min(4, (inodeCount - 1) / 512 + 1);
        ExecutorService inodeFilterService = Executors.newFixedThreadPool(workerCount);
        int workerIdx = 0;
        while (workerIdx < workerCount) {
            final int startIdx = workerIdx++;
            Callable<List<INodesInPath>> c = new Callable<List<INodesInPath>>(){

                @Override
                public List<INodesInPath> call() {
                    ArrayList iNodesInPaths = Lists.newArrayList();
                    for (int idx = startIdx; idx < inodeCount; idx += workerCount) {
                        INode inode = inodes[idx];
                        if (!inode.isFile()) continue;
                        INodesInPath inodesInPath = INodesInPath.fromINode(LeaseManager.this.fsnamesystem.getFSDirectory().getRoot(), inode.asFile());
                        if (ancestorDir != null && !inodesInPath.isDescendant(ancestorDir)) continue;
                        iNodesInPaths.add(inodesInPath);
                    }
                    return iNodesInPaths;
                }
            };
            futureList.add(inodeFilterService.submit(c));
        }
        inodeFilterService.shutdown();
        for (Future f : futureList) {
            try {
                iipSet.addAll((Collection)f.get());
            }
            catch (Exception e) {
                throw new IOException("Failed to get files with active leases", e);
            }
        }
        long endTimeMs = Time.monotonicNow();
        if (endTimeMs - startTimeMs > 1000L) {
            LOG.info("Took {} ms to collect {} open files with leases {}", new Object[]{endTimeMs - startTimeMs, iipSet.size(), ancestorDir != null ? " under " + ancestorDir.getFullPathName() : "."});
        }
        return iipSet;
    }

    public BatchedRemoteIterator.BatchedListEntries<OpenFileEntry> getUnderConstructionFiles(long prevId) throws IOException {
        return this.getUnderConstructionFiles(prevId, "/");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BatchedRemoteIterator.BatchedListEntries<OpenFileEntry> getUnderConstructionFiles(long prevId, String path) throws IOException {
        NavigableMap<Long, Lease> remainingLeases;
        assert (this.fsnamesystem.hasReadLock());
        LeaseManager leaseManager = this;
        synchronized (leaseManager) {
            remainingLeases = this.leasesById.tailMap(prevId, false);
        }
        Set inodeIds = remainingLeases.keySet();
        int numResponses = Math.min(this.fsnamesystem.getMaxListOpenFilesResponses(), inodeIds.size());
        ArrayList openFileEntries = Lists.newArrayListWithExpectedSize((int)numResponses);
        int count = 0;
        String fullPathName = null;
        Iterator inodeIdIterator = inodeIds.iterator();
        while (inodeIdIterator.hasNext()) {
            Long inodeId = (Long)inodeIdIterator.next();
            INode ucFile = this.fsnamesystem.getFSDirectory().getInode(inodeId);
            if (ucFile == null) continue;
            INodeFile inodeFile = ucFile.asFile();
            if (!inodeFile.isUnderConstruction()) {
                LOG.warn("The file {} is not under construction but has lease.", (Object)inodeFile.getFullPathName());
                continue;
            }
            fullPathName = inodeFile.getFullPathName();
            if (StringUtils.isEmpty((CharSequence)path) || DFSUtil.isParentEntry(fullPathName, path)) {
                openFileEntries.add(new OpenFileEntry(inodeFile.getId(), fullPathName, inodeFile.getFileUnderConstructionFeature().getClientName(), inodeFile.getFileUnderConstructionFeature().getClientMachine()));
                ++count;
            }
            if (count < numResponses) continue;
            break;
        }
        boolean hasMore = inodeIdIterator.hasNext();
        return new BatchedRemoteIterator.BatchedListEntries((List)openFileEntries, hasMore);
    }

    public synchronized Lease getLease(INodeFile src) {
        return this.leasesById.get(src.getId());
    }

    @VisibleForTesting
    public synchronized int countLease() {
        return this.leases.size();
    }

    synchronized long countPath() {
        return this.leasesById.size();
    }

    synchronized Lease addLease(String holder, long inodeId) {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder);
            this.leases.put(holder, lease);
        } else {
            this.renewLease(lease);
        }
        this.leasesById.put(inodeId, lease);
        lease.files.add(inodeId);
        return lease;
    }

    synchronized void removeLease(long inodeId) {
        Lease lease = this.leasesById.get(inodeId);
        if (lease != null) {
            this.removeLease(lease, inodeId);
        }
    }

    private synchronized void removeLease(Lease lease, long inodeId) {
        this.leasesById.remove(inodeId);
        if (!lease.removeFile(inodeId)) {
            LOG.debug("inode {} not found in lease.files (={})", (Object)inodeId, (Object)lease);
        }
        if (!lease.hasFiles() && this.leases.remove(lease.holder) == null) {
            LOG.error("{} not found", (Object)lease);
        }
    }

    synchronized void removeLease(String holder, INodeFile src) {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, src.getId());
        } else {
            LOG.warn("Removing non-existent lease! holder={} src={}", (Object)holder, (Object)src.getFullPathName());
        }
    }

    synchronized void removeAllLeases() {
        this.leasesById.clear();
        this.leases.clear();
    }

    synchronized Lease reassignLease(Lease lease, INodeFile src, String newHolder) {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            this.removeLease(lease, src.getId());
        }
        return this.addLease(newHolder, src.getId());
    }

    synchronized void renewLease(String holder) {
        this.renewLease(this.getLease(holder));
    }

    synchronized void renewLease(Lease lease) {
        if (lease != null) {
            lease.renew();
        }
    }

    synchronized void renewAllLeases() {
        for (Lease l : this.leases.values()) {
            this.renewLease(l);
        }
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    private synchronized Collection<Lease> getExpiredCandidateLeases() {
        long now = Time.monotonicNow();
        HashSet<Lease> expired = new HashSet<Lease>();
        for (Lease lease : this.leases.values()) {
            if (!lease.expiredHardLimit(now)) continue;
            expired.add(lease);
        }
        return expired;
    }

    @VisibleForTesting
    synchronized boolean checkLeases() {
        return this.checkLeases(this.getExpiredCandidateLeases());
    }

    private synchronized boolean checkLeases(Collection<Lease> leasesToCheck) {
        boolean needSync = false;
        assert (this.fsnamesystem.hasWriteLock());
        long start = Time.monotonicNow();
        for (Lease leaseToCheck : leasesToCheck) {
            if (this.isMaxLockHoldToReleaseLease(start)) break;
            if (!leaseToCheck.expiredHardLimit(Time.monotonicNow())) continue;
            LOG.info("{} has expired hard limit", (Object)leaseToCheck);
            ArrayList<Long> removing = new ArrayList<Long>();
            Collection files = leaseToCheck.getFiles();
            Long[] leaseINodeIds = files.toArray(new Long[files.size()]);
            FSDirectory fsd = this.fsnamesystem.getFSDirectory();
            String p = null;
            String newHolder = this.getInternalLeaseHolder();
            for (Long id : leaseINodeIds) {
                try {
                    INodesInPath iip = INodesInPath.fromINode(fsd.getInode(id));
                    p = iip.getPath();
                    if (!p.startsWith("/")) {
                        throw new IOException("Invalid path in the lease " + p);
                    }
                    INodeFile lastINode = iip.getLastINode().asFile();
                    if (this.fsnamesystem.isFileDeleted(lastINode)) {
                        this.removeLease(lastINode.getId());
                        continue;
                    }
                    boolean completed = false;
                    try {
                        completed = this.fsnamesystem.internalReleaseLease(leaseToCheck, p, iip, newHolder);
                    }
                    catch (IOException e) {
                        LOG.warn("Cannot release the path {} in the lease {}. It will be retried.", new Object[]{p, leaseToCheck, e});
                        continue;
                    }
                    if (LOG.isDebugEnabled()) {
                        if (completed) {
                            LOG.debug("Lease recovery for inode {} is complete. File closed.", (Object)id);
                        } else {
                            LOG.debug("Started block recovery {} lease {}", (Object)p, (Object)leaseToCheck);
                        }
                    }
                    if (!needSync && !completed) {
                        needSync = true;
                    }
                }
                catch (IOException e) {
                    LOG.warn("Removing lease with an invalid path: {},{}", new Object[]{p, leaseToCheck, e});
                    removing.add(id);
                }
                if (!this.isMaxLockHoldToReleaseLease(start)) continue;
                LOG.debug("Breaking out of checkLeases after {} ms.", (Object)this.fsnamesystem.getMaxLockHoldToReleaseLeaseMs());
                break;
            }
            for (Long id : removing) {
                this.removeLease(leaseToCheck, id);
            }
        }
        return needSync;
    }

    private boolean isMaxLockHoldToReleaseLease(long start) {
        return Time.monotonicNow() - start > this.fsnamesystem.getMaxLockHoldToReleaseLeaseMs();
    }

    public synchronized String toString() {
        return this.getClass().getSimpleName() + "= {\n leases=" + this.leases + "\n leasesById=" + this.leasesById + "\n}";
    }

    void startMonitor() {
        Preconditions.checkState((this.lmthread == null ? 1 : 0) != 0, (Object)"Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon((Runnable)new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn("Encountered exception ", (Throwable)ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    public void triggerMonitorCheckNow() {
        Preconditions.checkState((this.lmthread != null ? 1 : 0) != 0, (Object)"Lease monitor is not running");
        this.lmthread.interrupt();
    }

    @VisibleForTesting
    public void runLeaseChecks() {
        this.checkLeases();
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();

        Monitor() {
        }

        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                boolean needSync = false;
                try {
                    Thread.sleep(LeaseManager.this.fsnamesystem.getLeaseRecheckIntervalMs());
                    Collection candidates = LeaseManager.this.getExpiredCandidateLeases();
                    if (candidates.isEmpty()) continue;
                    LeaseManager.this.fsnamesystem.writeLockInterruptibly();
                    try {
                        if (LeaseManager.this.fsnamesystem.isInSafeMode()) continue;
                        needSync = LeaseManager.this.checkLeases(candidates);
                    }
                    finally {
                        LeaseManager.this.fsnamesystem.writeUnlock("leaseManager");
                        if (!needSync) continue;
                        LeaseManager.this.fsnamesystem.getEditLog().logSync();
                    }
                }
                catch (InterruptedException ie) {
                    LOG.debug("{} is interrupted", (Object)this.name, (Object)ie);
                }
                catch (Throwable e) {
                    LOG.warn("Unexpected throwable: ", e);
                }
            }
        }
    }

    class Lease {
        private final String holder;
        private long lastUpdate;
        private final HashSet<Long> files = new HashSet();

        private Lease(String h) {
            this.holder = h;
            this.renew();
        }

        private void renew() {
            this.lastUpdate = Time.monotonicNow();
        }

        public boolean expiredHardLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredHardLimit(long now) {
            return now - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredSoftLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.softLimit;
        }

        boolean hasFiles() {
            return !this.files.isEmpty();
        }

        boolean removeFile(long inodeId) {
            return this.files.remove(inodeId);
        }

        public String toString() {
            return "[Lease.  Holder: " + this.holder + ", pending creates: " + this.files.size() + "]";
        }

        public int hashCode() {
            return this.holder.hashCode();
        }

        private Collection<Long> getFiles() {
            return Collections.unmodifiableCollection(this.files);
        }

        String getHolder() {
            return this.holder;
        }

        @VisibleForTesting
        long getLastUpdate() {
            return this.lastUpdate;
        }
    }
}

