/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import com.obs.services.ObsClient;
import com.obs.services.exception.ObsException;
import com.obs.services.model.AbortMultipartUploadRequest;
import com.obs.services.model.DeleteObjectsRequest;
import com.obs.services.model.DeleteObjectsResult;
import com.obs.services.model.KeyAndVersion;
import com.obs.services.model.ListMultipartUploadsRequest;
import com.obs.services.model.ListObjectsRequest;
import com.obs.services.model.MultipartUpload;
import com.obs.services.model.MultipartUploadListing;
import com.obs.services.model.ObjectListing;
import com.obs.services.model.ObjectMetadata;
import com.obs.services.model.ObsObject;
import com.obs.services.model.PutObjectRequest;
import com.obs.services.model.PutObjectResult;
import com.obs.services.model.UploadPartRequest;
import com.obs.services.model.UploadPartResult;
import com.obs.services.model.fs.FSStatusEnum;
import com.obs.services.model.fs.GetAttributeRequest;
import com.obs.services.model.fs.GetBucketFSStatusRequest;
import com.obs.services.model.fs.GetBucketFSStatusResult;
import com.obs.services.model.fs.ObsFSAttribute;
import com.obs.services.model.fs.WriteFileRequest;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.obs.OBSFileStatus;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.obs.OBSFsDFSListing;
import org.apache.hadoop.fs.obs.OBSIOException;
import org.apache.hadoop.fs.obs.OBSListing;
import org.apache.hadoop.fs.obs.OBSLoginHelper;
import org.apache.hadoop.fs.obs.OBSObjectBucketUtils;
import org.apache.hadoop.fs.obs.OBSPosixBucketUtils;
import org.apache.hadoop.security.ProviderUtils;
import org.apache.hadoop.util.Lists;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class OBSCommonUtils {
    private static final Logger LOG = LoggerFactory.getLogger(OBSCommonUtils.class);
    static final int MOVED_PERMANENTLY_CODE = 301;
    static final int UNAUTHORIZED_CODE = 401;
    static final int FORBIDDEN_CODE = 403;
    static final int NOT_FOUND_CODE = 404;
    static final int CONFLICT_CODE = 409;
    static final int GONE_CODE = 410;
    static final int EOF_CODE = 416;
    static final String CREDENTIAL_PROVIDER_PATH = "hadoop.security.credential.provider.path";
    static final int MAX_RETRY_TIME = 3;
    static final int DELAY_TIME = 10;
    static final int MAX_KEYS_FOR_CHECK_FOLDER_EMPTY = 3;
    static final int BYTE_TO_INT_MASK = 255;

    private OBSCommonUtils() {
    }

    static boolean getBucketFsStatus(ObsClient obs, String bucketName) throws FileNotFoundException, IOException {
        try {
            GetBucketFSStatusRequest getBucketFsStatusRequest = new GetBucketFSStatusRequest();
            getBucketFsStatusRequest.setBucketName(bucketName);
            GetBucketFSStatusResult getBucketFsStatusResult = obs.getBucketFSStatus(getBucketFsStatusRequest);
            FSStatusEnum fsStatus = getBucketFsStatusResult.getStatus();
            return fsStatus == FSStatusEnum.ENABLED;
        }
        catch (ObsException e) {
            LOG.error(e.toString());
            throw OBSCommonUtils.translateException("getBucketFsStatus", bucketName, e);
        }
    }

    static String pathToKey(OBSFileSystem owner, Path path) {
        Path absolutePath = path;
        if (!path.isAbsolute()) {
            absolutePath = new Path(owner.getWorkingDirectory(), path);
        }
        if (absolutePath.toUri().getScheme() != null && absolutePath.toUri().getPath().isEmpty()) {
            return "";
        }
        return absolutePath.toUri().getPath().substring(1);
    }

    static String maybeAddTrailingSlash(String key) {
        if (!StringUtils.isEmpty((CharSequence)key) && !key.endsWith("/")) {
            return key + '/';
        }
        return key;
    }

    static Path keyToPath(String key) {
        return new Path("/" + key);
    }

    static Path keyToQualifiedPath(OBSFileSystem owner, String key) {
        return OBSCommonUtils.qualify(owner, OBSCommonUtils.keyToPath(key));
    }

    static Path qualify(OBSFileSystem owner, Path path) {
        return path.makeQualified(owner.getUri(), owner.getWorkingDirectory());
    }

    static String maybeDeleteBeginningSlash(String key) {
        return !StringUtils.isEmpty((CharSequence)key) && key.startsWith("/") ? key.substring(1) : key;
    }

    static String maybeAddBeginningSlash(String key) {
        return !StringUtils.isEmpty((CharSequence)key) && !key.startsWith("/") ? "/" + key : key;
    }

    static IOException translateException(String operation, String path, ObsException exception) {
        IOException ioe;
        String message = String.format("%s%s: status [%d] - request id [%s] - error code [%s] - error message [%s] - trace :%s ", new Object[]{operation, path != null ? " on " + path : "", exception.getResponseCode(), exception.getErrorRequestId(), exception.getErrorCode(), exception.getErrorMessage(), exception});
        int status = exception.getResponseCode();
        switch (status) {
            case 301: {
                message = String.format("Received permanent redirect response, status [%d] - request id [%s] - error code [%s] - message [%s]", exception.getResponseCode(), exception.getErrorRequestId(), exception.getErrorCode(), exception.getErrorMessage());
                ioe = new OBSIOException(message, exception);
                break;
            }
            case 401: 
            case 403: {
                ioe = new AccessDeniedException(path, null, message);
                ioe.initCause(exception);
                break;
            }
            case 404: 
            case 410: {
                ioe = new FileNotFoundException(message);
                ioe.initCause(exception);
                break;
            }
            case 416: {
                ioe = new EOFException(message);
                break;
            }
            default: {
                ioe = new OBSIOException(message, exception);
            }
        }
        return ioe;
    }

    static void blockRootDelete(String bucket, String key) throws InvalidRequestException {
        if (key.isEmpty() || "/".equals(key)) {
            throw new InvalidRequestException("Bucket " + bucket + " cannot be deleted");
        }
    }

    static void deleteObject(OBSFileSystem owner, String key) throws IOException {
        OBSCommonUtils.blockRootDelete(owner.getBucket(), key);
        ObsException lastException = null;
        for (int retryTime = 1; retryTime <= 3; ++retryTime) {
            try {
                owner.getObsClient().deleteObject(owner.getBucket(), key);
                owner.getSchemeStatistics().incrementWriteOps(1);
                return;
            }
            catch (ObsException e) {
                lastException = e;
                LOG.warn("Delete path failed with [{}], retry time [{}] - request id [{}] - error code [{}] - error message [{}]", new Object[]{e.getResponseCode(), retryTime, e.getErrorRequestId(), e.getErrorCode(), e.getErrorMessage()});
                if (retryTime >= 3) continue;
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw OBSCommonUtils.translateException("delete", key, e);
                }
            }
        }
        throw OBSCommonUtils.translateException(String.format("retry max times [%s] delete failed", 3), key, lastException);
    }

    static void deleteObjects(OBSFileSystem owner, DeleteObjectsRequest deleteRequest) throws IOException {
        List errorResults;
        DeleteObjectsResult result;
        deleteRequest.setQuiet(true);
        try {
            result = owner.getObsClient().deleteObjects(deleteRequest);
            owner.getSchemeStatistics().incrementWriteOps(1);
        }
        catch (ObsException e) {
            LOG.warn("delete objects failed, request [{}], request id [{}] - error code [{}] - error message [{}]", new Object[]{deleteRequest, e.getErrorRequestId(), e.getErrorCode(), e.getErrorMessage()});
            for (KeyAndVersion keyAndVersion : deleteRequest.getKeyAndVersionsList()) {
                OBSCommonUtils.deleteObject(owner, keyAndVersion.getKey());
            }
            return;
        }
        if (result != null && !(errorResults = result.getErrorResults()).isEmpty()) {
            LOG.warn("bulk delete {} objects, {} failed, begin to delete one by one.", (Object)deleteRequest.getKeyAndVersionsList().size(), (Object)errorResults.size());
            for (DeleteObjectsResult.ErrorResult errorResult : errorResults) {
                OBSCommonUtils.deleteObject(owner, errorResult.getObjectKey());
            }
        }
    }

    static PutObjectRequest newPutObjectRequest(OBSFileSystem owner, String key, ObjectMetadata metadata, File srcfile) {
        Preconditions.checkNotNull((Object)srcfile);
        PutObjectRequest putObjectRequest = new PutObjectRequest(owner.getBucket(), key, srcfile);
        putObjectRequest.setAcl(owner.getCannedACL());
        putObjectRequest.setMetadata(metadata);
        if (owner.getSse().isSseCEnable()) {
            putObjectRequest.setSseCHeader(owner.getSse().getSseCHeader());
        } else if (owner.getSse().isSseKmsEnable()) {
            putObjectRequest.setSseKmsHeader(owner.getSse().getSseKmsHeader());
        }
        return putObjectRequest;
    }

    static PutObjectRequest newPutObjectRequest(OBSFileSystem owner, String key, ObjectMetadata metadata, InputStream inputStream) {
        Preconditions.checkNotNull((Object)inputStream);
        PutObjectRequest putObjectRequest = new PutObjectRequest(owner.getBucket(), key, inputStream);
        putObjectRequest.setAcl(owner.getCannedACL());
        putObjectRequest.setMetadata(metadata);
        if (owner.getSse().isSseCEnable()) {
            putObjectRequest.setSseCHeader(owner.getSse().getSseCHeader());
        } else if (owner.getSse().isSseKmsEnable()) {
            putObjectRequest.setSseKmsHeader(owner.getSse().getSseKmsHeader());
        }
        return putObjectRequest;
    }

    static PutObjectResult putObjectDirect(OBSFileSystem owner, PutObjectRequest putObjectRequest) throws ObsException {
        long len = putObjectRequest.getFile() != null ? putObjectRequest.getFile().length() : putObjectRequest.getMetadata().getContentLength().longValue();
        PutObjectResult result = owner.getObsClient().putObject(putObjectRequest);
        owner.getSchemeStatistics().incrementWriteOps(1);
        owner.getSchemeStatistics().incrementBytesWritten(len);
        return result;
    }

    static UploadPartResult uploadPart(OBSFileSystem owner, UploadPartRequest request) throws ObsException {
        long len = request.getPartSize();
        UploadPartResult uploadPartResult = owner.getObsClient().uploadPart(request);
        owner.getSchemeStatistics().incrementWriteOps(1);
        owner.getSchemeStatistics().incrementBytesWritten(len);
        return uploadPartResult;
    }

    static void removeKeys(OBSFileSystem owner, List<KeyAndVersion> keysToDelete, boolean clearKeys, boolean checkRootDelete) throws IOException {
        if (keysToDelete.isEmpty()) {
            return;
        }
        if (checkRootDelete) {
            for (KeyAndVersion keyVersion : keysToDelete) {
                OBSCommonUtils.blockRootDelete(owner.getBucket(), keyVersion.getKey());
            }
        }
        if (!owner.isEnableMultiObjectDelete() || keysToDelete.size() < owner.getMultiDeleteThreshold()) {
            for (KeyAndVersion keyVersion : keysToDelete) {
                OBSCommonUtils.deleteObject(owner, keyVersion.getKey());
            }
        } else if (keysToDelete.size() <= owner.getMaxEntriesToDelete()) {
            DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(owner.getBucket());
            deleteObjectsRequest.setKeyAndVersions(keysToDelete.toArray(new KeyAndVersion[0]));
            OBSCommonUtils.deleteObjects(owner, deleteObjectsRequest);
        } else {
            ArrayList<KeyAndVersion> keys = new ArrayList<KeyAndVersion>(owner.getMaxEntriesToDelete());
            for (KeyAndVersion key : keysToDelete) {
                keys.add(key);
                if (keys.size() != owner.getMaxEntriesToDelete()) continue;
                OBSCommonUtils.removeKeys(owner, keys, true, false);
            }
            OBSCommonUtils.removeKeys(owner, keys, true, false);
        }
        if (clearKeys) {
            keysToDelete.clear();
        }
    }

    static IOException translateException(String operation, Path path, ObsException exception) {
        return OBSCommonUtils.translateException(operation, path.toString(), exception);
    }

    static FileStatus[] innerListStatus(OBSFileSystem owner, Path f, boolean recursive) throws FileNotFoundException, IOException, ObsException {
        Path path = OBSCommonUtils.qualify(owner, f);
        String key = OBSCommonUtils.pathToKey(owner, path);
        FileStatus fileStatus = owner.getFileStatus(path);
        if (fileStatus.isDirectory()) {
            key = OBSCommonUtils.maybeAddTrailingSlash(key);
            String delimiter = recursive ? null : "/";
            ListObjectsRequest request = OBSCommonUtils.createListObjectsRequest(owner, key, delimiter);
            LOG.debug("listStatus: doing listObjects for directory {} - recursive {}", (Object)f, (Object)recursive);
            OBSListing.FileStatusListingIterator files = owner.getObsListing().createFileStatusListingIterator(path, request, OBSListing.ACCEPT_ALL, new OBSListing.AcceptAllButSelfAndS3nDirs(path));
            ArrayList<FileStatus> result = new ArrayList<FileStatus>(files.getBatchSize());
            while (files.hasNext()) {
                result.add(files.next());
            }
            return result.toArray(new FileStatus[0]);
        }
        LOG.debug("Adding: rd (not a dir): {}", (Object)path);
        FileStatus[] stats = new FileStatus[]{fileStatus};
        return stats;
    }

    static ListObjectsRequest createListObjectsRequest(OBSFileSystem owner, String key, String delimiter) {
        return OBSCommonUtils.createListObjectsRequest(owner, key, delimiter, -1);
    }

    static ListObjectsRequest createListObjectsRequest(OBSFileSystem owner, String key, String delimiter, int maxKeyNum) {
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(owner.getBucket());
        if (maxKeyNum > 0 && maxKeyNum < owner.getMaxKeys()) {
            request.setMaxKeys(maxKeyNum);
        } else {
            request.setMaxKeys(owner.getMaxKeys());
        }
        request.setPrefix(key);
        if (delimiter != null) {
            request.setDelimiter(delimiter);
        }
        return request;
    }

    static boolean rejectRootDirectoryDelete(String bucket, boolean isEmptyDir, boolean recursive) throws IOException {
        LOG.info("obs delete the {} root directory of {}", (Object)bucket, (Object)recursive);
        if (isEmptyDir) {
            return true;
        }
        if (recursive) {
            return false;
        }
        throw new PathIOException(bucket, "Cannot delete root path");
    }

    static boolean innerMkdirs(OBSFileSystem owner, Path path) throws IOException, FileAlreadyExistsException, ObsException {
        LOG.debug("Making directory: {}", (Object)path);
        try {
            FileStatus fileStatus = owner.getFileStatus(path);
            if (fileStatus.isDirectory()) {
                return true;
            }
            throw new FileAlreadyExistsException("Path is a file: " + path);
        }
        catch (FileNotFoundException e) {
            Path fPart = path.getParent();
            do {
                try {
                    FileStatus fileStatus = owner.getFileStatus(fPart);
                    if (fileStatus.isDirectory()) break;
                    if (!fileStatus.isFile()) continue;
                    throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s' since it is a file.", fPart));
                }
                catch (FileNotFoundException fnfe) {
                    LOG.debug("file {} not fount, but ignore.", (Object)path);
                }
            } while ((fPart = fPart.getParent()) != null);
            String key = OBSCommonUtils.pathToKey(owner, path);
            if (owner.isFsBucket()) {
                OBSPosixBucketUtils.fsCreateFolder(owner, key);
            } else {
                OBSObjectBucketUtils.createFakeDirectory(owner, key);
            }
            return true;
        }
    }

    static ObjectListing listObjects(OBSFileSystem owner, ListObjectsRequest request) throws IOException {
        if (request.getDelimiter() == null && request.getMarker() == null && owner.isFsBucket() && owner.isObsClientDFSListEnable()) {
            return OBSFsDFSListing.fsDFSListObjects(owner, request);
        }
        return OBSCommonUtils.commonListObjects(owner, request);
    }

    static ObjectListing commonListObjects(OBSFileSystem owner, ListObjectsRequest request) {
        for (int retryTime = 1; retryTime < 3; ++retryTime) {
            try {
                owner.getSchemeStatistics().incrementReadOps(1);
                return owner.getObsClient().listObjects(request);
            }
            catch (ObsException e) {
                LOG.warn("Failed to commonListObjects for request[{}], retry time [{}], due to exception[{}]", new Object[]{request, retryTime, e});
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.error("Failed to commonListObjects for request[{}], retry time [{}], due to exception[{}]", new Object[]{request, retryTime, e});
                    throw e;
                }
            }
        }
        owner.getSchemeStatistics().incrementReadOps(1);
        return owner.getObsClient().listObjects(request);
    }

    static ObjectListing continueListObjects(OBSFileSystem owner, ObjectListing objects) throws IOException {
        if (objects.getDelimiter() == null && owner.isFsBucket() && owner.isObsClientDFSListEnable()) {
            return OBSFsDFSListing.fsDFSContinueListObjects(owner, (OBSFsDFSListing)objects);
        }
        return OBSCommonUtils.commonContinueListObjects(owner, objects);
    }

    private static ObjectListing commonContinueListObjects(OBSFileSystem owner, ObjectListing objects) {
        String delimiter = objects.getDelimiter();
        int maxKeyNum = objects.getMaxKeys();
        ListObjectsRequest request = new ListObjectsRequest();
        request.setMarker(objects.getNextMarker());
        request.setBucketName(owner.getBucket());
        request.setPrefix(objects.getPrefix());
        if (maxKeyNum > 0 && maxKeyNum < owner.getMaxKeys()) {
            request.setMaxKeys(maxKeyNum);
        } else {
            request.setMaxKeys(owner.getMaxKeys());
        }
        if (delimiter != null) {
            request.setDelimiter(delimiter);
        }
        return OBSCommonUtils.commonContinueListObjects(owner, request);
    }

    static ObjectListing commonContinueListObjects(OBSFileSystem owner, ListObjectsRequest request) {
        for (int retryTime = 1; retryTime < 3; ++retryTime) {
            try {
                owner.getSchemeStatistics().incrementReadOps(1);
                return owner.getObsClient().listObjects(request);
            }
            catch (ObsException e) {
                LOG.warn("Continue list objects failed for request[{}], retry time[{}], due to exception[{}]", new Object[]{request, retryTime, e});
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.error("Continue list objects failed for request[{}], retry time[{}], due to exception[{}]", new Object[]{request, retryTime, e});
                    throw e;
                }
            }
        }
        owner.getSchemeStatistics().incrementReadOps(1);
        return owner.getObsClient().listObjects(request);
    }

    public static boolean objectRepresentsDirectory(String name, long size) {
        return !name.isEmpty() && name.charAt(name.length() - 1) == '/' && size == 0L;
    }

    public static long dateToLong(Date date) {
        if (date == null) {
            return 0L;
        }
        return date.getTime() / 1000L * 1000L;
    }

    static boolean isFolderEmpty(OBSFileSystem owner, String key) throws FileNotFoundException, ObsException {
        for (int retryTime = 1; retryTime < 3; ++retryTime) {
            try {
                return OBSCommonUtils.innerIsFolderEmpty(owner, key);
            }
            catch (ObsException e) {
                LOG.warn("Failed to check empty folder for [{}], retry time [{}], exception [{}]", new Object[]{key, retryTime, e});
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        return OBSCommonUtils.innerIsFolderEmpty(owner, key);
    }

    private static boolean isFolderEmpty(String key, ObjectListing objects) {
        int count = objects.getObjects().size();
        if (count >= 2) {
            return false;
        }
        if (count == 1 && !((ObsObject)objects.getObjects().get(0)).getObjectKey().equals(key)) {
            return false;
        }
        count = objects.getCommonPrefixes().size();
        if (count >= 2) {
            return false;
        }
        return count != 1 || ((String)objects.getCommonPrefixes().get(0)).equals(key);
    }

    static boolean innerIsFolderEmpty(OBSFileSystem owner, String key) throws FileNotFoundException, ObsException {
        String obsKey = OBSCommonUtils.maybeAddTrailingSlash(key);
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(owner.getBucket());
        request.setPrefix(obsKey);
        request.setDelimiter("/");
        request.setMaxKeys(3);
        owner.getSchemeStatistics().incrementReadOps(1);
        ObjectListing objects = owner.getObsClient().listObjects(request);
        if (!objects.getCommonPrefixes().isEmpty() || !objects.getObjects().isEmpty()) {
            if (OBSCommonUtils.isFolderEmpty(obsKey, objects)) {
                LOG.debug("Found empty directory {}", (Object)obsKey);
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Found path as directory (with /): {}/{}", (Object)objects.getCommonPrefixes().size(), (Object)objects.getObjects().size());
                for (ObsObject summary : objects.getObjects()) {
                    LOG.debug("Summary: {} {}", (Object)summary.getObjectKey(), (Object)summary.getMetadata().getContentLength());
                }
                for (String prefix : objects.getCommonPrefixes()) {
                    LOG.debug("Prefix: {}", (Object)prefix);
                }
            }
            LOG.debug("Found non-empty directory {}", (Object)obsKey);
            return false;
        }
        if (obsKey.isEmpty()) {
            LOG.debug("Found root directory");
            return true;
        }
        if (owner.isFsBucket()) {
            LOG.debug("Found empty directory {}", (Object)obsKey);
            return true;
        }
        LOG.debug("Not Found: {}", (Object)obsKey);
        throw new FileNotFoundException("No such file or directory: " + obsKey);
    }

    static LocatedFileStatus toLocatedFileStatus(OBSFileSystem owner, FileStatus status) throws IOException {
        return new LocatedFileStatus(status, status.isFile() ? owner.getFileBlockLocations(status, 0L, status.getLen()) : null);
    }

    static WriteFileRequest newAppendFileRequest(OBSFileSystem owner, String key, long recordPosition, File tmpFile) throws IOException {
        ObsFSAttribute obsFsAttribute;
        Preconditions.checkNotNull((Object)key);
        Preconditions.checkNotNull((Object)tmpFile);
        try {
            GetAttributeRequest getAttributeReq = new GetAttributeRequest(owner.getBucket(), key);
            obsFsAttribute = owner.getObsClient().getAttribute(getAttributeReq);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("GetAttributeRequest", key, e);
        }
        long appendPosition = Math.max(recordPosition, obsFsAttribute.getContentLength());
        if (recordPosition != obsFsAttribute.getContentLength()) {
            LOG.warn("append url[{}] position[{}], file contentLength[{}] not equal to recordPosition[{}].", new Object[]{key, appendPosition, obsFsAttribute.getContentLength(), recordPosition});
        }
        WriteFileRequest writeFileReq = new WriteFileRequest(owner.getBucket(), key, tmpFile, appendPosition);
        writeFileReq.setAcl(owner.getCannedACL());
        return writeFileReq;
    }

    static WriteFileRequest newAppendFileRequest(OBSFileSystem owner, String key, long recordPosition, InputStream inputStream) throws IOException {
        ObsFSAttribute obsFsAttribute;
        Preconditions.checkNotNull((Object)key);
        Preconditions.checkNotNull((Object)inputStream);
        try {
            GetAttributeRequest getAttributeReq = new GetAttributeRequest(owner.getBucket(), key);
            obsFsAttribute = owner.getObsClient().getAttribute(getAttributeReq);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("GetAttributeRequest", key, e);
        }
        long appendPosition = Math.max(recordPosition, obsFsAttribute.getContentLength());
        if (recordPosition != obsFsAttribute.getContentLength()) {
            LOG.warn("append url[{}] position[{}], file contentLength[{}] not equal to recordPosition[{}].", new Object[]{key, appendPosition, obsFsAttribute.getContentLength(), recordPosition});
        }
        WriteFileRequest writeFileReq = new WriteFileRequest(owner.getBucket(), key, inputStream, appendPosition);
        writeFileReq.setAcl(owner.getCannedACL());
        return writeFileReq;
    }

    static void appendFile(OBSFileSystem owner, WriteFileRequest appendFileRequest) throws IOException {
        long len = 0L;
        if (appendFileRequest.getFile() != null) {
            len = appendFileRequest.getFile().length();
        }
        try {
            LOG.debug("Append file, key {} position {} size {}", new Object[]{appendFileRequest.getObjectKey(), appendFileRequest.getPosition(), len});
            owner.getObsClient().writeFile(appendFileRequest);
            owner.getSchemeStatistics().incrementWriteOps(1);
            owner.getSchemeStatistics().incrementBytesWritten(len);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("AppendFile", appendFileRequest.getObjectKey(), e);
        }
    }

    static void closeAll(Closeable ... closeables) {
        for (Closeable c : closeables) {
            if (c == null) continue;
            try {
                if (LOG != null) {
                    LOG.debug("Closing {}", (Object)c);
                }
                c.close();
            }
            catch (Exception e) {
                if (LOG == null || !LOG.isDebugEnabled()) continue;
                LOG.debug("Exception in closing {}", (Object)c, (Object)e);
            }
        }
    }

    static IOException extractException(String operation, String path, ExecutionException ee) {
        Throwable cause = ee.getCause();
        IOException ioe = cause instanceof ObsException ? OBSCommonUtils.translateException(operation, path, (ObsException)cause) : (cause instanceof IOException ? (IOException)cause : new IOException(operation + " failed: " + cause, cause));
        return ioe;
    }

    static OBSFileStatus createFileStatus(Path keyPath, ObsObject summary, long blockSize, String owner) {
        if (OBSCommonUtils.objectRepresentsDirectory(summary.getObjectKey(), summary.getMetadata().getContentLength())) {
            return new OBSFileStatus(keyPath, owner);
        }
        return new OBSFileStatus(summary.getMetadata().getContentLength(), OBSCommonUtils.dateToLong(summary.getMetadata().getLastModified()), keyPath, blockSize, owner);
    }

    static OBSLoginHelper.Login getOBSAccessKeys(URI name, Configuration conf) throws IOException {
        OBSLoginHelper.Login login = OBSLoginHelper.extractLoginDetailsWithWarnings(name);
        Configuration c = ProviderUtils.excludeIncompatibleCredentialProviders((Configuration)conf, OBSFileSystem.class);
        String accessKey = OBSCommonUtils.getPassword(c, "fs.obs.access.key", login.getUser());
        String secretKey = OBSCommonUtils.getPassword(c, "fs.obs.secret.key", login.getPassword());
        String sessionToken = OBSCommonUtils.getPassword(c, "fs.obs.session.token", login.getToken());
        return new OBSLoginHelper.Login(accessKey, secretKey, sessionToken);
    }

    private static String getPassword(Configuration conf, String key, String val) throws IOException {
        return StringUtils.isEmpty((CharSequence)val) ? OBSCommonUtils.lookupPassword(conf, key) : val;
    }

    private static String lookupPassword(Configuration conf, String key) throws IOException {
        try {
            char[] pass = conf.getPassword(key);
            return pass != null ? new String(pass).trim() : "";
        }
        catch (IOException ioe) {
            throw new IOException("Cannot find password option " + key, ioe);
        }
    }

    static String stringify(ObsObject summary) {
        return summary.getObjectKey() + " size=" + summary.getMetadata().getContentLength();
    }

    static int intOption(Configuration conf, String key, int defVal, int min) {
        int v = conf.getInt(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    static long longOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLong(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    static long longBytesOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLongBytes(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    public static long getMultipartSizeProperty(Configuration conf, String property, long defVal) {
        long partSize = conf.getLongBytes(property, defVal);
        if (partSize < 0x500000L) {
            LOG.warn("{} must be at least 5 MB; configured value is {}", (Object)property, (Object)partSize);
            partSize = 0x500000L;
        }
        return partSize;
    }

    static int ensureOutputParameterInRange(String name, long size) {
        if (size > Integer.MAX_VALUE) {
            LOG.warn("obs: {} capped to ~2.14GB (maximum allowed size with current output mechanism)", (Object)name);
            return Integer.MAX_VALUE;
        }
        return (int)size;
    }

    static Configuration propagateBucketOptions(Configuration source, String bucket) {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)bucket), (Object)"bucket");
        String bucketPrefix = "fs.obs.bucket." + bucket + '.';
        LOG.debug("Propagating entries under {}", (Object)bucketPrefix);
        Configuration dest = new Configuration(source);
        for (Map.Entry entry : source) {
            String key = (String)entry.getKey();
            String value = (String)entry.getValue();
            if (!key.startsWith(bucketPrefix) || bucketPrefix.equals(key)) continue;
            String stripped = key.substring(bucketPrefix.length());
            if (stripped.startsWith("bucket.") || "impl".equals(stripped)) {
                LOG.debug("Ignoring bucket option {}", (Object)key);
                continue;
            }
            String generic = "fs.obs." + stripped;
            LOG.debug("Updating {}", (Object)generic);
            dest.set(generic, value, key);
        }
        return dest;
    }

    static void patchSecurityCredentialProviders(Configuration conf) {
        Collection customCredentials = conf.getStringCollection("fs.obs.security.credential.provider.path");
        Collection hadoopCredentials = conf.getStringCollection(CREDENTIAL_PROVIDER_PATH);
        if (!customCredentials.isEmpty()) {
            ArrayList all = Lists.newArrayList((Iterable)customCredentials);
            all.addAll(hadoopCredentials);
            String joined = StringUtils.join((Iterable)all, (char)',');
            LOG.debug("Setting {} to {}", (Object)CREDENTIAL_PROVIDER_PATH, (Object)joined);
            conf.set(CREDENTIAL_PROVIDER_PATH, joined, "patch of fs.obs.security.credential.provider.path");
        }
    }

    static void verifyBucketExists(OBSFileSystem owner) throws FileNotFoundException, IOException {
        int retryTime = 1;
        while (true) {
            try {
                if (!owner.getObsClient().headBucket(owner.getBucket())) {
                    throw new FileNotFoundException("Bucket " + owner.getBucket() + " does not exist");
                }
                return;
            }
            catch (ObsException e) {
                LOG.warn("Failed to head bucket for [{}], retry time [{}], exception [{}]", new Object[]{owner.getBucket(), retryTime, OBSCommonUtils.translateException("doesBucketExist", owner.getBucket(), e)});
                if (3 == retryTime) {
                    throw OBSCommonUtils.translateException("doesBucketExist", owner.getBucket(), e);
                }
                try {
                    Thread.sleep(10L);
                }
                catch (InterruptedException ie) {
                    throw e;
                }
                ++retryTime;
                continue;
            }
            break;
        }
    }

    static void initMultipartUploads(OBSFileSystem owner, Configuration conf) throws IOException {
        boolean purgeExistingMultipart = conf.getBoolean("fs.obs.multipart.purge", false);
        long purgeExistingMultipartAge = OBSCommonUtils.longOption(conf, "fs.obs.multipart.purge.age", 86400L, 0L);
        if (!purgeExistingMultipart) {
            return;
        }
        Date purgeBefore = new Date(new Date().getTime() - purgeExistingMultipartAge * 1000L);
        try {
            ListMultipartUploadsRequest request = new ListMultipartUploadsRequest(owner.getBucket());
            while (true) {
                MultipartUploadListing uploadListing = owner.getObsClient().listMultipartUploads(request);
                for (MultipartUpload upload : uploadListing.getMultipartTaskList()) {
                    if (upload.getInitiatedDate().compareTo(purgeBefore) >= 0) continue;
                    owner.getObsClient().abortMultipartUpload(new AbortMultipartUploadRequest(owner.getBucket(), upload.getObjectKey(), upload.getUploadId()));
                }
                if (uploadListing.isTruncated()) {
                    request.setUploadIdMarker(uploadListing.getNextUploadIdMarker());
                    request.setKeyMarker(uploadListing.getNextKeyMarker());
                    continue;
                }
                break;
            }
        }
        catch (ObsException e) {
            if (e.getResponseCode() == 403) {
                LOG.debug("Failed to purging multipart uploads against {}, FS may be read only", (Object)owner.getBucket(), (Object)e);
            }
            throw OBSCommonUtils.translateException("purging multipart uploads", owner.getBucket(), e);
        }
    }

    static void shutdownAll(ExecutorService ... executors) {
        for (ExecutorService exe : executors) {
            if (exe == null) continue;
            try {
                if (LOG != null) {
                    LOG.debug("Shutdown {}", (Object)exe);
                }
                exe.shutdown();
            }
            catch (Exception e) {
                if (LOG == null || !LOG.isDebugEnabled()) continue;
                LOG.debug("Exception in shutdown {}", (Object)exe, (Object)e);
            }
        }
    }
}

