/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.snapshot;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.classification.InterfaceStability;
import org.apache.hadoop.hbase.snapshot.SnapshotDescriptionUtils;
import org.apache.hadoop.hbase.util.FSUtils;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class SnapshotFileCache
implements Stoppable {
    private static final Log LOG = LogFactory.getLog(SnapshotFileCache.class);
    private volatile boolean stop = false;
    private final FileSystem fs;
    private final SnapshotFileInspector fileInspector;
    private final Path snapshotDir;
    private final Set<String> cache = new HashSet<String>();
    private final Map<String, SnapshotDirectoryInfo> snapshots = new HashMap<String, SnapshotDirectoryInfo>();
    private final Timer refreshTimer;
    private long lastModifiedTime = Long.MIN_VALUE;

    public SnapshotFileCache(Configuration conf, long cacheRefreshPeriod, String refreshThreadName, SnapshotFileInspector inspectSnapshotFiles) throws IOException {
        this(FSUtils.getCurrentFileSystem(conf), FSUtils.getRootDir(conf), 0L, cacheRefreshPeriod, refreshThreadName, inspectSnapshotFiles);
    }

    public SnapshotFileCache(FileSystem fs, Path rootDir, long cacheRefreshPeriod, long cacheRefreshDelay, String refreshThreadName, SnapshotFileInspector inspectSnapshotFiles) {
        this.fs = fs;
        this.fileInspector = inspectSnapshotFiles;
        this.snapshotDir = SnapshotDescriptionUtils.getSnapshotsDir(rootDir);
        this.refreshTimer = new Timer(refreshThreadName, true);
        this.refreshTimer.scheduleAtFixedRate((TimerTask)new RefreshCacheTask(), cacheRefreshDelay, cacheRefreshPeriod);
    }

    public void triggerCacheRefreshForTesting() {
        try {
            this.refreshCache();
        }
        catch (IOException e) {
            LOG.warn((Object)"Failed to refresh snapshot hfile cache!", (Throwable)e);
        }
        LOG.debug((Object)("Current cache:" + this.cache));
    }

    public synchronized Iterable<FileStatus> getUnreferencedFiles(Iterable<FileStatus> files) throws IOException {
        ArrayList unReferencedFiles = Lists.newArrayList();
        List<String> snapshotsInProgress = null;
        boolean refreshed = false;
        for (FileStatus file : files) {
            String fileName = file.getPath().getName();
            if (!refreshed && !this.cache.contains(fileName)) {
                this.refreshCache();
                refreshed = true;
            }
            if (this.cache.contains(fileName)) continue;
            if (snapshotsInProgress == null) {
                snapshotsInProgress = this.getSnapshotsInProgress();
            }
            if (snapshotsInProgress.contains(fileName)) continue;
            unReferencedFiles.add(file);
        }
        return unReferencedFiles;
    }

    private synchronized void refreshCache() throws IOException {
        long lastTimestamp = Long.MAX_VALUE;
        boolean hasChanges = false;
        try {
            FileStatus dirStatus = this.fs.getFileStatus(this.snapshotDir);
            lastTimestamp = dirStatus.getModificationTime();
            hasChanges |= lastTimestamp >= this.lastModifiedTime;
        }
        catch (FileNotFoundException e) {
            if (this.cache.size() > 0) {
                LOG.error((Object)("Snapshot directory: " + this.snapshotDir + " doesn't exist"));
            }
            return;
        }
        try {
            FileStatus[] tmpSnapshots;
            Path snapshotTmpDir = new Path(this.snapshotDir, ".tmp");
            FileStatus tempDirStatus = this.fs.getFileStatus(snapshotTmpDir);
            lastTimestamp = Math.min(lastTimestamp, tempDirStatus.getModificationTime());
            if (!(hasChanges |= lastTimestamp >= this.lastModifiedTime) && (tmpSnapshots = FSUtils.listStatus(this.fs, this.snapshotDir)) != null) {
                FileStatus[] fileStatusArray = tmpSnapshots;
                int n = fileStatusArray.length;
                for (int i = 0; i < n; ++i) {
                    FileStatus dirStatus = fileStatusArray[i];
                    lastTimestamp = Math.min(lastTimestamp, dirStatus.getModificationTime());
                }
                hasChanges |= lastTimestamp >= this.lastModifiedTime;
            }
        }
        catch (FileNotFoundException snapshotTmpDir) {
            // empty catch block
        }
        if (!hasChanges) {
            return;
        }
        this.lastModifiedTime = lastTimestamp;
        this.cache.clear();
        HashMap<String, SnapshotDirectoryInfo> known = new HashMap<String, SnapshotDirectoryInfo>();
        FileStatus[] snapshots = FSUtils.listStatus(this.fs, this.snapshotDir);
        if (snapshots == null) {
            if (LOG.isDebugEnabled() && this.snapshots.size() > 0) {
                LOG.debug((Object)"No snapshots on-disk, cache empty");
            }
            this.snapshots.clear();
            return;
        }
        for (FileStatus snapshot : snapshots) {
            String name = snapshot.getPath().getName();
            if (name.equals(".tmp")) continue;
            SnapshotDirectoryInfo files = this.snapshots.remove(name);
            if (files == null || files.hasBeenModified(snapshot.getModificationTime())) {
                Collection<String> storedFiles = this.fileInspector.filesUnderSnapshot(snapshot.getPath());
                files = new SnapshotDirectoryInfo(snapshot.getModificationTime(), storedFiles);
            }
            this.cache.addAll(files.getFiles());
            known.put(name, files);
        }
        this.snapshots.clear();
        this.snapshots.putAll(known);
    }

    @VisibleForTesting
    List<String> getSnapshotsInProgress() throws IOException {
        ArrayList snapshotInProgress = Lists.newArrayList();
        Path snapshotTmpDir = new Path(this.snapshotDir, ".tmp");
        FileStatus[] running = FSUtils.listStatus(this.fs, snapshotTmpDir);
        if (running != null) {
            for (FileStatus run : running) {
                snapshotInProgress.addAll(this.fileInspector.filesUnderSnapshot(run.getPath()));
            }
        }
        return snapshotInProgress;
    }

    public void stop(String why) {
        if (!this.stop) {
            this.stop = true;
            this.refreshTimer.cancel();
        }
    }

    public boolean isStopped() {
        return this.stop;
    }

    private static class SnapshotDirectoryInfo {
        long lastModified;
        Collection<String> files;

        public SnapshotDirectoryInfo(long mtime, Collection<String> files) {
            this.lastModified = mtime;
            this.files = files;
        }

        public Collection<String> getFiles() {
            return this.files;
        }

        public boolean hasBeenModified(long mtime) {
            return this.lastModified < mtime;
        }
    }

    public class RefreshCacheTask
    extends TimerTask {
        @Override
        public void run() {
            try {
                SnapshotFileCache.this.refreshCache();
            }
            catch (IOException e) {
                LOG.warn((Object)"Failed to refresh snapshot hfile cache!", (Throwable)e);
            }
        }
    }

    static interface SnapshotFileInspector {
        public Collection<String> filesUnderSnapshot(Path var1) throws IOException;
    }
}

