/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableDescriptor;
import org.apache.hadoop.hbase.TableDescriptors;
import org.apache.hadoop.hbase.TableInfoMissingException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;

@InterfaceAudience.Private
public class FSTableDescriptors
implements TableDescriptors {
    private static final Log LOG = LogFactory.getLog(FSTableDescriptors.class);
    private final FileSystem fs;
    private final Path rootdir;
    private final boolean fsreadonly;
    private volatile boolean usecache;
    private volatile boolean fsvisited;
    @VisibleForTesting
    long cachehits = 0L;
    @VisibleForTesting
    long invocations = 0L;
    static final String TABLEINFO_FILE_PREFIX = ".tableinfo";
    static final String TABLEINFO_DIR = ".tabledesc";
    static final String TMP_DIR = ".tmp";
    private final Map<TableName, TableDescriptor> cache = new ConcurrentHashMap<TableName, TableDescriptor>();
    private final HTableDescriptor metaTableDescritor;
    @VisibleForTesting
    static final Comparator<FileStatus> TABLEINFO_FILESTATUS_COMPARATOR = new Comparator<FileStatus>(){

        @Override
        public int compare(FileStatus left, FileStatus right) {
            return right.compareTo((Object)left);
        }
    };
    private static final PathFilter TABLEINFO_PATHFILTER = new PathFilter(){

        public boolean accept(Path p) {
            return p.getName().startsWith(FSTableDescriptors.TABLEINFO_FILE_PREFIX);
        }
    };
    @VisibleForTesting
    static final int WIDTH_OF_SEQUENCE_ID = 10;
    private static final Pattern TABLEINFO_FILE_REGEX = Pattern.compile(".tableinfo(\\.([0-9]{10}))?$");

    public FSTableDescriptors(Configuration conf) throws IOException {
        this(conf, FSUtils.getCurrentFileSystem(conf), FSUtils.getRootDir(conf));
    }

    public FSTableDescriptors(Configuration conf, FileSystem fs, Path rootdir) throws IOException {
        this(conf, fs, rootdir, false, true);
    }

    public FSTableDescriptors(Configuration conf, FileSystem fs, Path rootdir, boolean fsreadonly, boolean usecache) throws IOException {
        this.fs = fs;
        this.rootdir = rootdir;
        this.fsreadonly = fsreadonly;
        this.usecache = usecache;
        this.metaTableDescritor = TableDescriptor.metaTableDescriptor(conf);
    }

    @Override
    public void setCacheOn() throws IOException {
        this.cache.clear();
        this.usecache = true;
    }

    @Override
    public void setCacheOff() throws IOException {
        this.usecache = false;
        this.cache.clear();
    }

    @VisibleForTesting
    public boolean isUsecache() {
        return this.usecache;
    }

    @Override
    @Nullable
    public TableDescriptor getDescriptor(TableName tablename) throws IOException {
        TableDescriptor cachedtdm;
        ++this.invocations;
        if (TableName.META_TABLE_NAME.equals((Object)tablename)) {
            ++this.cachehits;
            return new TableDescriptor(this.metaTableDescritor);
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(tablename.getNameAsString())) {
            throw new IOException("No descriptor found for non table = " + tablename);
        }
        if (this.usecache && (cachedtdm = this.cache.get(tablename)) != null) {
            ++this.cachehits;
            return cachedtdm;
        }
        TableDescriptor tdmt = null;
        try {
            tdmt = FSTableDescriptors.getTableDescriptorFromFs(this.fs, this.rootdir, tablename, !this.fsreadonly);
        }
        catch (NullPointerException e) {
            LOG.debug((Object)("Exception during readTableDecriptor. Current table name = " + tablename), (Throwable)e);
        }
        catch (IOException ioe) {
            LOG.debug((Object)("Exception during readTableDecriptor. Current table name = " + tablename), (Throwable)ioe);
        }
        if (this.usecache && tdmt != null) {
            this.cache.put(tablename, tdmt);
        }
        return tdmt;
    }

    @Override
    public HTableDescriptor get(TableName tableName) throws IOException {
        if (TableName.META_TABLE_NAME.equals((Object)tableName)) {
            ++this.cachehits;
            return this.metaTableDescritor;
        }
        TableDescriptor descriptor = this.getDescriptor(tableName);
        return descriptor == null ? null : descriptor.getHTableDescriptor();
    }

    @Override
    public Map<String, TableDescriptor> getAllDescriptors() throws IOException {
        TreeMap<String, TableDescriptor> tds = new TreeMap<String, TableDescriptor>();
        if (this.fsvisited && this.usecache) {
            for (Map.Entry<TableName, TableDescriptor> entry : this.cache.entrySet()) {
                tds.put(entry.getKey().toString(), entry.getValue());
            }
            tds.put(this.metaTableDescritor.getNameAsString(), new TableDescriptor(this.metaTableDescritor));
        } else {
            LOG.debug((Object)"Fetching table descriptors from the filesystem.");
            boolean allvisited = true;
            for (Path d : FSUtils.getTableDirs(this.fs, this.rootdir)) {
                TableDescriptor htd = null;
                try {
                    htd = this.getDescriptor(FSUtils.getTableName(d));
                }
                catch (FileNotFoundException fnfe) {
                    LOG.warn((Object)"Trouble retrieving htd", (Throwable)fnfe);
                }
                if (htd == null) {
                    allvisited = false;
                    continue;
                }
                tds.put(htd.getHTableDescriptor().getTableName().getNameAsString(), htd);
                this.fsvisited = allvisited;
            }
        }
        return tds;
    }

    @Override
    public Map<String, HTableDescriptor> getAll() throws IOException {
        TreeMap<String, HTableDescriptor> htds = new TreeMap<String, HTableDescriptor>();
        Map<String, TableDescriptor> allDescriptors = this.getAllDescriptors();
        for (Map.Entry<String, TableDescriptor> entry : allDescriptors.entrySet()) {
            htds.put(entry.getKey(), entry.getValue().getHTableDescriptor());
        }
        return htds;
    }

    @Override
    public Map<String, HTableDescriptor> getByNamespace(String name) throws IOException {
        TreeMap<String, HTableDescriptor> htds = new TreeMap<String, HTableDescriptor>();
        List<Path> tableDirs = FSUtils.getLocalTableDirs(this.fs, FSUtils.getNamespaceDir(this.rootdir, name));
        for (Path d : tableDirs) {
            HTableDescriptor htd = null;
            try {
                htd = this.get(FSUtils.getTableName(d));
            }
            catch (FileNotFoundException fnfe) {
                LOG.warn((Object)"Trouble retrieving htd", (Throwable)fnfe);
            }
            if (htd == null) continue;
            htds.put(FSUtils.getTableName(d).getNameAsString(), htd);
        }
        return htds;
    }

    @Override
    public void add(TableDescriptor htd) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot add a table descriptor - in read only mode");
        }
        TableName tableName = htd.getHTableDescriptor().getTableName();
        if (TableName.META_TABLE_NAME.equals((Object)tableName)) {
            throw new NotImplementedException();
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(tableName.getNameAsString())) {
            throw new NotImplementedException("Cannot add a table descriptor for a reserved subdirectory name: " + htd.getHTableDescriptor().getNameAsString());
        }
        this.updateTableDescriptor(htd);
    }

    @Override
    public void add(HTableDescriptor htd) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot add a table descriptor - in read only mode");
        }
        TableName tableName = htd.getTableName();
        if (TableName.META_TABLE_NAME.equals((Object)tableName)) {
            throw new NotImplementedException();
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(tableName.getNameAsString())) {
            throw new NotImplementedException("Cannot add a table descriptor for a reserved subdirectory name: " + htd.getNameAsString());
        }
        TableDescriptor descriptor = this.getDescriptor(htd.getTableName());
        if (descriptor == null) {
            descriptor = new TableDescriptor(htd);
        } else {
            descriptor.setHTableDescriptor(htd);
        }
        this.updateTableDescriptor(descriptor);
    }

    @Override
    public HTableDescriptor remove(TableName tablename) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot remove a table descriptor - in read only mode");
        }
        Path tabledir = this.getTableDir(tablename);
        if (this.fs.exists(tabledir) && !this.fs.delete(tabledir, true)) {
            throw new IOException("Failed delete of " + tabledir.toString());
        }
        TableDescriptor descriptor = this.cache.remove(tablename);
        if (descriptor == null) {
            return null;
        }
        return descriptor.getHTableDescriptor();
    }

    public boolean isTableInfoExists(TableName tableName) throws IOException {
        return this.getTableInfoPath(tableName) != null;
    }

    private FileStatus getTableInfoPath(TableName tableName) throws IOException {
        Path tableDir = this.getTableDir(tableName);
        return this.getTableInfoPath(tableDir);
    }

    private FileStatus getTableInfoPath(Path tableDir) throws IOException {
        return FSTableDescriptors.getTableInfoPath(this.fs, tableDir, !this.fsreadonly);
    }

    public static FileStatus getTableInfoPath(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableInfoPath(fs, tableDir, false);
    }

    private static FileStatus getTableInfoPath(FileSystem fs, Path tableDir, boolean removeOldFiles) throws IOException {
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        return FSTableDescriptors.getCurrentTableInfoStatus(fs, tableInfoDir, removeOldFiles);
    }

    static FileStatus getCurrentTableInfoStatus(FileSystem fs, Path dir, boolean removeOldFiles) throws IOException {
        FileStatus[] status = FSUtils.listStatus(fs, dir, TABLEINFO_PATHFILTER);
        if (status == null || status.length < 1) {
            return null;
        }
        FileStatus mostCurrent = null;
        for (FileStatus file : status) {
            if (mostCurrent != null && TABLEINFO_FILESTATUS_COMPARATOR.compare(file, mostCurrent) >= 0) continue;
            mostCurrent = file;
        }
        if (removeOldFiles && status.length > 1) {
            for (FileStatus file : status) {
                Path path = file.getPath();
                if (file == mostCurrent) continue;
                if (!fs.delete(file.getPath(), false)) {
                    LOG.warn((Object)("Failed cleanup of " + path));
                    continue;
                }
                LOG.debug((Object)("Cleaned up old tableinfo file " + path));
            }
        }
        return mostCurrent;
    }

    @VisibleForTesting
    Path getTableDir(TableName tableName) {
        return FSUtils.getTableDir(this.rootdir, tableName);
    }

    private static String formatTableInfoSequenceId(int number) {
        byte[] b = new byte[10];
        int d = Math.abs(number);
        for (int i = b.length - 1; i >= 0; --i) {
            b[i] = (byte)(d % 10 + 48);
            d /= 10;
        }
        return Bytes.toString((byte[])b);
    }

    @VisibleForTesting
    static int getTableInfoSequenceId(Path p) {
        if (p == null) {
            return 0;
        }
        Matcher m = TABLEINFO_FILE_REGEX.matcher(p.getName());
        if (!m.matches()) {
            throw new IllegalArgumentException(p.toString());
        }
        String suffix = m.group(2);
        if (suffix == null || suffix.length() <= 0) {
            return 0;
        }
        return Integer.parseInt(m.group(2));
    }

    @VisibleForTesting
    static String getTableInfoFileName(int sequenceid) {
        return ".tableinfo." + FSTableDescriptors.formatTableInfoSequenceId(sequenceid);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException {
        Path tableDir = FSUtils.getTableDir(hbaseRootDir, tableName);
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName, boolean rewritePb) throws IOException {
        Path tableDir = FSUtils.getTableDir(hbaseRootDir, tableName);
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, rewritePb);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, false);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir, boolean rewritePb) throws IOException {
        FileStatus status = FSTableDescriptors.getTableInfoPath(fs, tableDir, false);
        if (status == null) {
            throw new TableInfoMissingException("No table descriptor file under " + tableDir);
        }
        return FSTableDescriptors.readTableDescriptor(fs, status, rewritePb);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static TableDescriptor readTableDescriptor(FileSystem fs, FileStatus status, boolean rewritePb) throws IOException {
        int len = Ints.checkedCast((long)status.getLen());
        byte[] content = new byte[len];
        try (FSDataInputStream fsDataInputStream = fs.open(status.getPath());){
            fsDataInputStream.readFully(content);
        }
        TableDescriptor td = null;
        try {
            td = TableDescriptor.parseFrom(content);
        }
        catch (DeserializationException e) {
            try {
                HTableDescriptor htd = HTableDescriptor.parseFrom((byte[])content);
                LOG.warn((Object)("Found old table descriptor, converting to new format for table " + htd.getTableName() + "; NOTE table will be in ENABLED state!"));
                td = new TableDescriptor(htd);
                if (rewritePb) {
                    FSTableDescriptors.rewriteTableDescriptor(fs, status, td);
                }
            }
            catch (DeserializationException e1) {
                throw new IOException("content=" + Bytes.toShort((byte[])content), e);
            }
        }
        if (rewritePb && !ProtobufUtil.isPBMagicPrefix((byte[])content)) {
            FSTableDescriptors.rewriteTableDescriptor(fs, status, td);
        }
        return td;
    }

    private static void rewriteTableDescriptor(FileSystem fs, FileStatus status, TableDescriptor td) throws IOException {
        Path tableInfoDir = status.getPath().getParent();
        Path tableDir = tableInfoDir.getParent();
        FSTableDescriptors.writeTableDescriptor(fs, td, tableDir, status);
    }

    @VisibleForTesting
    Path updateTableDescriptor(TableDescriptor td) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot update a table descriptor - in read only mode");
        }
        TableName tableName = td.getHTableDescriptor().getTableName();
        Path tableDir = this.getTableDir(tableName);
        Path p = FSTableDescriptors.writeTableDescriptor(this.fs, td, tableDir, this.getTableInfoPath(tableDir));
        if (p == null) {
            throw new IOException("Failed update");
        }
        LOG.info((Object)("Updated tableinfo=" + p));
        if (this.usecache) {
            this.cache.put(td.getHTableDescriptor().getTableName(), td);
        }
        return p;
    }

    public void deleteTableDescriptorIfExists(TableName tableName) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot delete a table descriptor - in read only mode");
        }
        Path tableDir = this.getTableDir(tableName);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        FSTableDescriptors.deleteTableDescriptorFiles(this.fs, tableInfoDir, Integer.MAX_VALUE);
    }

    private static void deleteTableDescriptorFiles(FileSystem fs, Path dir, int maxSequenceId) throws IOException {
        FileStatus[] status;
        for (FileStatus file : status = FSUtils.listStatus(fs, dir, TABLEINFO_PATHFILTER)) {
            Path path = file.getPath();
            int sequenceId = FSTableDescriptors.getTableInfoSequenceId(path);
            if (sequenceId > maxSequenceId) continue;
            boolean success = FSUtils.delete(fs, path, false);
            if (success) {
                LOG.debug((Object)("Deleted table descriptor at " + path));
                continue;
            }
            LOG.error((Object)("Failed to delete descriptor at " + path));
        }
    }

    private static Path writeTableDescriptor(FileSystem fs, TableDescriptor htd, Path tableDir, FileStatus currentDescriptorFile) throws IOException {
        int currentSequenceId;
        Path tmpTableDir = new Path(tableDir, TMP_DIR);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        int newSequenceId = currentSequenceId = currentDescriptorFile == null ? 0 : FSTableDescriptors.getTableInfoSequenceId(currentDescriptorFile.getPath());
        int retries = 10;
        int retrymax = currentSequenceId + retries;
        Path tableInfoDirPath = null;
        do {
            String filename;
            Path tempPath;
            if (fs.exists(tempPath = new Path(tmpTableDir, filename = FSTableDescriptors.getTableInfoFileName(++newSequenceId)))) {
                LOG.debug((Object)(tempPath + " exists; retrying up to " + retries + " times"));
                continue;
            }
            tableInfoDirPath = new Path(tableInfoDir, filename);
            try {
                FSTableDescriptors.writeTD(fs, tempPath, htd);
                fs.mkdirs(tableInfoDirPath.getParent());
                if (!fs.rename(tempPath, tableInfoDirPath)) {
                    throw new IOException("Failed rename of " + tempPath + " to " + tableInfoDirPath);
                }
                LOG.debug((Object)("Wrote descriptor into: " + tableInfoDirPath));
                break;
            }
            catch (IOException ioe) {
                LOG.debug((Object)"Failed write and/or rename; retrying", (Throwable)ioe);
                if (!FSUtils.deleteDirectory(fs, tempPath)) {
                    LOG.warn((Object)("Failed cleanup of " + tempPath));
                }
                tableInfoDirPath = null;
            }
        } while (newSequenceId < retrymax);
        if (tableInfoDirPath != null) {
            FSTableDescriptors.deleteTableDescriptorFiles(fs, tableInfoDir, newSequenceId - 1);
        }
        return tableInfoDirPath;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeTD(FileSystem fs, Path p, TableDescriptor htd) throws IOException {
        try (FSDataOutputStream out = fs.create(p, false);){
            out.write(htd.toByteArray());
        }
    }

    public boolean createTableDescriptor(TableDescriptor htd) throws IOException {
        return this.createTableDescriptor(htd, false);
    }

    public boolean createTableDescriptor(HTableDescriptor htd) throws IOException {
        return this.createTableDescriptor(new TableDescriptor(htd), false);
    }

    public boolean createTableDescriptor(TableDescriptor htd, boolean forceCreation) throws IOException {
        Path tableDir = this.getTableDir(htd.getHTableDescriptor().getTableName());
        return this.createTableDescriptorForTableDirectory(tableDir, htd, forceCreation);
    }

    public boolean createTableDescriptor(HTableDescriptor htd, boolean forceCreation) throws IOException {
        return this.createTableDescriptor(new TableDescriptor(htd), forceCreation);
    }

    public boolean createTableDescriptorForTableDirectory(Path tableDir, TableDescriptor htd, boolean forceCreation) throws IOException {
        Path p;
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot create a table descriptor - in read only mode");
        }
        FileStatus status = FSTableDescriptors.getTableInfoPath(this.fs, tableDir);
        if (status != null) {
            LOG.debug((Object)("Current tableInfoPath = " + status.getPath()));
            if (!forceCreation && this.fs.exists(status.getPath()) && status.getLen() > 0L && FSTableDescriptors.readTableDescriptor(this.fs, status, false).equals(htd)) {
                LOG.debug((Object)"TableInfo already exists.. Skipping creation");
                return false;
            }
        }
        return (p = FSTableDescriptors.writeTableDescriptor(this.fs, htd, tableDir, status)) != null;
    }
}

