/*
 * Copyright (C) 2010.
 * All rights reserved.
 */
package ro.isdc.wro.extensions.processor.js;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ro.isdc.wro.WroRuntimeException;
import ro.isdc.wro.extensions.processor.support.ObjectPoolHelper;
import ro.isdc.wro.extensions.processor.support.packer.PackerJs;
import ro.isdc.wro.model.group.processor.Minimize;
import ro.isdc.wro.model.resource.Resource;
import ro.isdc.wro.model.resource.ResourceType;
import ro.isdc.wro.model.resource.SupportedResourceType;
import ro.isdc.wro.model.resource.processor.Destroyable;
import ro.isdc.wro.model.resource.processor.ResourcePostProcessor;
import ro.isdc.wro.model.resource.processor.ResourcePreProcessor;
import ro.isdc.wro.util.ObjectFactory;


/**
 *  Uses <a href="http://dean.edwards.name/packer/">Dean Edwards packer utility</a> to pack js resources.
 *
 * @author Alex Objelean
 * @created 31 Jul 2010
 */
@Minimize
@SupportedResourceType(ResourceType.JS)
public class PackerJsProcessor
  implements ResourcePreProcessor, ResourcePostProcessor, Destroyable {

  private static final Logger LOG = LoggerFactory.getLogger(PackerJsProcessor.class);
  public static final String ALIAS = "packerJs";
  private ObjectPoolHelper<PackerJs> enginePool;

  public PackerJsProcessor() {
    enginePool = new ObjectPoolHelper<PackerJs>(new ObjectFactory<PackerJs>() {
      @Override
      public PackerJs create() {
        return newPackerJs();
      }
    });
  }

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void process(final Resource resource, final Reader reader, final Writer writer) throws IOException {

		final String content = IOUtils.toString(reader);
		final PackerJs packerJs = enginePool.getObject();

		try (reader; writer) {
			writer.write(packerJs.pack(content));
		} catch (final WroRuntimeException e) {
			onException(e);
			final String resourceUri = resource == null ? StringUtils.EMPTY : "[" + resource.getUri() + "]";
			LOG.warn("Exception while applying " + getClass().getSimpleName() + " processor on the " + resourceUri
					+ " resource, no processing applied...", e);
		} finally {
			enginePool.returnObject(packerJs);
		}
	}

  /**
   * Invoked when a processing exception occurs.
   */
  protected void onException(final WroRuntimeException e) {
    throw e;
  }

  /**
   * @return PackerJs engine.
   */
  protected PackerJs newPackerJs() {
    return new PackerJs();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void process(final Reader reader, final Writer writer)
    throws IOException {
    process(null, reader, writer);
  }

  @Override
  public void destroy() throws Exception {
    enginePool.destroy();
  }
}
