package com.atlassian.cache;

import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;

import com.atlassian.annotations.PublicApi;

/**
 * A builder for creating {@link CacheSettings} instances.
 * @since 2.0
 */
@PublicApi
public class CacheSettingsBuilder {
    private Long expireAfterAccess;
    private Long expireAfterWrite;
    private Boolean flushable;
    private Boolean local;
    private Integer maxEntries;
    private Boolean replicateAsynchronously;
    private Boolean replicateViaCopy;
    private Boolean statisticsEnabled;

    /**
     * Constructor
     */
    public CacheSettingsBuilder() {}

    /**
     * Constructor that initializes the builder with the settings from a {@link CacheSettings} instance
     *
     * @param settings the instance to initialize the builder with
     */
    public CacheSettingsBuilder(CacheSettings settings) {
        this.expireAfterAccess = settings.getExpireAfterAccess();
        this.expireAfterWrite = settings.getExpireAfterWrite();
        this.flushable = settings.getFlushable();
        this.local = settings.getLocal();
        this.maxEntries = settings.getMaxEntries();
        this.replicateAsynchronously = settings.getReplicateAsynchronously();
        this.replicateViaCopy = settings.getReplicateViaCopy();
        this.statisticsEnabled = settings.getStatisticsEnabled();
    }

    /**
     * @return a new {@link CacheSettings} instance with
     */
    @Nonnull
    public CacheSettings build() {
        return new DefaultCacheSettings(
                expireAfterAccess,
                expireAfterWrite,
                flushable,
                local,
                maxEntries,
                replicateAsynchronously,
                replicateViaCopy,
                statisticsEnabled);
    }

    /**
     * Set a hint for the cache regarding how long entries should be held in the cache.
     * @param expireAfter Time to retain entries for since their last access.
     * @param timeUnit The {@link TimeUnit} for the time
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder expireAfterAccess(long expireAfter, @Nonnull TimeUnit timeUnit) {
        this.expireAfterAccess = timeUnit.toMillis(expireAfter);
        return this;
    }

    /**
     * Set a hint for the cache regarding how long entries should be held in the cache.
     * @param expireAfter Time to retain entries from when they were created.
     * @param timeUnit The {@link TimeUnit} for the time
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder expireAfterWrite(long expireAfter, @Nonnull TimeUnit timeUnit) {
        this.expireAfterWrite = timeUnit.toMillis(expireAfter);
        return this;
    }

    /**
     * Indicates that this cache can be flushed by the cache manager when desired.
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder flushable() {
        this.flushable = true;
        return this;
    }

    /**
     * Indicates that this cache cannot be flushed by the cache manager when desired.
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder unflushable() {
        this.flushable = false;
        return this;
    }

    /**
     * Indicates that this cache can have a maximum number of entries.
     * @param maxEntries The maximum number of entries. Must be greater than zero.
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder maxEntries(int maxEntries) {
        if (0 >= maxEntries) {
            throw new IllegalArgumentException("maxEntries must be greater than zero, passed: " + maxEntries);
        }
        this.maxEntries = maxEntries;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate asynchronously. By default, it is up to the host
     * application as to whether replication is synchronous or asynchronous.
     *
     * <br>
     * When it is synchronous, cache mutations block on the originating node until
     * all nodes in the cluster have replicated the mutation. Asynchronous
     * replication provides greater responsiveness at the expense of
     * consistency.
     *
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder replicateAsynchronously() {
        this.replicateAsynchronously = true;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate synchronously.
     *
     * @return this builder
     * @see #replicateAsynchronously
     * @since 2.3.0
     */
    @Nonnull
    public CacheSettingsBuilder replicateSynchronously() {
        this.replicateAsynchronously = false;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate put and update operations by copying the relevant
     * key and value across the wire (requiring both of them to be {@link
     * java.io.Serializable}).
     * <br>
     * By default, it is up to the host application as to whether replication is via
     * copy or invalidation.
     *
     * @return this builder
     * @see #replicateViaInvalidation()
     */
    @Nonnull
    public CacheSettingsBuilder replicateViaCopy() {
        this.replicateViaCopy = true;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate by sending only the key across the wire, for invalidation by
     * the other nodes in the cluster; this requires only the key to be {@link
     * java.io.Serializable}.
     * <br>
     * By default, it is up to the host application as to whether replication is via
     * copy or invalidation.
     *
     * @return this builder
     * @see #replicateViaCopy
     * @since 2.3.0
     */
    @Nonnull
    public CacheSettingsBuilder replicateViaInvalidation() {
        this.replicateViaCopy = false;
        return this;
    }

    /**
     * Indicates that this cache should be local to the node (JVM) where the cache is created.
     * By default, caches will be clustered in a clustered deployment.
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder local() {
        this.local = true;
        return this;
    }

    /**
     * Indicates that this cache should be clustered in a clustered deployment.
     * @return this builder
     */
    @Nonnull
    public CacheSettingsBuilder remote() {
        this.local = false;
        return this;
    }

    /**
     * Indicates that this cache should record statistics.
     * @return this builder
     */
    public CacheSettingsBuilder statisticsEnabled() {
        this.statisticsEnabled = true;
        return this;
    }

    /**
     * Indicates that this cache should not record statistics.
     * @return this builder
     */
    public CacheSettingsBuilder statisticsDisabled() {
        this.statisticsEnabled = false;
        return this;
    }
}
